<?php

namespace App\Exports;

use App\Models\MasterDocument;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Route;
use Carbon\Carbon;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\WithCustomValueBinder;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Cell\Cell;
use PhpOffice\PhpSpreadsheet\Cell\DataType;
use PhpOffice\PhpSpreadsheet\Cell\DefaultValueBinder;
use PhpOffice\PhpSpreadsheet\Shared\Date as ExcelDate;
use PhpOffice\PhpSpreadsheet\Style\Conditional;
use PhpOffice\PhpSpreadsheet\Style\Color;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

class MasterDocumentsExport extends DefaultValueBinder implements
    FromQuery, WithMapping, WithHeadings, WithEvents, WithTitle, WithColumnFormatting, WithCustomValueBinder
{
    // 🎨 Theme (Deep Navy Blue)
    private string $indigo = 'FF2C5282'; // header bg (Deep Blue)
    private string $indigoDark = 'FF1A202C'; // header border (Navy Dark)
    private string $zebra = 'FFE0E7FA';  // subtle row stripe

    public function __construct(private int $projectId, private array $filters = []) {}

    public function title(): string { return 'MDR'; }

    public function query()
    {
        return MasterDocument::query()
            ->where('project_id', $this->projectId)
            ->with([
                'category:id,name',
                'discipline:id,name',
                // لود کردن آخرین رویژن برای POI و progress_percent
                'revisions' => fn($q) => $q->latest('date')->limit(1),
                // لود کردن آخرین ترنسمیتال مرتبط
                'transmittalDocuments' => fn($q) => $q->with('transmittal:id,transmittal_number,status,sent_at,subject')->latest()->limit(1),
            ])
            ->when(!empty($this->filters['search']), function (Builder $q) {
                $s = trim($this->filters['search']);
                $q->where(function ($qq) use ($s) {
                    $qq->where('document_number','like',"%{$s}%")
                        ->orWhere('title','like',"%{$s}%");
                });
            })
            ->when(!empty($this->filters['status']), fn($q)=>$q->where('status',$this->filters['status']))
            ->when(!empty($this->filters['discipline_id']), fn($q)=>$q->where('discipline_id',$this->filters['discipline_id']))
            ->when(!empty($this->filters['transmittal_status']), function ($q) {
                $ts = $this->filters['transmittal_status'];
                if ($ts === 'بدون ارسال') $q->whereDoesntHave('transmittalDocuments');
                else $q->whereHas('transmittalDocuments.transmittal', fn($qq)=>$qq->where('status',$ts));
            })
            ->orderBy('discipline_id')->orderBy('document_number');
    }

    public function headings(): array
    {
        return [
            'ردیف','شماره سند','عنوان','دیسپلین','کلاس','نوع','نسخه','وضعیت',
            'وزن (W)', 'درصد پیشرفت','POI نهایی',
            'تاریخ برنامه‌ریزی‌شده (Plan)', 'تاریخ انتشار واقعی (Actual)',
            'شماره Trans','وضعیت Trans','تاریخ ارسال Trans','در انتظار (روز)','لینک فایل','لینک ترنسمیتال'
        ];
    }

    private function toCarbon($value): ?Carbon
    {
        if (empty($value)) return null;
        if ($value instanceof \DateTimeInterface) return Carbon::instance($value);
        try { return Carbon::parse($value); } catch (\Throwable $e) { return null; }
    }

    public function map($doc): array
    {
        static $row = 0; $row++;

        $lastTd = optional($doc->transmittalDocuments)->sortByDesc(fn($td) => optional($td->transmittal)->sent_at)->first();
        $trans  = $lastTd?->transmittal;
        $latestRev = optional($doc->revisions)->first();

        // تلاش برای پیدا کردن آخرین فایل
        $latestFile = $latestRev?->files()->latest()->first();
        $fileUrl = $latestFile ? Storage::disk('public')->url($latestFile->file_path) : '';

        $transUrl = '';
        if ($trans) {
            $name = Route::has('transmittals.show') ? 'transmittals.show' : (Route::has('transmittal.show') ? 'transmittal.show' : null);
            $transUrl = $name ? route($name, $trans->id, false) : url('/transmittals/'.$trans->id);
        }

        $sentAtExcel = '';
        $waitingDays = '';
        if ($trans && $trans->sent_at) {
            if ($c = $this->toCarbon($trans->sent_at)) {
                $sentAtExcel = ExcelDate::dateTimeToExcel($c);
                $waitingDays = Carbon::now()->diffInDays($c);
            }
        }

        $plannedDate = $doc->planned_issue_date ? ExcelDate::dateTimeToExcel($this->toCarbon($doc->planned_issue_date)) : '';
        $actualDate  = $doc->actual_issue_date ? ExcelDate::dateTimeToExcel($this->toCarbon($doc->actual_issue_date)) : '';

        return [
            $row,
            (string)$doc->document_number,
            (string)$doc->title,
            (string)($doc->discipline->name ?? '-'),
            (string)($doc->category->name ?? '-'),
            (string)($doc->document_type ?? '-'),
            (string)($latestRev->revision ?? '-'),
            (string)($doc->status ?? 'نامشخص'),

            (float)$doc->weight,
            $doc->progress_percent . '%',
            (string)($latestRev->poi_type ?? '-'),

            $plannedDate,
            $actualDate,

            (string)($trans->transmittal_number ?? '-'),
            (string)($trans->status ?? '-'),
            $sentAtExcel,
            $waitingDays,
            $fileUrl,
            $transUrl,
        ];
    }

    public function bindValue(Cell $cell, $value)
    {
        if ($cell->getColumn() === 'B') {
            $cell->setValueExplicit((string)$value, DataType::TYPE_STRING);
            return true;
        }
        return parent::bindValue($cell, $value);
    }

    public function columnFormats(): array
    {
        return [
            'L' => NumberFormat::FORMAT_DATE_YYYYMMDD2,
            'M' => NumberFormat::FORMAT_DATE_YYYYMMDD2,
            'P' => NumberFormat::FORMAT_DATE_YYYYMMDD2,
            'I' => NumberFormat::FORMAT_NUMBER_00,
            'J' => NumberFormat::FORMAT_TEXT,
            'Q' => NumberFormat::FORMAT_NUMBER,
        ];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $e){
                $s = $e->sheet->getDelegate();
                $s->setRightToLeft(true);
                $s->freezePane('A2');
                $s->getParent()->getDefaultStyle()->getFont()->setName('Vazirmatn')->setSize(11);

                $headerRange = 'A1:S1';
                $s->getStyle($headerRange)->applyFromArray([
                    'font' => ['bold' => true, 'color' => ['argb' => Color::COLOR_WHITE]],
                    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['argb' => $this->indigo]],
                    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER, 'wrapText' => true],
                    'borders' => ['bottom' => ['borderStyle' => Border::BORDER_MEDIUM, 'color' => ['argb' => $this->indigoDark]]],
                ]);
                $s->getRowDimension(1)->setRowHeight(30);
                $s->setAutoFilter($headerRange);

                // تنظیم عرض ستون‌ها
                foreach (['A'=>8,'B'=>20,'C'=>50,'D'=>18,'E'=>18,'F'=>12,'G'=>10,'H'=>12,'I'=>10,'J'=>10,'K'=>12,'L'=>16,'M'=>16,'N'=>18,'O'=>18,'P'=>14,'Q'=>14,'R'=>28,'S'=>28] as $c=>$w) {
                    $s->getColumnDimension($c)->setWidth($w);
                }
                // Wrap Text برای ستون عنوان
                $s->getStyle('C2:C'.$s->getHighestRow())->getAlignment()->setWrapText(true)->setHorizontal(Alignment::HORIZONTAL_RIGHT);

                // Zebra Striping
                $max = $s->getHighestRow();
                for ($r=2; $r <= $max; $r+=2) {
                    $s->getStyle("A{$r}:S{$r}")->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setARGB($this->zebra);
                }

                // لینک‌سازی (R=لینک فایل، S=لینک ترنسمیتال)
                for ($r=2; $r<=$max; $r++) {
                    if ($url = $s->getCell("R{$r}")->getValue()) {
                        $s->getCell("R{$r}")->getHyperlink()->setUrl($url);
                        $s->setCellValue("R{$r}", 'دانلود فایل');
                        $s->getStyle("R{$r}")->getFont()->getColor()->setARGB(Color::COLOR_BLUE)->setUnderline(true);
                    }
                    if ($url = $s->getCell("S{$r}")->getValue()) {
                        $s->getCell("S{$r}")->getHyperlink()->setUrl($url);
                        $s->setCellValue("S{$r}", 'نمایش Trans');
                        $s->getStyle("S{$r}")->getFont()->getColor()->setARGB(Color::COLOR_BLUE)->setUnderline(true);
                    }
                }

                // استایل وضعیت‌ها و دیرکرد
                $this->statusCond($s, 'H', 2, $max);
                $this->statusCond($s, 'O', 2, $max);
                $this->greaterThanFill($s, 'Q', 2, $max, 10, 'FFF59E0B', 'FF111827');
            }
        ];
    }

    private function statusCond($sheet, string $col, int $start, int $end): void
    {
        $range = "{$col}{$start}:{$col}{$end}";

        $mk = function($text,$bg){
            $c = new Conditional();
            $c->setConditionType(Conditional::CONDITION_CONTAINSTEXT)->setText($text);
            $st = $c->getStyle();
            $st->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setARGB($bg);
            $st->getFont()->getColor()->setARGB(Color::COLOR_WHITE);
            return $c;
        };

        $conds = [
            $mk('تایید شده','FF16A34A'), // سبز
            $mk('رد شده','FFDC2626'),    // قرمز
            $mk('ارسال شده','FFF59E0B'), // زرد
        ];
        $sheet->getStyle($range)->setConditionalStyles($conds);
    }

    private function greaterThanFill($sheet, string $col, int $start, int $end, int $threshold, string $bg, string $fg): void
    {
        $range = "{$col}{$start}:{$col}{$end}";
        $cond = new Conditional();
        $cond->setConditionType(Conditional::CONDITION_CELLIS)
            ->setOperatorType(Conditional::OPERATOR_GREATERTHAN)
            ->addCondition((string)$threshold);
        $style = $cond->getStyle();
        $style->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setARGB($bg);
        $style->getFont()->getColor()->setARGB($fg);
        $sheet->getStyle($range)->setConditionalStyles([$cond]);
    }

    // این متد برای جلوگیری از از دست دادن صفرهای اولیه (برای شماره سند) در اکسل مورد نیاز است
    public function setCell(Cell $cell, $value) {
        if ($cell->getColumn() === 'B') {
            $cell->setValueExplicit((string)$value, DataType::TYPE_STRING);
            return true;
        }
        return parent::bindValue($cell, $value);
    }
}
