<?php

namespace App\Exports\Mdr\Sheets;

use App\Models\MasterDocument;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Route;
use Carbon\Carbon;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\WithCustomValueBinder;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Cell\Cell;
use PhpOffice\PhpSpreadsheet\Cell\DataType;
use PhpOffice\PhpSpreadsheet\Cell\DefaultValueBinder;
use PhpOffice\PhpSpreadsheet\Shared\Date as ExcelDate;
use PhpOffice\PhpSpreadsheet\Style\Conditional;
use PhpOffice\PhpSpreadsheet\Style\Color;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

class MasterDocumentsSheet extends DefaultValueBinder implements
    FromQuery, WithMapping, WithHeadings, WithEvents, WithTitle, WithColumnFormatting, WithCustomValueBinder
{
    private string $indigo = 'FF4F46E5';
    private string $indigoDark = 'FF3730A3';
    private string $zebra = 'FFF8FAFF';

    public function __construct(private int $projectId, private array $filters = []) {}

    public function title(): string { return 'MDR'; }

    public function query()
    {
        return MasterDocument::query()
            ->where('project_id', $this->projectId)
            ->with([
                'category:id,name',
                'discipline:id,name',
                'documentFiles' => fn($q) => $q->latest('created_at')->limit(1),
                'transmittalDocuments.transmittal:id,transmittal_number,status,sent_at',
            ])
            ->when(!empty($this->filters['search']), function (Builder $q) {
                $s = trim($this->filters['search']);
                $q->where(function ($qq) use ($s) {
                    $qq->where('document_number','like',"%{$s}%")
                        ->orWhere('title','like',"%{$s}%");
                });
            })
            ->when(!empty($this->filters['status']), fn($q)=>$q->where('status',$this->filters['status']))
            ->when(!empty($this->filters['discipline_id']), fn($q)=>$q->where('discipline_id',$this->filters['discipline_id']))
            ->when(!empty($this->filters['transmittal_status']), function ($q) {
                $ts = $this->filters['transmittal_status'];
                if ($ts === 'بدون ارسال') $q->whereDoesntHave('transmittalDocuments');
                else $q->whereHas('transmittalDocuments.transmittal', fn($qq)=>$qq->where('status',$ts));
            })
            ->orderBy('id','desc');
    }

    public function headings(): array
    {
        return [
            'ردیف','شماره سند','عنوان','دسته‌بندی','دیسپلین','نوع','نسخه','وضعیت',
            'آخرین ترنسمیتال','وضعیت ترنسمیتال','تاریخ ارسال','در انتظار (روز)','لینک فایل','لینک ترنسمیتال'
        ];
    }

    public function map($doc): array
    {
        static $row = 0; $row++;

        $lastTd = optional($doc->transmittalDocuments)->sortByDesc(fn($td) => optional($td->transmittal)->sent_at)->first();
        $trans  = $lastTd?->transmittal;
        $latestFile = $doc->documentFiles->first();

        $fileUrl = $latestFile ? Storage::disk('public')->url($latestFile->file_path) : '';

        $transUrl = '';
        if ($trans) {
            $transUrl = url('/transmittals/' . $trans->id);
        }

        $sentAtExcel = '';
        $waitingDays = '';
        if ($trans && $trans->sent_at) {
            $c = Carbon::parse($trans->sent_at);
            $sentAtExcel = ExcelDate::dateTimeToExcel($c);
            $waitingDays = Carbon::now()->diffInDays($c);
        }

        return [
            $row,
            $doc->document_number, // در bindValue هندل می‌شود
            (string)$doc->title,
            (string)($doc->category->name ?? '-'),
            (string)($doc->discipline->name ?? '-'),
            (string)($doc->document_type ?? '-'),
            (string)($doc->revision ?? '-'),
            (string)($doc->status ?? 'نامشخص'),
            (string)($trans->transmittal_number ?? '-'),
            (string)($trans->status ?? '-'),
            $sentAtExcel,
            $waitingDays,
            $fileUrl,
            $transUrl,
        ];
    }

    public function bindValue(Cell $cell, $value)
    {
        // حفظ صفرهای ابتدایی در شماره سند (ستون B)
        if ($cell->getColumn() === 'B') {
            $cell->setValueExplicit($value, DataType::TYPE_STRING);
            return true;
        }
        return parent::bindValue($cell, $value);
    }

    public function columnFormats(): array
    {
        return [
            'K' => NumberFormat::FORMAT_DATE_YYYYMMDD2,
            'L' => NumberFormat::FORMAT_NUMBER,
        ];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $e){
                $s = $e->sheet->getDelegate();
                $s->setRightToLeft(true);
                $s->freezePane('A2');
                $s->getParent()->getDefaultStyle()->getFont()->setName('Vazirmatn')->setSize(11);

                // استایل هدر (A تا N)
                $s->getStyle('A1:N1')->applyFromArray([
                    'font' => ['bold' => true, 'color' => ['argb' => Color::COLOR_WHITE]],
                    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['argb' => $this->indigo]],
                    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
                ]);

                // تنظیم عرض ستون‌ها (فقط تا N)
                $widths = ['A'=>8,'B'=>20,'C'=>50,'D'=>18,'E'=>18,'F'=>12,'G'=>10,'H'=>15,'I'=>20,'J'=>18,'K'=>15,'L'=>12,'M'=>25,'N'=>25];
                foreach ($widths as $col => $width) {
                    $s->getColumnDimension($col)->setWidth($width);
                }

                $maxRow = $s->getHighestRow();

                // لینک‌سازی و Zebra Stripping
                for ($r = 2; $r <= $maxRow; $r++) {
                    // Zebra
                    if ($r % 2 == 0) {
                        $s->getStyle("A{$r}:N{$r}")->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setARGB($this->zebra);
                    }

                    // لینک فایل (M)
                    if ($url = $s->getCell("M{$r}")->getValue()) {
                        $s->getCell("M{$r}")->getHyperlink()->setUrl($url);
                        $s->setCellValue("M{$r}", 'دانلود فایل');
                        $s->getStyle("M{$r}")->getFont()->getColor()->setARGB(Color::COLOR_BLUE)->setUnderline(true);
                    }
                    // لینک ترنسمیتال (N)
                    if ($url = $s->getCell("N{$r}")->getValue()) {
                        $s->getCell("N{$r}")->getHyperlink()->setUrl($url);
                        $s->setCellValue("N{$r}", 'نمایش ترنسمیتال');
                        $s->getStyle("N{$r}")->getFont()->getColor()->setARGB(Color::COLOR_BLUE)->setUnderline(true);
                    }
                }

                $this->statusCond($s, 'H', 2, $maxRow); // وضعیت سند
                $this->statusCond($s, 'J', 2, $maxRow); // وضعیت ترنسمیتال
                $this->greaterThanFill($s, 'L', 2, $maxRow, 10, 'FFF59E0B', 'FF111827');
            }
        ];
    }

    private function statusCond($sheet, string $col, int $start, int $end): void
    {
        $range = "{$col}{$start}:{$col}{$end}";
        $conds = [];

        $states = [
            'تایید شده' => 'FF16A34A',
            'رد شده' => 'FFDC2626',
            'ارسال شده' => 'FFF59E0B'
        ];

        foreach ($states as $text => $color) {
            $c = new Conditional();
            $c->setConditionType(Conditional::CONDITION_CONTAINSTEXT)->setText($text);
            $c->getStyle()->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setARGB($color);
            $c->getStyle()->getFont()->getColor()->setARGB(Color::COLOR_WHITE);
            $conds[] = $c;
        }
        $sheet->getStyle($range)->setConditionalStyles($conds);
    }

    private function greaterThanFill($sheet, string $col, int $start, int $end, int $threshold, string $bg, string $fg): void
    {
        $range = "{$col}{$start}:{$col}{$end}";
        $cond = new Conditional();
        $cond->setConditionType(Conditional::CONDITION_CELLIS)
            ->setOperatorType(Conditional::OPERATOR_GREATERTHAN)
            ->addCondition((string)$threshold);
        $cond->getStyle()->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setARGB($bg);
        $cond->getStyle()->getFont()->getColor()->setARGB($fg);
        $sheet->getStyle($range)->setConditionalStyles([$cond]);
    }
}
