<?php

namespace App\Exports\Mdr\Sheets;

use App\Models\DocumentRevision;
use Illuminate\Database\Eloquent\Builder;
use Carbon\Carbon;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Shared\Date as ExcelDate;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Color;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

class RevisionsSheet implements FromQuery, WithMapping, WithHeadings, WithTitle, WithEvents, WithColumnFormatting
{
    // 🎨 Theme (Deep Navy Blue)
    private string $indigo = 'FF2C5282';
    private string $indigoDark = 'FF1A202C';
    private string $zebra = 'FFE0E7FA';

    public function __construct(private int $projectId, private array $filters = []) {}

    public function title(): string { return 'Revisions'; }

    public function query()
    {
        return DocumentRevision::query()
            ->with(['masterDocument:id,project_id,document_number,title'])
            ->whereHas('masterDocument', fn($q)=>$q->where('project_id',$this->projectId))
            ->when(!empty($this->filters['search']), function (Builder $q) {
                $s = trim($this->filters['search']);
                $q->whereHas('masterDocument', function($qq) use ($s){
                    $qq->where('document_number','like',"%{$s}%")
                        ->orWhere('title','like',"%{$s}%");
                });
            })
            ->orderBy('date','desc');
    }

    public function headings(): array
    {
        return ['شماره سند','عنوان سند','رویژن','نوع','وضعیت','تاریخ','توضیح'];
    }

    private function toCarbon($value): ?Carbon
    {
        if (empty($value)) return null;
        if ($value instanceof \DateTimeInterface) return Carbon::instance($value);
        try { return Carbon::parse($value); } catch (\Throwable $e) { return null; }
    }

    public function map($rev): array
    {
        $excelDate = '';
        if ($rev->date && ($c = $this->toCarbon($rev->date))) {
            $excelDate = ExcelDate::dateTimeToExcel($c);
        }

        return [
            (string)$rev->masterDocument?->document_number,
            (string)$rev->masterDocument?->title,
            (string)$rev->revision,
            (string)$rev->revision_type,
            (string)$rev->status,
            $excelDate,
            (string)($rev->remark ?? ''),
        ];
    }

    public function columnFormats(): array
    {
        return ['F' => NumberFormat::FORMAT_DATE_YYYYMMDD2];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $e){
                $s = $e->sheet->getDelegate();
                $s->setRightToLeft(true);
                $s->freezePane('A2');
                $s->getParent()->getDefaultStyle()->getFont()->setName('Vazirmatn')->setSize(11);

                // header
                $s->getStyle('A1:G1')->applyFromArray([
                    'font' => ['bold' => true, 'color' => ['argb' => Color::COLOR_WHITE]],
                    'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['argb' => $this->indigo]],
                    'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER, 'wrapText' => true],
                    'borders' => ['bottom' => ['borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_MEDIUM, 'color' => ['argb' => $this->indigoDark]]],
                ]);
                $s->getRowDimension(1)->setRowHeight(24);
                $s->setAutoFilter('A1:G1');

                foreach (['A'=>18,'B'=>45,'C'=>12,'D'=>12,'E'=>14,'F'=>14,'G'=>50] as $c=>$w) {
                    $s->getColumnDimension($c)->setWidth($w);
                }
                $max = $s->getHighestRow();
                for ($r=2; $r <= $max; $r+=2) {
                    $s->getStyle("A{$r}:G{$r}")->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setARGB($this->zebra);
                }
                $s->getStyle('B2:B'.$max)->getAlignment()->setWrapText(true)->setHorizontal(Alignment::HORIZONTAL_RIGHT);
            }
        ];
    }
}
