<?php

namespace App\Exports\Mdr\Sheets;

use App\Models\MasterDocument;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Color;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

class SummarySheet implements FromCollection, WithTitle, WithEvents
{
    // 🎨 Theme (Deep Navy Blue)
    private string $indigo = 'FF2C5282';
    private string $zebra = 'FFE0E7FA';

    public function __construct(private int $projectId, private array $filters = []) {}

    public function title(): string { return 'Summary'; }

    public function collection()
    {
        $apply = function ($q) {
            if (!empty($this->filters['search'])) {
                $s = trim($this->filters['search']);
                $q->where(function ($qq) use ($s) {
                    $qq->where('document_number','like',"%{$s}%")
                        ->orWhere('title','like',"%{$s}%");
                });
            }
            if (!empty($this->filters['status']))        $q->where('status',$this->filters['status']);
            if (!empty($this->filters['discipline_id'])) $q->where('discipline_id',$this->filters['discipline_id']);
            if (!empty($this->filters['transmittal_status'])) {
                $ts = $this->filters['transmittal_status'];
                if ($ts === 'بدون ارسال') $q->whereDoesntHave('transmittalDocuments');
                else $q->whereHas('transmittalDocuments.transmittal', fn($qq)=>$qq->where('status',$ts));
            }
        };

        $q = MasterDocument::where('project_id',$this->projectId);
        $apply($q);

        $total    = (clone $q)->count();
        $sent     = (clone $q)->where('status','ارسال شده')->count();
        $approved = (clone $q)->where('status','تایید شده')->count();
        $rejected = (clone $q)->where('status','رد شده')->count();

        $byDisc = (clone $q)->selectRaw('discipline_id, COUNT(*) cnt')->groupBy('discipline_id')->get()
            ->map(function($r){
                $name = optional(\App\Models\Discipline::find($r->discipline_id))->name ?? 'نامشخص';
                return ['دیسپلین'=>$name,'تعداد'=>$r->cnt];
            });

        $rows = new Collection();
        $rows->push(['گزارش خلاصه MDR']);
        $rows->push(['Project ID', $this->projectId]);
        $rows->push(['Search', $this->filters['search'] ?? '-']);
        $rows->push(['Status', $this->filters['status'] ?? 'همه']);
        $rows->push(['Discipline', $this->filters['discipline_id'] ?: 'همه']);
        $rows->push(['Transmittal Status', $this->filters['transmittal_status'] ?? 'همه']);
        $rows->push([]);
        $rows->push(['Totals', 'Count']);
        $rows->push(['کل', $total]);
        $rows->push(['ارسال شده', $sent]);
        $rows->push(['تایید شده', $approved]);
        $rows->push(['رد شده', $rejected]);
        $rows->push([]);
        $rows->push(['به تفکیک دیسپلین', 'تعداد']);
        foreach ($byDisc as $d) $rows->push([$d['دیسپلین'],$d['تعداد']]);
        return $rows;
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $e){
                $s = $e->sheet->getDelegate();
                $s->setRightToLeft(true);
                $s->freezePane('A2');
                $s->getParent()->getDefaultStyle()->getFont()->setName('Vazirmatn')->setSize(11);

                // Title badge
                $s->mergeCells('A1:B1');
                $s->getStyle('A1')->applyFromArray([
                    'font' => ['bold'=>true, 'size'=>13, 'color'=>['argb'=>Color::COLOR_WHITE]],
                    'fill' => ['fillType'=>Fill::FILL_SOLID, 'startColor'=>['argb'=>$this->indigo]],
                    'alignment' => ['horizontal'=>Alignment::HORIZONTAL_CENTER, 'vertical'=>Alignment::VERTICAL_CENTER],
                ]);
                $s->getRowDimension(1)->setRowHeight(28);

                // Section headers
                foreach ([8,14] as $row) {
                    $s->getStyle("A{$row}:B{$row}")->applyFromArray([
                        'font'=>['bold'=>true],
                        'fill'=>['fillType'=>Fill::FILL_SOLID,'startColor'=>['argb'=>$this->zebra]],
                    ]);
                }

                // widths
                $s->getColumnDimension('A')->setWidth(28);
                $s->getColumnDimension('B')->setWidth(60);
            }
        ];
    }
}
