<?php

namespace App\Livewire;

use App\Models\User;
use App\Notifications\AccountActivationRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use Livewire\Component;
use Throwable; // برای مدیریت جامع خطا
use Illuminate\Support\Facades\DB;
use Illuminate\Http\RedirectResponse; // برای اشاره به نوع بازگشتی Redirect

class AccountInactiveNotifier extends Component
{
    public bool $notificationSent = false;
    public ?string $errorMessage = null;

    protected int $requestLimitSeconds = 120; // 2 دقیقه
    public bool $isProcessing = false; // برای جلوگیری از Double Submit در فرانت‌اند

    public function render()
    {
        return view('livewire.account-inactive-notifier');
    }

    /**
     * ارسال درخواست فعال‌سازی حساب به مدیران
     * * @return void|RedirectResponse
     */
    public function notifyAdmin()
    {
        // 1. چک امنیتی و وضعیت پردازش
        $this->isProcessing = true;
        $this->errorMessage = null;

        try {
            // 🛑 چک حیاتی: اطمینان از لاگین بودن کاربر
            /** @var User $user */
            $user = Auth::user();
            if (! $user) {
                // اگر کاربر لاگین نیست، بلافاصله به صفحه ورود هدایت شود
                $this->redirect(route('login'));
                return;
            }

            // 2. ✅ اعتبارسنجی Rate Limit
            if ($user->last_activation_request_at) {
                $secondsSinceLastRequest = now()->diffInSeconds($user->last_activation_request_at);
                if ($secondsSinceLastRequest < $this->requestLimitSeconds) {
                    $waitSeconds = $this->requestLimitSeconds - $secondsSinceLastRequest;
                    $this->errorMessage = "لطفاً {$waitSeconds} ثانیه دیگر صبر کنید.";
                    return;
                }
            }

            // 3. ✅ بازیابی مدیران فعال
            $admins = User::query()
                ->where('role', User::ROLE_ADMIN)
                ->where('is_active', true)
                ->get();

            if ($admins->isEmpty()) {
                $msg = 'هیچ مدیر فعالی برای دریافت درخواست یافت نشد.';
                Log::warning('AccountActivation: No active admins found.', ['user_id' => $user->id]);
                $this->errorMessage = $msg;
                return;
            }

            // 4. شروع تراکنش دیتابیس
            DB::beginTransaction();

            // 5. ✅ ارسال نوتیفیکیشن
            Notification::send($admins, new AccountActivationRequest($user));

            // 6. ✅ به‌روزرسانی زمان درخواست
            $user->last_activation_request_at = now();
            $user->save();

            DB::commit();

            // 7. ✅ بازخورد موفقیت‌آمیز
            $this->notificationSent = true;
            $this->errorMessage = null;

        } catch (Throwable $e) {
            // Rollback در صورت بروز خطا
            if (DB::transactionLevel() > 0) {
                DB::rollBack();
            }

            // لاگ خطا
            Log::error('AccountActivationRequest failed unexpectedly.', [
                'user_id' => optional(Auth::user())->id,
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            $this->errorMessage = 'خطایی در ارسال درخواست رخ داد. لطفاً با پشتیبانی تماس بگیرید.';

        } finally {
            $this->isProcessing = false;
        }
    }
}
