<?php

namespace App\Livewire\Admin\Announcement;

use App\Models\Announcement;
use App\Models\AnnouncementAttachment;
use App\Models\Company;
use App\Models\Department;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Str;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

#[Layout('admin.master')]
class CreateAnnouncement extends Component
{
    use WithFileUploads, WithPagination;

    public $title = '';
    public $content = '';

    /** چندفایلی **/
    /** @var TemporaryUploadedFile[] */
    public array $files = [];
    public array $filePreviews = []; // [['name'=>..., 'url'=>..., 'is_image'=>bool, 'size_kb'=>...], ...]

    /** هدف ارسال **/
    public $target_type = 'all'; // all, company, department, custom

    /** توجه: اینا عدد می‌شن با wire:model.number */
    public $company_id = null;
    public $department_id = null;

    /** دپارتمان‌ها را خودمان مدیریت می‌کنیم (نه computed) */
    public array $departmentsOptions = [];
    protected int $depsVersion = 0; // برای key کردن بلاک دپارتمان

    /** انتخاب دستی */
    public $selected_users = [];
    public $search_users = '';
    public int $perPage = 20;

    protected $listeners = ['confirmedSend' => 'confirmedSend'];

    /** ========== mount ========== */
    public function mount(): void
    {
        // اگر company_id از قبل هست (مثلا از querystring/قدرت‌های دیگر)، load کن
        if ($this->company_id) {
            $this->reloadDepartments();
        }
    }

    /** ========== Providers ========== */
    public function getCompaniesProperty()
    {
        return Company::query()->orderBy('name')->select('id','name')->get();
    }

    /** به‌جای computed users، فقط برای custom صفحه‌بندی می‌کنیم */
    public function getUsersProperty()
    {
        if ($this->target_type !== 'custom') return collect();

        $q = User::query()
            ->where('status','active')->where('is_active', true)
            ->select('id','name','email','company_id','department_id')
            ->with(['company:id,name','department:id,name']);

        if ($this->company_id)    $q->where('company_id', (int)$this->company_id);
        if ($this->department_id) $q->where('department_id', (int)$this->department_id);

        if (filled($this->search_users)) {
            $s = trim($this->search_users);
            $q->where(fn($qq) => $qq->where('name','like',"%{$s}%")->orWhere('email','like',"%{$s}%"));
        }

        return $q->paginate($this->perPage);
    }

    /** ========== Lifecycle ========== */
    public function updatingSearchUsers(){ $this->resetPage(); }

    public function updatedTargetType(): void
    {
        $this->company_id     = null;
        $this->department_id  = null;
        $this->departmentsOptions = [];
        $this->depsVersion++;
        $this->selected_users = [];
        $this->search_users   = '';
        $this->resetPage();
        $this->resetValidation(); // فرم تمیز
    }

    public function updatedCompanyId($val): void
    {
        // همیشه عدد کن
        $this->company_id = $val ? (int) $val : null;

        // ریست وابسته‌ها
        $this->department_id = null;
        $this->selected_users = [];
        $this->search_users   = '';

        // خطای فیلد دپارتمان اگر قبلا بوده پاک شود
        $this->resetValidation(['department_id']);

        // بارگذاری مجدد دپارتمان‌ها
        $this->reloadDepartments();

        $this->resetPage();
    }

    public function updatedDepartmentId($val): void
    {
        $this->department_id = $val ? (int) $val : null;
        $this->selected_users = [];
        $this->search_users   = '';
        $this->resetPage();
    }

    /** بازسازی لیست دپارتمان‌ها (منبع حقیقت واحد) */
    protected function reloadDepartments(): void
    {
        $this->departmentsOptions = [];

        if (!$this->company_id) {
            $this->depsVersion++; // key را عوض کن تا UI تازه شود
            return;
        }

        $companyId = (int) $this->company_id;

        $deps = Department::query()
            ->where('company_id', $companyId)
            ->orderBy('name')
            ->get(['id','name']);

        if ($deps->isEmpty()) {
            Log::warning('Departments empty for company (reloadDepartments)', ['company_id' => $companyId]);
        }

        // تبدیل به آرایه ساده برای رندر سریع
        $this->departmentsOptions = $deps->map(fn($d)=>['id'=>$d->id,'name'=>$d->name])->all();

        // نسخه را افزایش بده که بلاک با key جدید re-mount شود
        $this->depsVersion++;
    }

    /** چندفایلی: ساخت پیش‌نمایش‌ها */
    public function updatedFiles(): void
    {
        $this->filePreviews = [];

        foreach ($this->files as $file) {
            if (!$file instanceof TemporaryUploadedFile) continue;

            $name = $file->getClientOriginalName();
            $url  = $file->temporaryUrl();
            $ext  = strtolower($file->getClientOriginalExtension());
            $isImage = in_array($ext, ['jpg','jpeg','png','gif','webp']);

            $this->filePreviews[] = [
                'name'     => $name,
                'url'      => $url,
                'is_image' => $isImage,
                'size_kb'  => round($file->getSize()/1024, 1),
            ];
        }
    }

    public function removeFile(int $index): void
    {
        if (!isset($this->files[$index])) return;
        unset($this->files[$index], $this->filePreviews[$index]);
        $this->files = array_values($this->files);
        $this->filePreviews = array_values($this->filePreviews);
    }

    /** ========== Validation ========== */
    public function rules()
    {
        $companyRule    = ['nullable'];
        $departmentRule = ['nullable'];

        if ($this->target_type === 'company') {
            $companyRule = ['required','integer','exists:companies,id'];
        }

        if ($this->target_type === 'department') {
            $companyRule = ['required','integer','exists:companies,id'];
            $departmentRule = [
                'required',
                'integer',
                Rule::exists('departments','id')->where(fn($q) =>
                $q->where('company_id', (int) $this->company_id)
                ),
            ];
        }

        return [
            'title'            => ['required','string','max:255'],
            'content'          => ['required','string'],

            'files'            => ['array','max:10'],
            'files.*'          => ['file','max:10240','mimes:pdf,doc,docx,xls,xlsx,jpg,jpeg,png,zip,webp'],

            'target_type'      => ['required','in:all,company,department,custom'],
            'company_id'       => $companyRule,
            'department_id'    => $departmentRule,
            'selected_users'   => $this->target_type === 'custom' ? ['array','min:1'] : ['nullable'],
            'selected_users.*' => $this->target_type === 'custom' ? ['integer','exists:users,id'] : ['nullable'],
        ];
    }

    protected $validationAttributes = [
        'title' => 'عنوان', 'content' => 'محتوا',
        'files' => 'فایل پیوست', 'files.*' => 'فایل پیوست',
        'target_type' => 'نوع هدف', 'company_id' => 'شرکت', 'department_id' => 'دپارتمان',
        'selected_users' => 'کاربران انتخابی',
    ];

    protected function validateSafely(): void
    {
        try {
            $this->validate();
        } catch (ValidationException $e) {
            Log::warning('Announcement validation failed', [
                'user_id'      => Auth::id(),
                'target_type'  => $this->target_type,
                'company_id'   => $this->company_id,
                'department_id'=> $this->department_id,
                'errors'       => $e->errors(),
            ]);
            throw $e;
        }
    }

    /** ========== Actions ========== */
    public function sendAnnouncement(): void
    {
        $this->validateSafely();

        if ($this->target_type === 'department' && $this->company_id && $this->department_id) {
            $ok = Department::where('id', (int)$this->department_id)
                ->where('company_id', (int)$this->company_id)
                ->exists();
            if (!$ok) {
                throw ValidationException::withMessages([
                    'department_id' => 'این دپارتمان متعلق به شرکت انتخاب‌شده نیست.',
                ]);
            }
        }

        $this->dispatch('confirm:send', [
            'title' => 'ارسال اطلاعیه',
            'text'  => 'آیا از ارسال این اطلاعیه اطمینان دارید؟',
        ]);
    }

    public function confirmedSend(): void
    {
        $this->validateSafely();

        try {
            DB::beginTransaction();

            $announcement = Announcement::create([
                'sender_id' => Auth::id(),
                'title'     => trim($this->title),
                'content'   => trim($this->content),
                'file'      => null, // برای سازگاری عقب‌رو می‌گذاریم اگر فایلی بود اولی را ست کنیم
            ]);

            $storedFirstPath = null;

            foreach ($this->files as $i => $file) {
                if (!$file instanceof TemporaryUploadedFile) continue;

                $orig = $file->getClientOriginalName();
                $safe = Str::limit(Str::slug(pathinfo($orig, PATHINFO_FILENAME)), 120, '');
                $ext  = strtolower($file->getClientOriginalExtension());
                $path = $file->storeAs('announcements', now()->format('Ymd_His')."_{$i}_{$safe}.{$ext}", 'public');

                AnnouncementAttachment::create([
                    'announcement_id' => $announcement->id,
                    'path'            => $path,
                    'original_name'   => $orig,
                    'mime'            => $file->getMimeType(),
                    'size_kb'         => round($file->getSize()/1024, 1),
                ]);

                if ($i === 0) $storedFirstPath = $path;
            }

            if ($storedFirstPath) {
                $announcement->update(['file' => $storedFirstPath]);
            }

            $recipients = User::query()
                ->where('status','active')->where('is_active', true)
                ->select('id');

            if ($this->target_type === 'company' && $this->company_id) {
                $recipients->where('company_id', (int)$this->company_id);
            } elseif ($this->target_type === 'department' && $this->department_id) {
                $recipients->where('company_id', (int)$this->company_id)
                    ->where('department_id', (int)$this->department_id);
            } elseif ($this->target_type === 'custom') {
                $recipients->whereIn('id', $this->selected_users);
            }

            $ids = $recipients->pluck('id');
            if ($ids->isNotEmpty()) {
                $announcement->recipients()->sync($ids);
            }

            DB::commit();

            // پاک‌سازی
            $this->reset([
                'title','content','files','filePreviews',
                'company_id','department_id','departmentsOptions','selected_users','search_users'
            ]);
            $this->depsVersion++;
            $this->target_type = 'all';
            $this->resetPage();

            session()->flash('success', '✅ اطلاعیه با موفقیت ارسال شد.');
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('Announcement send failed', [
                'user_id'     => Auth::id(),
                'message'     => $e->getMessage(),
                'trace'       => $e->getTraceAsString(),
                'target_type' => $this->target_type,
                'company_id'  => $this->company_id,
                'department_id'=> $this->department_id,
            ]);
            session()->flash('error', '❌ خطا در ارسال اطلاعیه.');
        }
    }

    /** انتخاب دستی برای custom */
    public function toggleUserSelection(int $userId): void
    {
        if ($this->target_type !== 'custom') return;
        if (in_array($userId, $this->selected_users, true)) {
            $this->selected_users = array_values(array_diff($this->selected_users, [$userId]));
        } else {
            $this->selected_users[] = $userId;
        }
    }

    public function selectAllOnPage(): void
    {
        if ($this->target_type !== 'custom') return;
        $ids = ($this->users instanceof \Illuminate\Pagination\LengthAwarePaginator)
            ? $this->users->pluck('id')->all()
            : [];
        $this->selected_users = array_values(array_unique(array_merge($this->selected_users, $ids)));
    }

    public function clearAllUsers(): void
    {
        $this->selected_users = [];
    }

    public function render()
    {
        return view('livewire.admin.announcement.create-announcement', [
            'companies'   => $this->companies,
            // دپارتمان‌ها از departmentsOptions می‌آید
            'users'       => $this->users,
            'depsVersion' => $this->depsVersion, // برای wire:key
        ]);
    }
}
