<?php

namespace App\Livewire\Admin\Approval;

use App\Models\ApprovalRule;
use App\Models\Approver;
use App\Models\Project;
use App\Models\User;
use App\Models\Role;
use Illuminate\Support\Facades\Gate;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\Attributes\Computed;

class ApprovalManagement extends Component
{
    use WithPagination;
    protected $paginationTheme = 'bootstrap';

    // --- فیلدهای فرم Rule ---
    public $name;
    public $projectId;
    public $amountMin;
    public $amountMax;
    public $priority = 1;
    public $isActive = 1;
    public $isDefault = 0;
    public $editRuleId;

    // --- فیلدهای فرم Approver ---
    public $ruleId;
    public $userId;
    public $roleId;
    public $approverPriority = 1;

    public $searchRule = '';
    public $statusFilter = '';
    public $isDefaultFilter = '';
    public $activeProject = null;
    public $viewMode = 'table';

    protected function dispatchSuccess($message)
    {
        $this->dispatch('show-toast', type: 'success', message: $message);
    }
    protected function dispatchError($message)
    {
        $this->dispatch('show-toast', type: 'error', message: $message);
    }

    protected function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'projectId' => 'sometimes|nullable|exists:projects,id',
            'amountMin' => 'sometimes|nullable|numeric|min:0',
            'amountMax' => 'sometimes|nullable|numeric|min:0',
            'priority' => 'required|integer|min:1',
            'isActive' => 'required|boolean',
            'isDefault' => 'sometimes|boolean',
        ];
    }
    protected $approverRules = [
        'ruleId' => 'required|exists:approval_rules,id',
        'userId' => 'required|exists:users,id',
        'roleId' => 'required|exists:roles,id',
        'approverPriority' => 'required|integer|min:1',
    ];

    #[Computed]
    public function projects() { return Project::all(); }
    #[Computed]
    public function users() { return User::all(); }
    #[Computed]
    public function roles() { return Role::all(); }

    public function updatedRuleId($value)
    {
        if ($value) {
            $nextPriority = \App\Models\Approver::where('approval_rule_id', $value)->max('priority');
            $this->approverPriority = ($nextPriority ?? 0) + 1;
        } else {
            $this->approverPriority = 1;
        }
    }


    public function mount()
    {
        if (Gate::denies('is-admin')) {
            // یا از abort(403) استفاده کنید، یا به صفحه دیگری ریدایرکت کنید
            abort(403, 'شما مجوز مدیریت قوانین تأیید را ندارید.');
        }
    }


    #[Layout('admin.master')]
    public function render()
    {
        $projects = $this->projects;
        $users = $this->users;
        $roles = $this->roles;

        // 💡 فیلتر کلیدی: فقط قوانین نوع 'purchase'
        $rules = ApprovalRule::with(['project', 'approvers.user', 'approvers.role'])
            ->where('type', 'purchase')
            ->when($this->searchRule, function ($query) {
                $query->where('name', 'like', '%' . $this->searchRule . '%')
                    ->orWhereHas('project', fn ($q) => $q->where('name', 'like', '%' . $this->searchRule . '%'));
            })
            ->when($this->statusFilter !== '', fn ($query) => $query->where('is_active', $this->statusFilter))
            ->when($this->isDefaultFilter !== '', fn ($query) => $query->where('is_default', $this->isDefaultFilter))
            ->when($this->activeProject, fn ($query) => $query->where('project_id', $this->activeProject))
            ->orderBy('priority')
            ->paginate(10);

        return view('livewire.admin.approval.approval-management', [
            'rules' => $rules,
            'projects' => $projects,
            'users' => $users,
            'roles' => $roles,
        ]);
    }

    public function saveRule()
    {
        $this->validate($this->rules());

        if ($this->isDefault) {
            ApprovalRule::query()->where('type', 'purchase')->update(['is_default' => 0]);
        }

        ApprovalRule::create([
            'name' => $this->name,
            'project_id' => $this->projectId,
            'amount_min' => $this->amountMin,
            'amount_max' => $this->amountMax,
            'priority' => $this->priority,
            'is_active' => $this->isActive,
            'is_default' => $this->isDefault,
            'type' => 'purchase', // ✅ ست کردن نوع ثابت
        ]);

        $this->resetForm();
        $this->dispatchSuccess('قانون خرید با موفقیت اضافه شد.');
    }

    public function editRule($id)
    {
        $rule = ApprovalRule::findOrFail($id);
        $this->name = $rule->name;
        $this->projectId = $rule->project_id;
        $this->amountMin = $rule->amount_min;
        $this->amountMax = $rule->amount_max;
        $this->priority = $rule->priority;
        $this->isActive = $rule->is_active;
        $this->isDefault = $rule->is_default;
        $this->editRuleId = $rule->id;
    }

    public function updateRule()
    {
        $this->validate($this->rules());

        if ($this->isDefault) {
            ApprovalRule::query()->where('type', 'purchase')->update(['is_default' => 0]);
        }

        $rule = ApprovalRule::findOrFail($this->editRuleId);
        $rule->update([
            'name' => $this->name,
            'project_id' => $this->projectId,
            'amount_min' => $this->amountMin,
            'amount_max' => $this->amountMax,
            'priority' => $this->priority,
            'is_active' => $this->isActive,
            'is_default' => $this->isDefault,
        ]);

        $this->resetForm();
        $this->dispatchSuccess('قانون با موفقیت ویرایش شد.');
    }

    public function resetForm()
    {
        $this->reset(['name', 'projectId', 'amountMin', 'amountMax', 'priority', 'isActive', 'isDefault', 'editRuleId', 'ruleId', 'userId', 'roleId', 'approverPriority']);
        $this->approverPriority = 1;
        $this->resetValidation();
    }

    public function deleteRule($id)
    {
        $rule = ApprovalRule::findOrFail($id);
        if ($rule->type !== 'purchase') {
            $this->dispatchError('فقط مجاز به حذف قوانین خرید هستید.');
            return;
        }
        $rule->delete();
        $this->dispatchSuccess('قانون با موفقیت حذف شد.');
    }

    public function setDefaultRule($id)
    {
        ApprovalRule::query()->where('type', 'purchase')->update(['is_default' => 0]);
        $rule = ApprovalRule::findOrFail($id);

        if ($rule->type !== 'purchase') {
            $this->dispatchError('این قانون متعلق به بخش دیگری است.');
            return;
        }

        $rule->update(['is_default' => 1]);
        $this->dispatchSuccess('قانون پیش‌فرض با موفقیت تعیین شد.');
    }

    public function saveApprover()
    {
        $this->validate($this->approverRules);

        $exists = \App\Models\Approver::where('approval_rule_id', $this->ruleId)
            ->where('user_id', $this->userId)
            ->exists();

        if ($exists) {
            $this->dispatchError('این کاربر قبلاً به این قانون اضافه شده است.');
            return;
        }

        \App\Models\Approver::create([
            'approval_rule_id' => $this->ruleId,
            'user_id' => $this->userId,
            'role_id' => $this->roleId,
            'priority' => $this->approverPriority,
        ]);

        $this->reset(['ruleId', 'userId', 'roleId', 'approverPriority']);
        $this->dispatchSuccess('تأییدکننده با موفقیت اضافه شد.');
    }

    public function deleteApprover($id)
    {
        \App\Models\Approver::findOrFail($id)->delete();
        $this->dispatchSuccess('تأییدکننده با موفقیت حذف شد.');
    }
}
