<?php

namespace App\Livewire\Admin\Attendence;

use App\Models\AttendanceApprovalRule;
use App\Models\Company;
use App\Models\Department;
use App\Models\Project;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('admin.master')]
class ManageApprovalRules extends Component
{
    use WithPagination;

    // حالت‌ها
    public bool $showCreateModal = false;
    public bool $showEditModal = false;
    public ?int $editingRuleId = null;

    // فیلدهای فرم
    public string $name = '';
    public ?string $type = null;
    public ?string $leave_type = null;
    public ?int $project_id = null;
    public ?int $department_id = null;
    public ?int $company_id = null;
    public ?int $min_duration_hours = null;
    public ?int $max_duration_hours = null;
    public int $step_order = 1;
    public array $selectedApprovers = [];
    public bool $is_active = true;

    // فیلترها
    public string $search = '';
    public string $filterType = '';

    // جستجوی پویا برای کاربران
    public string $userSearch = '';
    public array $userOptions = [];

    // لیست شرکت‌ها
    public function getCompaniesProperty()
    {
        return Company::select('id', 'name')->get();
    }

    // لیست دپارتمان‌های فیلتر شده بر اساس شرکت
    public function getFilteredDepartmentsProperty()
    {
        $query = Department::select('id', 'name');
        if ($this->company_id) {
            $query->where('company_id', $this->company_id);
        }
        return $query->get();
    }

    // بارگذاری کاربران بر اساس فیلترهای انتخاب شده
    public function loadUserOptions()
    {
        $query = User::query()
            ->select([
                'id',
                'name',
                'email',
                DB::raw('COALESCE(department_id, 0) as department_id'),
                DB::raw('COALESCE(company_id, 0) as company_id'),
            ]);

        // فیلتر بر اساس شرکت
        if ($this->company_id) {
            $query->where('company_id', $this->company_id);
        }

        // فیلتر بر اساس دپارتمان
        if ($this->department_id) {
            $query->where('department_id', $this->department_id);
        }

        // جستجوی متنی
        if ($this->userSearch) {
            $query->where(function ($q) {
                $q->where('name', 'like', '%' . $this->userSearch . '%')
                    ->orWhere('email', 'like', '%' . $this->userSearch . '%');
            });
        }

        $this->userOptions = $query->limit(20)->get()->map(function ($user) {
            // تبدیل مقادیر 0 به NULL برای نمایش صحیح
            $user->department_id = $user->department_id == 0 ? null : $user->department_id;
            $user->company_id = $user->company_id == 0 ? null : $user->company_id;

            // تبدیل به آرایه و اطمینان از وجود کلیدها
            return [
                'id' => $user->id,
                'name' => $user->name ?? '—',
                'email' => $user->email ?? '—',
                'company_id' => $user->company_id,
                'department_id' => $user->department_id,
            ];
        })->toArray();
    }

    // هنگام تغییر شرکت، دپارتمان و لیست کاربران را ریست کن
    public function updatedCompany_id()
    {
        $this->department_id = null;
        $this->loadUserOptions();
    }

    // هنگام تغییر دپارتمان، لیست کاربران را بارگذاری مجدد کن
    public function updatedDepartment_id()
    {
        $this->loadUserOptions();
    }

    public function mount()
    {
        if (Gate::denies('manage-positions')) {
            abort(403, 'شما مجوز مدیریت جایگاه‌های سازمانی را ندارید.');
        }

        $this->loadUserOptions();
    }

    public function updatedUserSearch()
    {
        $this->loadUserOptions();
    }

    // تغییر وضعیت انتخاب کاربر
    public function toggleUserSelection($userId)
    {
        if (($key = array_search($userId, $this->selectedApprovers)) !== false) {
            unset($this->selectedApprovers[$key]);
        } else {
            $this->selectedApprovers[] = $userId;
        }
        $this->selectedApprovers = array_values($this->selectedApprovers);
    }

    // باز کردن مودال ایجاد
    public function openCreateModal()
    {
        $this->resetForm();
        $this->showCreateModal = true;
    }

    // باز کردن مودال ویرایش
    public function openEditModal($ruleId)
    {
        $rule = AttendanceApprovalRule::findOrFail($ruleId);
        $this->editingRuleId = $ruleId;
        $this->name = $rule->name;
        $this->type = $rule->type;
        $this->leave_type = $rule->leave_type;
        $this->project_id = $rule->project_id;
        $this->department_id = $rule->department_id;
        $this->company_id = $rule->company_id;
        $this->min_duration_hours = $rule->min_duration_hours;
        $this->max_duration_hours = $rule->max_duration_hours;
        $this->step_order = $rule->step_order;
        $this->selectedApprovers = $rule->approvers ?: [];
        $this->is_active = $rule->is_active;

        $this->showEditModal = true;
        $this->loadUserOptions(); // بارگذاری مجدد کاربران بر اساس فیلترهای جدید
    }

    // ریست فرم
    public function resetForm()
    {
        $this->name = '';
        $this->type = null;
        $this->leave_type = null;
        $this->project_id = null;
        $this->department_id = null;
        $this->company_id = null;
        $this->min_duration_hours = null;
        $this->max_duration_hours = null;
        $this->step_order = 1;
        $this->selectedApprovers = [];
        $this->is_active = true;
        $this->userSearch = '';
        $this->loadUserOptions();
    }

    // ذخیره یا به‌روزرسانی قانون
    public function saveRule()
    {
        $this->validate([
            'name' => 'required|string|max:255',
            'step_order' => 'required|integer|min:1',
            'selectedApprovers' => 'required|array|min:1',
            'selectedApprovers.*' => 'exists:users,id',
        ]);

        if ($this->type === 'leave') {
            $this->validate([
                'leave_type' => 'required|string|in:annual,sick,unpaid,overtime_comp,other',
            ]);
        }

        $data = [
            'name' => $this->name,
            'company_id' => $this->company_id,
            'type' => $this->type,
            'leave_type' => $this->leave_type,
            'project_id' => $this->project_id,
            'department_id' => $this->department_id,
            'min_duration_hours' => $this->min_duration_hours,
            'max_duration_hours' => $this->max_duration_hours,
            'step_order' => $this->step_order,
            'approvers' => $this->selectedApprovers,
            'is_active' => $this->is_active,
        ];

        DB::transaction(function () use ($data) {
            if ($this->editingRuleId) {
                $rule = AttendanceApprovalRule::findOrFail($this->editingRuleId);
                $rule->update($data);
                session()->flash('message', '✅ قانون با موفقیت به‌روزرسانی شد.');
            } else {
                AttendanceApprovalRule::create($data);
                session()->flash('message', '✅ قانون جدید با موفقیت ایجاد شد.');
            }
        });

        $this->closeModals();
        $this->resetForm();
        $this->dispatch('$refresh');
    }

    // بستن مودال‌ها
    public function closeModals()
    {
        $this->showCreateModal = false;
        $this->showEditModal = false;
        $this->editingRuleId = null;
    }

    // حذف قانون
    public function deleteRule($ruleId)
    {
        $rule = AttendanceApprovalRule::findOrFail($ruleId);
        $rule->delete();
        session()->flash('message', '🗑️ قانون با موفقیت حذف شد.');
        $this->dispatch('$refresh');
    }

    // دریافت لیست قوانین با فیلتر
    public function getRulesQueryProperty()
    {
        return AttendanceApprovalRule::with([
            'project:id,name',
            'department:id,name',
            'company:id,name'
        ])
            ->when($this->search, function ($query) {
                $query->where('name', 'like', '%' . $this->search . '%');
            })
            ->when($this->filterType, function ($query) {
                $query->where('type', $this->filterType);
            })
            ->orderBy('step_order', 'asc')
            ->paginate(10);
    }

    // دریافت لیست پروژه‌ها
    public function getProjectsProperty()
    {
        return Project::select('id', 'name')->get();
    }

    // رندر ویو
    public function render()
    {
        return view('livewire.admin.attendence.manage-approval-rules', [
            'rules' => $this->rulesQuery,
            'projects' => $this->projects,
            'companies' => $this->companies,
            'filteredDepartments' => $this->filteredDepartments,
        ]);
    }
}
