<?php

namespace App\Livewire\Admin\Attendence\Requests;

use App\Models\LeaveMissionRequest as LMR;
use App\Models\ApprovalStep;
use Illuminate\Validation\ValidationException;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;
use Morilog\Jalali\Jalalian;
use Symfony\Component\HttpFoundation\StreamedResponse;

class ApproverInbox extends Component
{
    use WithPagination;

    /* ---------- URL state / Filters ---------- */
    #[Url] public ?string $search = null;
    #[Url] public ?string $type   = null; // leave | mission
    #[Url] public ?string $from_j = null;
    #[Url] public ?string $to_j   = null;
    #[Url] public string  $sortField = 'submitted_at'; // submitted_at|start_datetime|end_datetime|id
    #[Url] public string  $sortDirection = 'desc';
    #[Url] public int     $perPage = 15;

    /* ---------- Selection / Bulk ---------- */
    public array $selected = [];
    public bool $selectAllOnPage = false;

    /* ---------- Reject modal ---------- */
    public bool $showRejectModal = false;
    public ?int $rejectRequestId = null;
    public string $rejectReason = '';

    /* ---------- Quick view (offcanvas) ---------- */
    public bool $showQuickView = false;
    public ?LMR $quickReq = null;

    /* ---------- Helpers ---------- */
    protected function toJ($dt){ return $dt ? Jalalian::fromCarbon($dt)->format('Y/m/d H:i') : null; }

    /** نرمال‌سازی ارقام فارسی/عربی */
    protected function normalizeDigits(?string $s): string
    {
        if ($s === null) return '';
        $fa = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹','،']; $en = ['0','1','2','3','4','5','6','7','8','9',','];
        $ar = ['٠','١','٢','٣','٤','٥','٦','٧','٨','٩'];
        $s = str_replace($fa, $en, $s);
        $s = str_replace($ar, $en, $s);
        return trim($s);
    }

    protected function jToCarbon(?string $j, string $fmt='Y/m/d H:i'){
        if(!$j || !trim($j)) return null;
        try { return Jalalian::fromFormat($fmt, $this->normalizeDigits($j))->toCarbon(); } catch (\Throwable) { return null; }
    }

    public function updating($field){
        if (in_array($field, ['search','type','from_j','to_j','perPage','sortField','sortDirection'])) {
            $this->resetPage();
        }
        if (!in_array($field, ['selected','selectAllOnPage'])) {
            $this->selected = [];
            $this->selectAllOnPage = false;
        }
    }

    public function sortBy(string $field): void
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = in_array($field, ['submitted_at','id']) ? 'desc' : 'asc';
        }
        $this->resetPage();
    }

    public function resetFilters(): void
    {
        $this->search = $this->type = $this->from_j = $this->to_j = '';
        $this->sortField = 'submitted_at';
        $this->sortDirection = 'desc';
        $this->perPage = 15;
        $this->selected = [];
        $this->selectAllOnPage = false;
        $this->resetPage();
    }

    /* ---------- Date presets ---------- */
    public function presetRange(string $key): void
    {
        $now = now();
        $f = $t = null;
        switch ($key) {
            case 'today':  $f=$now->copy()->startOfDay();  $t=$now->copy()->endOfDay(); break;
            case 'week':   $f=$now->copy()->startOfWeek(); $t=$now->copy()->endOfWeek(); break;
            case 'month':  $f=$now->copy()->startOfMonth();$t=$now->copy()->endOfMonth(); break;
            case 'clear':  $this->from_j=$this->to_j=''; $this->resetPage(); return;
        }
        $this->from_j = $f ? Jalalian::fromCarbon($f)->format('Y/m/d H:i') : '';
        $this->to_j   = $t ? Jalalian::fromCarbon($t)->format('Y/m/d H:i') : '';
        $this->resetPage();
    }

    /* ---------- Quick view ---------- */
    public function openQuickView(int $id): void
    {
        $this->quickReq = LMR::with([
            'user:id,name,email',
            'details',
            'approvals' => fn($q)=> $q->orderBy('step_order')->with('approver:id,name,email')
        ])
            ->pendingFor(auth()->id())
            ->findOrFail($id);

        if ($this->quickReq->details){
            $this->quickReq->details->start_j = $this->toJ($this->quickReq->details->start_datetime);
            $this->quickReq->details->end_j   = $this->toJ($this->quickReq->details->end_datetime);
        }

        $this->showQuickView = true;
        $this->dispatch('open-offcanvas');
    }

    public function closeQuickView(): void
    {
        $this->showQuickView = false;
        $this->quickReq = null;
        $this->dispatch('close-offcanvas');
    }

    /* ---------- Selection ---------- */
    public function toggleSelectAllOnPage(array $ids): void
    {
        $this->selectAllOnPage = !$this->selectAllOnPage;
        $this->selected = $this->selectAllOnPage ? $ids : [];
    }

    /* ---------- Approve / Reject ---------- */
    public function approveQuick(int $requestId): void
    {
        /** @var LMR $req */
        $req = LMR::with('approvals')->findOrFail($requestId);
        $req->lockForUpdate(); // 🔒 قفل گذاری برای جلوگیری از همزمانی

        $cur = $req->currentApproval();
        if (!$cur || !$this->isMine($cur)) {
            throw ValidationException::withMessages(['approve' => 'این آیتم دیگر در نوبت شما نیست.']);
        }

        $cur->update([
            'status'     => ApprovalStep::ST_APPROVED,
            'decided_at' => now(),
            'decided_by' => auth()->id(),
        ]);

        $this->advanceOrFinish($req);
        $this->dispatch('toast', type:'success', text:'✅ درخواست با موفقیت تأیید شد.');
        $this->dispatch('$refresh');
        $this->closeQuickView();
    }

    public function bulkApproveSelected(): void
    {
        if (!count($this->selected)) return;

        $ok = $skip = 0;
        LMR::with('approvals')->whereIn('id',$this->selected)->get()->each(function(LMR $req) use (&$ok,&$skip){
            $req->lockForUpdate(); // 🔒 قفل گذاری

            $cur = $req->currentApproval();
            if (!$cur || !$this->isMine($cur)) {
                $skip++;
                return;
            }

            $cur->update([
                'status'     => ApprovalStep::ST_APPROVED,
                'decided_at' => now(),
                'decided_by' => auth()->id(),
            ]);

            $this->advanceOrFinish($req);
            $ok++;
        });

        $this->selected = [];
        $this->selectAllOnPage = false;
        $msg = "✅ تأیید شد: {$ok}".($skip? " | ⚠️ خارج از نوبت/اسکیپ: {$skip}" : '');
        $this->dispatch('toast', type:'success', text:$msg);
        $this->dispatch('$refresh');
    }

    public function openRejectModal(int $requestId): void
    {
        $this->rejectRequestId = $requestId;
        $this->rejectReason = '';
        $this->showRejectModal = true;
        $this->dispatch('showRejectModal');
    }

    public function closeRejectModal(): void
    {
        $this->showRejectModal = false;
        $this->dispatch('hideRejectModal');
    }

    public function submitReject(): void
    {
        $this->validate([
            'rejectRequestId' => 'required|integer|exists:leave_mission_requests,id',
            'rejectReason'    => 'required|string|min:3|max:2000',
        ]);

        /** @var LMR $req */
        $req = LMR::with('approvals')->findOrFail($this->rejectRequestId);
        $req->lockForUpdate(); // 🔒 قفل گذاری

        $cur = $req->currentApproval();
        if (!$cur || !$this->isMine($cur)) {
            throw ValidationException::withMessages(['reject' => 'این آیتم دیگر در نوبت شما نیست.']);
        }

        $cur->update([
            'status'     => ApprovalStep::ST_REJECTED,
            'comment'    => $this->rejectReason,
            'decided_at' => now(),
            'decided_by' => auth()->id(),
        ]);

        $req->update([
            'status'       => LMR::ST_REJECTED,
            'current_step' => 0,
        ]);

        $this->closeRejectModal();
        $this->dispatch('toast', type:'success', text:'❌ درخواست با ذکر علت رد شد.');
        $this->dispatch('$refresh');
    }

    protected function isMine(ApprovalStep $step): bool
    {
        $me = auth()->id();
        return in_array($me, array_filter([$step->approver_id, $step->delegated_to]));
    }

    protected function advanceOrFinish(LMR $req): void
    {
        $next = ApprovalStep::query()
            ->where('request_id', $req->id)
            ->where('step_order', '>', $req->current_step)
            ->orderBy('step_order')
            ->first();

        if ($next) {
            $req->update(['current_step' => $next->step_order]);
        } else {
            $req->update(['status' => LMR::ST_APPROVED, 'current_step' => 0]);
        }
    }

    /* ---------- Export ---------- */
    public function exportCsv(): StreamedResponse
    {
        $from = $this->jToCarbon($this->from_j);
        $to   = $this->jToCarbon($this->to_j);
        $q = $this->baseQuery($from,$to);
        $filename = 'approver_inbox_'.now()->format('Ymd_His').'.csv';
        return response()->streamDownload(function() use ($q){
            $out = fopen('php://output','w');
            fputcsv($out, ['ID','User','Type','Title','Start','End','Hours','Submitted At']);
            $q->chunk(500, function($chunk) use ($out){
                foreach ($chunk as $r) {
                    fputcsv($out, [
                        $r->id,
                        optional($r->user)->name,
                        $r->type,
                        (string)$r->title,
                        optional($r->details)->start_datetime,
                        optional($r->details)->end_datetime,
                        optional($r->details)->duration_hours,
                        $r->submitted_at,
                    ]);
                }
            });
            fclose($out);
        }, $filename, ['Content-Type'=>'text/csv; charset=UTF-8']);
    }

    /* ---------- Query builder (shared) ---------- */
    protected function baseQuery($from,$to)
    {
        $q = LMR::query()
            ->with([
                'user:id,name,email',
                'details:id,request_id,leave_type,project_id,mission_city,mission_purpose,start_datetime,end_datetime,duration_hours,overlap_ok'
            ])
            ->pendingFor(auth()->id());

        if ($this->type) $q->where('type', $this->type);

        if ($this->search && ($term = '%'.trim($this->search).'%')) {
            $q->where(function($qq) use ($term){
                $qq->where('title','like',$term)
                    ->orWhereHas('user', fn($u)=> $u->where('name','like',$term)->orWhere('email','like',$term))
                    ->orWhereHas('details', function($d) use ($term){
                        $d->where('mission_city','like',$term)
                            ->orWhere('mission_purpose','like',$term)
                            ->orWhere('leave_type','like',$term);
                    });
            });
        }

        if ($from || $to) {
            $q->whereHas('details', function($qd) use ($from,$to){
                if ($from) $qd->where('start_datetime','>=',$from);
                if ($to)   $qd->where('end_datetime','<=',$to);
            });
        }

        // سورت: اگر لازم باشد بر اساس جزئیات سورت کنیم، جوین امن می‌زنیم
        if (in_array($this->sortField, ['start_datetime','end_datetime','duration_hours'])) {
            $q->leftJoin('leave_mission_details as d','d.request_id','=','leave_mission_requests.id')
                ->select('leave_mission_requests.*','d.start_datetime','d.end_datetime','d.duration_hours')
                ->orderBy($this->sortField, $this->sortDirection)
                ->orderBy('leave_mission_requests.id','desc');
        } else {
            $q->orderBy($this->sortField === 'id' ? 'leave_mission_requests.id' : $this->sortField, $this->sortDirection)
                ->orderBy('leave_mission_requests.id','desc');
        }

        return $q;
    }

    /* ---------- Render ---------- */
    #[Layout('admin.master')]
    public function render()
    {
        $from = $this->jToCarbon($this->from_j);
        $to   = $this->jToCarbon($this->to_j);
        $q = $this->baseQuery($from,$to);
        $items = $q->paginate($this->perPage);

        foreach ($items as $r) {
            if($r->details){
                $r->details->start_j = $this->toJ($r->details->start_datetime);
                $r->details->end_j   = $this->toJ($r->details->end_datetime);
            }
        }

        $pageIds = $items->pluck('id')->map(fn($i)=>(string)$i)->all();
        return view('livewire.admin.attendence.requests.approver-inbox', compact('items','pageIds'));
    }
}
