<?php

namespace App\Livewire\Admin\Attendence\Requests;

use App\Models\AttendanceApprovalRule;
use App\Models\LeaveMissionRequest as LMR;
use App\Models\LeaveMissionDetail;
use App\Models\ApprovalStep;
use App\Models\User;
use App\Models\Project;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Morilog\Jalali\Jalalian;
use Carbon\Carbon;

class Create extends Component
{
    /** اگر ست شود وارد حالت ویرایش پیش‌نویس هستیم */
    public ?int $draftId = null;

    #[Validate('required|in:leave,mission')]
    public string $type = 'leave';

    public ?string $title = null;
    public ?string $description = null;

    #[Validate('required|string')]
    public string $start_jalali;

    #[Validate('required|string')]
    public string $end_jalali;

    public bool $is_full_day = true;

    // مرخصی
    public ?string $leave_type = 'annual'; // annual|sick|unpaid|overtime_comp|other

    // مأموریت
    public ?int $project_id = null;
    public string $projectQuery = '';
    public ?string $mission_city = null;
    public ?string $mission_purpose = null;

    // ✅ اجازه همپوشانی زمانی
    public bool $overlap_ok = false;

    /** پیش‌نمایش زنجیرهٔ تأیید */
    public array $approverPreview = [];

    /** حالت ویرایش؟ */
    public function getIsEditModeProperty(): bool
    {
        return !is_null($this->draftId);
    }

    /** مدت خوانا برای نمایش */
    public function getDurationTextProperty(): ?string
    {
        try {
            $start = $this->jalaliToCarbon($this->start_jalali);
            $end   = $this->jalaliToCarbon($this->end_jalali);
            if ($end->lessThanOrEqualTo($start)) return null;

            $minutes = $end->diffInMinutes($start);
            $hours   = intdiv($minutes, 60);
            $mins    = $minutes % 60;

            if ($this->is_full_day && $minutes % (24*60) === 0) {
                return ($minutes / (24*60)).' روز';
            }

            if ($hours >= 24) {
                $d = intdiv($hours, 24);
                $h = $hours % 24;
                $parts = [];
                if ($d) $parts[] = $d.' روز';
                if ($h) $parts[] = $h.' ساعت';
                if ($mins) $parts[] = $mins.' دقیقه';
                return implode(' و ', $parts);
            }

            $parts = [];
            if ($hours) $parts[] = $hours.' ساعت';
            if ($mins)  $parts[] = $mins.' دقیقه';
            return $parts ? implode(' و ', $parts) : 'کمتر از ۱ ساعت';
        } catch (\Throwable $e) {
            return null;
        }
    }

    /** آپشن‌های پروژه برای دراپ‌داون (جستجو بر اساس نام/کد) */
    public function getProjectOptionsProperty()
    {
        $q = trim($this->projectQuery);
        $builder = Project::query()
            ->select(['id','name','code'])
            ->orderByDesc('id')
            ->limit(12);

        if ($q !== '') {
            $builder->where(function ($qq) use ($q) {
                $qq->where('name', 'like', "%{$q}%")
                    ->orWhere('code', 'like', "%{$q}%");
            });
        }

        // اگر آیتم انتخاب‌شده در لیست نیست، آن را هم اضافه کنیم
        $list = $builder->get()->keyBy('id');
        if ($this->project_id && !$list->has($this->project_id)) {
            if ($p = Project::select(['id','name','code'])->find($this->project_id)) {
                $list->put($p->id, $p);
            }
        }

        return $list->values();
    }

    public function mount(?int $draft = null)
    {
        // مقدار اولیه: امروز تمام‌روز (به شمسی)
        $this->start_jalali = Jalalian::fromCarbon(now()->startOfDay())->format('Y/m/d H:i');
        $this->end_jalali   = Jalalian::fromCarbon(now()->endOfDay())->format('Y/m/d H:i');

        if ($draft) {
            $this->loadDraft($draft);
        }

        // پیش‌نمایش اولیهٔ زنجیرهٔ تایید
        $this->recomputeApproverPreview();
    }

    protected function loadDraft(int $draftId): void
    {
        $req = LMR::with('details')
            ->where('id', $draftId)
            ->where('user_id', auth()->id())
            ->where('status', LMR::ST_DRAFT)
            ->first();

        if (!$req) {
            session()->flash('ok', 'پیش‌نویس پیدا نشد یا قابل ویرایش نیست.');
            redirect()->route('admin.attendence.requests.index')->send();
            return;
        }

        $this->draftId     = $req->id;
        $this->type        = $req->type;
        $this->title       = $req->title;
        $this->description = $req->description;

        if ($req->details) {
            $this->start_jalali = Jalalian::fromCarbon($req->details->start_datetime)->format('Y/m/d H:i');
            $this->end_jalali   = Jalalian::fromCarbon($req->details->end_datetime)->format('Y/m/d H:i');
            $this->is_full_day  = (bool)$req->details->is_full_day;
            $this->overlap_ok   = (bool)$req->details->overlap_ok;

            if ($req->type === 'leave') {
                $this->leave_type = $req->details->leave_type ?: 'annual';
            } else {
                $this->project_id      = $req->details->project_id;
                $this->mission_city    = $req->details->mission_city;
                $this->mission_purpose = $req->details->mission_purpose;
            }
        }
    }

    #[Layout('admin.master')]
    public function render()
    {
        $projects = $this->projectOptions; // Collection
        return view('livewire.admin.attendence.requests.create', compact('projects'));
    }

    /** قوانین اعتبارسنجی پویا */
    protected function rules(): array
    {
        $rules = [
            'type'         => 'required|in:leave,mission',
            'title'        => 'nullable|string|max:150',
            'description'  => 'nullable|string|max:2000',
            'start_jalali' => 'required|string',
            'end_jalali'   => 'required|string',
            'is_full_day'  => 'boolean',
            'overlap_ok'   => 'boolean',
        ];

        if ($this->type === 'leave') {
            $rules['leave_type'] = 'required|in:annual,sick,unpaid,overtime_comp,other';
        } else {
            $rules['project_id']      = 'required|integer|exists:projects,id';
            $rules['mission_city']    = 'nullable|string|max:255';
            $rules['mission_purpose'] = 'nullable|string|max:500';
        }

        return $rules;
    }

    /** تبدیل ارقام فارسی/عربی به لاتین */
    protected function normalizeDigits(?string $s): string
    {
        if ($s === null) return '';
        $fa = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹','،'];
        $en = ['0','1','2','3','4','5','6','7','8','9',','];
        $ar = ['٠','١','٢','٣','٤','٥','٦','٧','٨','٩'];
        $s = str_replace($fa, $en, $s);
        $s = str_replace($ar, $en, $s);
        return trim($s);
    }

    /** تبدیل رشته شمسی به Carbon */
    protected function jalaliToCarbon(string $jalali, string $format='Y/m/d H:i'): Carbon
    {
        $jalali = $this->normalizeDigits($jalali);
        return Jalalian::fromFormat($format, $jalali)->toCarbon();
    }

    /** تغییر تمام‌روز => ساعات نرمال شود */
    public function updatedIsFullDay(bool $val): void
    {
        try {
            $start = $this->jalaliToCarbon($this->start_jalali);
            $end   = $this->jalaliToCarbon($this->end_jalali);

            if ($val) {
                $start = $start->copy()->startOfDay();
                $end   = $end->copy()->endOfDay();
            }

            $this->start_jalali = Jalalian::fromCarbon($start)->format('Y/m/d H:i');
            $this->end_jalali   = Jalalian::fromCarbon($end)->format('Y/m/d H:i');
        } catch (\Throwable $e) {
            // ignore
        }
    }

    /** تغییرات اثرگذار بر زنجیرهٔ تأیید → پیش‌نمایش را به‌روز کن */
    public function updated($field): void
    {
        if (in_array($field, ['type','project_id','leave_type','mission_city','mission_purpose'])) {
            $this->recomputeApproverPreview();
        }
    }

    /** میان‌بُرهای تاریخ */
    public function preset(string $key): void
    {
        $now = now();
        switch ($key) {
            case 'today':
                $start = $now->copy()->startOfDay();
                $end   = $now->copy()->endOfDay();
                $this->is_full_day = true;
                break;
            case 'tomorrow':
                $start = $now->copy()->addDay()->startOfDay();
                $end   = $now->copy()->addDay()->endOfDay();
                $this->is_full_day = true;
                break;
            case 'half_am':
                $start = $now->copy()->startOfDay()->setTime(8, 0);
                $end   = $now->copy()->startOfDay()->setTime(12, 0);
                $this->is_full_day = false;
                break;
            case 'half_pm':
                $start = $now->copy()->startOfDay()->setTime(13, 0);
                $end   = $now->copy()->startOfDay()->setTime(17, 0);
                $this->is_full_day = false;
                break;
            case 'next2days':
                $start = $now->copy()->startOfDay();
                $end   = $now->copy()->addDays(2)->endOfDay();
                $this->is_full_day = true;
                break;
            default:
                return;
        }

        $this->start_jalali = Jalalian::fromCarbon($start)->format('Y/m/d H:i');
        $this->end_jalali   = Jalalian::fromCarbon($end)->format('Y/m/d H:i');
    }

    /** ذخیره پیش‌نویس (ساخت/ویرایش) */
    public function saveDraft()
    {
        $this->validate();

        $start = $this->jalaliToCarbon($this->start_jalali);
        $end   = $this->jalaliToCarbon($this->end_jalali);

        if ($end->lessThanOrEqualTo($start)) {
            $this->addError('end_jalali','پایان باید بعد از شروع باشد.');
            return;
        }

        DB::transaction(function() use ($start, $end){
            if ($this->draftId) {
                /** @var LMR $req */
                $req = LMR::with('details')
                    ->where('id', $this->draftId)
                    ->where('user_id', auth()->id())
                    ->where('status', LMR::ST_DRAFT)
                    ->lockForUpdate()
                    ->firstOrFail();

                $req->update([
                    'type'        => $this->type,
                    'title'       => $this->title,
                    'description' => $this->description,
                ]);

                $hours = max(1, (int)ceil($end->diffInMinutes($start) / 60));
                $detailData = [
                    'leave_type'      => $this->type==='leave' ? $this->leave_type : null,
                    'project_id'      => $this->type==='mission' ? $this->project_id : null,
                    'mission_city'    => $this->type==='mission' ? $this->mission_city : null,
                    'mission_purpose' => $this->type==='mission' ? $this->mission_purpose : null,
                    'start_datetime'  => $start,
                    'end_datetime'    => $end,
                    'duration_hours'  => $hours,
                    'is_full_day'     => $this->is_full_day,
                    'overlap_ok'      => $this->overlap_ok,
                ];

                if ($req->details) {
                    $req->details->update($detailData);
                } else {
                    LeaveMissionDetail::create(array_merge($detailData, [
                        'request_id' => $req->id,
                    ]));
                }
            } else {
                $req = LMR::create([
                    'user_id'       => auth()->id(),
                    'company_id'    => auth()->user()->company_id,
                    'department_id' => auth()->user()->department_id,
                    'type'          => $this->type,
                    'status'        => LMR::ST_DRAFT,
                    'title'         => $this->title,
                    'description'   => $this->description,
                ]);

                $hours = max(1, (int)ceil($end->diffInMinutes($start) / 60));
                LeaveMissionDetail::create([
                    'request_id'      => $req->id,
                    'leave_type'      => $this->type==='leave' ? $this->leave_type : null,
                    'project_id'      => $this->type==='mission' ? $this->project_id : null,
                    'mission_city'    => $this->type==='mission' ? $this->mission_city : null,
                    'mission_purpose' => $this->type==='mission' ? $this->mission_purpose : null,
                    'start_datetime'  => $start,
                    'end_datetime'    => $end,
                    'duration_hours'  => $hours,
                    'is_full_day'     => $this->is_full_day,
                    'overlap_ok'      => $this->overlap_ok,
                ]);

                $this->draftId = $req->id;
            }
        });

        // آپدیت پیش‌نمایش پس از ذخیره
        $this->recomputeApproverPreview();
        session()->flash('ok', $this->isEditMode ? 'پیش‌نویس به‌روزرسانی شد.' : 'پیش‌نویس ذخیره شد.');
        return redirect()->route('admin.attendence.requests.index');
    }

    /** ارسال جهت تأیید */
    public function submit()
    {
        $this->validate();

        $start = $this->jalaliToCarbon($this->start_jalali);
        $end   = $this->jalaliToCarbon($this->end_jalali);

        if ($end->lessThanOrEqualTo($start)) {
            $this->addError('end_jalali','پایان باید بعد از شروع باشد.');
            return;
        }

        DB::transaction(function() use ($start,$end){
            if ($this->draftId) {
                /** @var LMR $req */
                $req = LMR::with('details')
                    ->where('id', $this->draftId)
                    ->where('user_id', auth()->id())
                    ->where('status', LMR::ST_DRAFT)
                    ->lockForUpdate()
                    ->firstOrFail();

                $req->update([
                    'type'        => $this->type,
                    'title'       => $this->title,
                    'description' => $this->description,
                ]);

                $hours = max(1, (int)ceil($end->diffInMinutes($start)/60));
                $detailData = [
                    'leave_type'      => $this->type==='leave' ? $this->leave_type : null,
                    'project_id'      => $this->type==='mission' ? $this->project_id : null,
                    'mission_city'    => $this->type==='mission' ? $this->mission_city : null,
                    'mission_purpose' => $this->type==='mission' ? $this->mission_purpose : null,
                    'start_datetime'  => $start,
                    'end_datetime'    => $end,
                    'duration_hours'  => $hours,
                    'is_full_day'     => $this->is_full_day,
                    'overlap_ok'      => $this->overlap_ok,
                ];

                if ($req->details) {
                    $req->details->update($detailData);
                } else {
                    LeaveMissionDetail::create(array_merge($detailData, [
                        'request_id' => $req->id,
                    ]));
                }
            } else {
                $req = LMR::create([
                    'user_id'       => auth()->id(),
                    'company_id'    => auth()->user()->company_id,
                    'department_id' => auth()->user()->department_id,
                    'type'          => $this->type,
                    'status'        => LMR::ST_DRAFT,
                    'title'         => $this->title,
                    'description'   => $this->description,
                ]);

                $hours = max(1, (int)ceil($end->diffInMinutes($start)/60));
                LeaveMissionDetail::create([
                    'request_id'      => $req->id,
                    'leave_type'      => $this->type==='leave' ? $this->leave_type : null,
                    'project_id'      => $this->type==='mission' ? $this->project_id : null,
                    'mission_city'    => $this->type==='mission' ? $this->mission_city : null,
                    'mission_purpose' => $this->type==='mission' ? $this->mission_purpose : null,
                    'start_datetime'  => $start,
                    'end_datetime'    => $end,
                    'duration_hours'  => $hours,
                    'is_full_day'     => $this->is_full_day,
                    'overlap_ok'      => $this->overlap_ok,
                ]);
            }

            // ✅ بررسی همپوشانی فقط اگر overlap_ok فعال نباشد
            if (!$this->overlap_ok) {
                $overlap = LMR::query()
                    ->where('user_id', auth()->id())
                    ->whereIn('status', [LMR::ST_PENDING, LMR::ST_APPROVED])
                    ->whereHas('details', function($q) use ($start,$end){
                        $q->where(function($qq) use ($start,$end){
                            $qq->whereBetween('start_datetime', [$start, $end])
                                ->orWhereBetween('end_datetime',   [$start, $end])
                                ->orWhere(function($qx) use ($start,$end){
                                    $qx->where('start_datetime','<=',$start)
                                        ->where('end_datetime','>=', $end);
                                });
                        });
                    })
                    ->exists();

                if ($overlap) {
                    throw new \Exception('همپوشانی زمانی با درخواست‌های قبلی وجود دارد. در صورت نیاز، گزینه "اجازه همپوشانی زمانی" را فعال کنید.');
                }
            }

            // ✅ زنجیره تأیید جدید: استفاده از AttendanceApprovalRule
            $req = $req ?? LMR::latest('id')->where('user_id',auth()->id())->first();
            ApprovalStep::where('request_id', $req->id)->delete();

            $steps = $this->computeChainWithRules($start, $end, $hours);
            foreach ($steps as $i=>$s){
                ApprovalStep::create([
                    'request_id' => $req->id,
                    'step_order' => $i+1,
                    'approver_id'=> $s['approver_id'],
                    'role_hint'  => $s['role_hint'] ?? null,
                ]);
            }

            $req->update([
                'status'       => LMR::ST_PENDING,
                'current_step' => count($steps) ? 1 : 0,
                'submitted_at' => now(),
            ]);
        });

        session()->flash('ok','درخواست ارسال شد.');
        return redirect()->route('admin.attendence.requests.index');
    }

    /** ✅ منطق جدید: محاسبه زنجیره تأیید بر اساس قوانین */
    protected function computeChainWithRules($start, $end, $durationHours): array
    {
        $user = auth()->user();
        $steps = [];

        // جستجوی قوانین مطابق
        $matchingRules = AttendanceApprovalRule::where('is_active', true)
            ->where(function ($q) use ($user) {
                $q->whereNull('company_id')
                    ->orWhere('company_id', $user->company_id);
            })
            ->where(function ($q) use ($user) {
                $q->whereNull('department_id')
                    ->orWhere('department_id', $user->department_id);
            })
            ->where(function ($q) {
                $q->whereNull('type')
                    ->orWhere('type', $this->type);
            })
            ->when($this->type === 'leave', function ($q) {
                $q->where(function ($qq) {
                    $qq->whereNull('leave_type')
                        ->orWhere('leave_type', $this->leave_type);
                });
            })
            ->when($this->type === 'mission', function ($q) {
                $q->where(function ($qq) {
                    $qq->whereNull('project_id')
                        ->orWhere('project_id', $this->project_id);
                });
            })
            ->where(function ($q) use ($durationHours) {
                $q->whereNull('min_duration_hours')
                    ->orWhere('min_duration_hours', '<=', $durationHours);
            })
            ->where(function ($q) use ($durationHours) {
                $q->whereNull('max_duration_hours')
                    ->orWhere('max_duration_hours', '>=', $durationHours);
            })
            ->orderBy('step_order', 'asc')
            ->get();

        foreach ($matchingRules as $rule) {
            $approvers = $rule->approvers ?: [];
            foreach ($approvers as $approverId) {
                if ($approverId != $user->id) { // جلوگیری از تأیید خودکار
                    $steps[] = [
                        'approver_id' => $approverId,
                        'role_hint' => 'custom_rule', // یا می‌توانید نام قانون را ذخیره کنید
                    ];
                }
            }
        }

        // یکتاسازی و حفظ ترتیب
        $seen = [];
        $uniqueSteps = [];
        foreach ($steps as $step) {
            if (!in_array($step['approver_id'], $seen)) {
                $seen[] = $step['approver_id'];
                $uniqueSteps[] = $step;
            }
        }

        return $uniqueSteps;
    }

    /** پرکردن پیش‌نمایش UI از روی computeChainWithRules */
    protected function recomputeApproverPreview(): void
    {
        try {
            $start = $this->jalaliToCarbon($this->start_jalali);
            $end   = $this->jalaliToCarbon($this->end_jalali);
            $durationHours = max(1, (int)ceil($end->diffInMinutes($start) / 60));

            $steps = $this->computeChainWithRules($start, $end, $durationHours);
        } catch (\Throwable $e) {
            $steps = [];
        }

        $uids  = array_values(array_unique(array_column($steps, 'approver_id')));
        if (empty($uids)) {
            $this->approverPreview = [];
            return;
        }

        $users = User::whereIn('id', $uids)->get()->keyBy('id');
        $roleLabel = [
            'custom_rule'   => 'تأییدکننده',
            'dept_manager'  => 'مدیر واحد',
            'upper_manager' => 'مدیر بالادست',
            'hr'            => 'منابع انسانی',
        ];

        $preview = [];
        foreach ($steps as $s) {
            $u = $users->get($s['approver_id']);
            $preview[] = [
                'id'        => $s['approver_id'],
                'name'      => $u->name ?? '—',
                'email'     => $u->email ?? null,
                'role_hint' => $s['role_hint'] ?? null,
                'role_text' => $roleLabel[$s['role_hint'] ?? ''] ?? 'تأییدکننده',
            ];
        }

        $this->approverPreview = $preview;
    }
}
