<?php

namespace App\Livewire\Admin\Attendence\Requests;

use App\Models\LeaveMissionRequest as LMR;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;
use Morilog\Jalali\Jalalian;
use Illuminate\Support\Str;
use Symfony\Component\HttpFoundation\StreamedResponse;

class Index extends Component
{
    use WithPagination;

    // فیلترها و یوآرال تمیز
    #[Url] public ?string $type    = null;   // leave|mission
    #[Url] public ?string $status  = null;   // draft|pending|approved|rejected|canceled
    #[Url] public ?string $search  = null;
    #[Url] public ?string $from_j  = null;
    #[Url] public ?string $to_j    = null;
    #[Url] public int     $perPage = 15;

    // سورت
    #[Url] public string $sortField = 'submitted_at'; // id|title|submitted_at|start_datetime|end_datetime|duration_hours
    #[Url] public string $sortDirection = 'desc';

    // انتخاب گروهی (فقط پیش‌نویس‌ها اکشن دارند)
    public array $selected = [];
    public bool $selectAllOnPage = false;

    protected $queryString = [
        'type'          => ['except' => ''],
        'status'        => ['except' => ''],
        'search'        => ['except' => ''],
        'from_j'        => ['except' => ''],
        'to_j'          => ['except' => ''],
        'perPage'       => ['except' => 15],
        'sortField'     => ['except' => 'submitted_at'],
        'sortDirection' => ['except' => 'desc'],
    ];

    /* ---------------- Helpers ---------------- */

    protected function toJ($dt){ return $dt ? Jalalian::fromCarbon($dt)->format('Y/m/d H:i') : null; }

    /** تبدیل ارقام فارسی/عربی به لاتین */
    protected function normalizeDigits(?string $s): string
    {
        if ($s === null) return '';
        $fa = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹','،']; $en = ['0','1','2','3','4','5','6','7','8','9',','];
        $ar = ['٠','١','٢','٣','٤','٥','٦','٧','٨','٩'];
        $s = str_replace($fa, $en, $s);
        $s = str_replace($ar, $en, $s);
        return trim($s);
    }

    protected function jToCarbon(?string $jalali, string $format='Y/m/d H:i'){
        if(!$jalali || !trim($jalali)) return null;
        try {
            return Jalalian::fromFormat($format, $this->normalizeDigits($jalali))->toCarbon();
        } catch (\Throwable $e) {
            return null;
        }
    }

    /* ---------------- Reactive tweaks ---------------- */

    public function updating($field) {
        if (in_array($field, ['type','status','search','from_j','to_j','perPage','sortField','sortDirection'])) {
            $this->resetPage();
        }
        if (!in_array($field, ['selected','selectAllOnPage'])) {
            $this->selected = [];
            $this->selectAllOnPage = false;
        }
    }

    public function sortBy(string $field): void
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = in_array($field, ['submitted_at','id']) ? 'desc' : 'asc';
        }
        $this->resetPage();
    }

    public function resetFilters(): void
    {
        $this->type = $this->status = $this->search = $this->from_j = $this->to_j = '';
        $this->perPage = 15;
        $this->sortField = 'submitted_at';
        $this->sortDirection = 'desc';
        $this->selected = [];
        $this->selectAllOnPage = false;
        $this->resetPage();
    }

    /* ---------------- Date quick presets ---------------- */

    public function presetRange(string $key): void
    {
        $now = now();
        $f = $t = null;

        switch ($key) {
            case 'today': $f=$now->copy()->startOfDay();  $t=$now->copy()->endOfDay();   break;
            case 'week':  $f=$now->copy()->startOfWeek(); $t=$now->copy()->endOfWeek();  break;
            case 'month': $f=$now->copy()->startOfMonth();$t=$now->copy()->endOfMonth(); break;
            case 'clear': $this->from_j=$this->to_j=''; $this->resetPage(); return;
        }

        $this->from_j = $f ? Jalalian::fromCarbon($f)->format('Y/m/d H:i') : '';
        $this->to_j   = $t ? Jalalian::fromCarbon($t)->format('Y/m/d H:i') : '';
        $this->resetPage();
    }

    /* ---------------- Bulk actions ---------------- */

    public function toggleSelectAllOnPage(array $ids): void
    {
        $this->selectAllOnPage = !$this->selectAllOnPage;
        $this->selected = $this->selectAllOnPage ? $ids : [];
    }

    public function bulkDeleteDrafts(): void
    {
        if (empty($this->selected)) return;

        LMR::query()
            ->whereIn('id', $this->selected)
            ->where('user_id', auth()->id())
            ->where('status', LMR::ST_DRAFT)
            ->delete();

        $this->selected = [];
        $this->selectAllOnPage = false;
        session()->flash('ok','✅ پیش‌نویس‌های انتخاب‌شده با موفقیت حذف شدند.');
        $this->resetPage();
    }

    /* ---------------- Export ---------------- */

    public function exportCsv(): StreamedResponse
    {
        $from = $this->jToCarbon($this->from_j);
        $to   = $this->jToCarbon($this->to_j);

        $q = $this->baseQuery($from, $to);

        $filename = 'requests_'.now()->format('Ymd_His').'.csv';

        return response()->streamDownload(function() use ($q){
            $out = fopen('php://output', 'w');
            fputcsv($out, ['ID','Type','Title','Status','Start','End','Hours','Submitted At']);
            $q->chunk(500, function($chunk) use ($out){
                foreach ($chunk as $r) {
                    fputcsv($out, [
                        $r->id,
                        $r->type,
                        Str::limit((string)$r->title, 80),
                        $r->status,
                        optional($r->details)->start_datetime,
                        optional($r->details)->end_datetime,
                        optional($r->details)->duration_hours,
                        $r->submitted_at,
                    ]);
                }
            });
            fclose($out);
        }, $filename, ['Content-Type' => 'text/csv; charset=UTF-8']);
    }

    /* ---------------- Query Builder ---------------- */

    protected function baseQuery($from, $to)
    {
        $q = LMR::query()
            ->with([
                'details:id,request_id,leave_type,project_id,mission_city,mission_purpose,start_datetime,end_datetime,duration_hours'
            ])
            ->mine(); // اسکوپ مربوط به کاربر جاری

        if ($this->type)   $q->where('type',$this->type);
        if ($this->status) $q->where('status',$this->status);

        if ($this->search && ($s = trim($this->search))) {
            $norm = $this->normalizeDigits($s);
            $q->where(function($qq) use ($s,$norm){
                // جستجوی مستقیم آی‌دی (#123 یا 123)
                if (preg_match('/^#?\d+$/', $norm)) {
                    $qq->where('leave_mission_requests.id', (int)ltrim($norm, '#'));
                } else {
                    $qq->where('title','like',"%{$s}%")
                        ->orWhere('description','like',"%{$s}%")
                        ->orWhereHas('details', function($qd) use ($s){
                            $qd->where('mission_city','like',"%{$s}%")
                                ->orWhere('mission_purpose','like',"%{$s}%")
                                ->orWhere('leave_type','like',"%{$s}%");
                        });
                }
            });
        }

        if ($from || $to) {
            $q->whereHas('details', function($qd) use ($from,$to){
                if ($from) $qd->where('start_datetime','>=',$from);
                if ($to)   $qd->where('end_datetime','<=',$to);
            });
        }

        // سورت ایمن
        if (in_array($this->sortField, ['start_datetime','end_datetime','duration_hours'])) {
            $q->leftJoin('leave_mission_details as d','d.request_id','=','leave_mission_requests.id')
                ->select('leave_mission_requests.*','d.start_datetime','d.end_datetime','d.duration_hours')
                ->orderBy($this->sortField, $this->sortDirection)
                ->orderBy('leave_mission_requests.id','desc');
        } else {
            $field = $this->sortField === 'id' ? 'leave_mission_requests.id' : $this->sortField;
            $q->orderBy($field, $this->sortDirection)
                ->orderBy('leave_mission_requests.id','desc');
        }

        return $q;
    }

    /* ---------------- Render ---------------- */

    #[Layout('admin.master')]
    public function render()
    {
        $from = $this->jToCarbon($this->from_j);
        $to   = $this->jToCarbon($this->to_j);

        // برای آمار، همان فیلترها اعمال شوند
        $statsQ = $this->baseQuery($from,$to)->cloneWithout(['orders','columns','joins']);
        $statsQ->select('leave_mission_requests.*');

        $total    = (clone $statsQ)->count();
        $draft    = (clone $statsQ)->where('status', LMR::ST_DRAFT)->count();
        $pending  = (clone $statsQ)->where('status', LMR::ST_PENDING)->count();
        $approved = (clone $statsQ)->where('status', LMR::ST_APPROVED)->count();
        $rejected = (clone $statsQ)->where('status', LMR::ST_REJECTED)->count();

        // لیست
        $items = $this->baseQuery($from,$to)->paginate($this->perPage);

        foreach ($items as $r) {
            if($r->details){
                $r->details->start_j = $this->toJ($r->details->start_datetime);
                $r->details->end_j   = $this->toJ($r->details->end_datetime);
            }
        }

        $pageIds = $items->pluck('id')->map(fn($i)=>(string)$i)->all();

        return view('livewire.admin.attendence.requests.index', compact(
            'items','total','draft','pending','approved','rejected','pageIds'
        ));
    }
}
