<?php

namespace App\Livewire\Admin\Attendence\Requests;

use App\Models\ApprovalStep;
use App\Models\LeaveMissionRequest as LMR;
use App\Models\Project;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Morilog\Jalali\Jalalian;
use Symfony\Component\HttpFoundation\StreamedResponse;

class Show extends Component
{
    public int $requestId;
    public LMR $req;

    // مدال تصمیم
    public bool $showDecisionModal = false;
    public string $decisionAction = ''; // approve|reject
    public ?string $decisionNote = null;

    /** وضعیت‌ها برای بَج */
    public array $statusMap = [
        LMR::ST_DRAFT    => ['label' => 'پیش‌نویس',        'class' => 'text-bg-secondary'],
        LMR::ST_PENDING  => ['label' => 'در انتظار تأیید', 'class' => 'text-bg-warning'],
        LMR::ST_APPROVED => ['label' => 'تأیید شده',       'class' => 'text-bg-success'],
        LMR::ST_REJECTED => ['label' => 'رد شده',          'class' => 'text-bg-danger'],
        LMR::ST_CANCELED => ['label' => 'لغو شده',         'class' => 'text-bg-dark'],
    ];

    public function mount(int $requestId): void
    {
        $this->requestId = $requestId;
        $this->loadRequest();
    }

    #[Layout('admin.master')]
    public function render()
    {
        $project = null;
        if ($this->req->details?->project_id) {
            $project = Project::select('id','name','code')->find($this->req->details->project_id);
        }

        // پیشرفت زنجیره تأیید
        $stepsTotal = $this->req->approvals->count();
        $stepsDone  = $this->req->approvals->where('status', ApprovalStep::ST_APPROVED)->count();
        $progress   = $stepsTotal ? (int)round(($stepsDone / $stepsTotal) * 100) : 100;

        // تأییدکنندگان بعدی (از قدم جاری به بعد که PENDING هستند)
        $upcomingApprovers = $this->req->approvals
            ->where('status', ApprovalStep::ST_PENDING)
            ->filter(fn($ap) => !$this->req->current_step || $ap->step_order >= $this->req->current_step)
            ->sortBy('step_order')
            ->values();

        return view('livewire.admin.attendence.requests.show', [
            'fJ'                 => fn($dt) => $this->fJ($dt),
            'durationText'       => fn($hours) => $this->durationText($hours),
            'approvalLabel'      => fn($st) => $this->approvalLabel($st),
            'approvalBadge'      => fn($st) => $this->approvalBadgeClass($st),
            'amCurrentApprover'  => $this->amCurrentApprover(),
            'canWithdraw'        => $this->canWithdraw(),
            'project'            => $project,
            'stepsTotal'         => $stepsTotal,
            'stepsDone'          => $stepsDone,
            'progress'           => $progress,
            'upcomingApprovers'  => $upcomingApprovers,
        ]);
    }

    protected function loadRequest(): void
    {
        $this->req = LMR::with([
            'user:id,name',
            'details',
            'approvals' => fn($q) => $q->orderBy('step_order')->with('approver:id,name'),
        ])->findOrFail($this->requestId);

        foreach ($this->req->approvals as $ap) {
            $ap->is_current = ($this->req->current_step === (int)$ap->step_order)
                && $ap->status === ApprovalStep::ST_PENDING;
        }
    }

    protected function amCurrentApprover(): bool
    {
        if ($this->req->status !== LMR::ST_PENDING) return false;
        $current = $this->req->approvals->firstWhere('is_current', true);
        if (!$current) return false;
        $uid = auth()->id();
        return in_array($uid, array_filter([$current->approver_id, $current->delegated_to]));
    }

    protected function canWithdraw(): bool
    {
        return $this->req->status === LMR::ST_PENDING && $this->req->user_id === auth()->id();
    }

    // --- مدال تصمیم (با یادداشت) ---
    public function openDecisionModal(string $action): void
    {
        if (!in_array($action, ['approve','reject'])) return;
        if (!$this->amCurrentApprover()) {
            session()->flash('err', 'شما تأییدکنندهٔ این قدم نیستید.');
            return;
        }
        $this->decisionAction = $action;
        $this->decisionNote = null;
        $this->showDecisionModal = true;
    }

    public function submitDecision(): void
    {
        if (!$this->amCurrentApprover()) {
            session()->flash('err', 'شما تأییدکنندهٔ این قدم نیستید.');
            $this->showDecisionModal = false;
            return;
        }

        DB::transaction(function () {
            /** @var ApprovalStep $cur */
            $cur = ApprovalStep::where('request_id', $this->req->id)
                ->where('step_order', $this->req->current_step)
                ->lockForUpdate()
                ->firstOrFail();

            if ($cur->status !== ApprovalStep::ST_PENDING) return;

            $isApprove = $this->decisionAction === 'approve';

            // توجه: برای سازگاری با سایر بخش‌ها از فیلد `comment` استفاده می‌کنیم
            $cur->update([
                'status'     => $isApprove ? ApprovalStep::ST_APPROVED : ApprovalStep::ST_REJECTED,
                'decided_at' => now(),
                'decided_by' => auth()->id(),
                'comment'    => $this->decisionNote,
            ]);

            if ($isApprove) {
                $next = ApprovalStep::where('request_id', $this->req->id)
                    ->where('step_order', '>', $cur->step_order)
                    ->orderBy('step_order')
                    ->first();

                if ($next) {
                    $this->req->update(['current_step' => $next->step_order]);
                } else {
                    $this->req->update([
                        'status'       => LMR::ST_APPROVED,
                        'current_step' => 0,
                    ]);
                }
            } else {
                $this->req->update([
                    'status'       => LMR::ST_REJECTED,
                    'current_step' => 0,
                ]);
            }
        });

        $this->showDecisionModal = false;
        session()->flash('ok', $this->decisionAction === 'approve' ? '✅ تأیید ثبت شد.' : '❌ رد درخواست ثبت شد.');
        $this->loadRequest();
    }

    public function withdraw(): void
    {
        if (!$this->canWithdraw()) {
            session()->flash('err', 'امکان لغو برای شما یا در این وضعیت وجود ندارد.');
            return;
        }

        DB::transaction(function () {
            $this->req->update([
                'status'       => LMR::ST_CANCELED,
                'current_step' => 0,
            ]);
        });

        session()->flash('ok', '✅ درخواست با موفقیت لغو شد.');
        $this->loadRequest();
    }

    /** خروجی ICS برای افزودن به تقویم */
    public function downloadIcs(): ?StreamedResponse
    {
        if (!$this->req->details) return null;

        $start = Carbon::parse($this->req->details->start_datetime)->clone()->utc();
        $end   = Carbon::parse($this->req->details->end_datetime)->clone()->utc();

        $summary = ($this->req->type==='leave' ? 'مرخصی' : 'ماموریت')
            . ($this->req->title ? ' - '.$this->req->title : '');
        $desc = trim(($this->req->description ?: ''));

        $uid = 'request-'.$this->req->id.'@'.parse_url(config('app.url') ?? 'example.com', PHP_URL_HOST);

        $ics = "BEGIN:VCALENDAR\r\nVERSION:2.0\r\nPRODID:-//HR//LeaveMission//FA\r\nCALSCALE:GREGORIAN\r\n".
            "BEGIN:VEVENT\r\nUID:{$uid}\r\nDTSTAMP:".now('UTC')->format('Ymd\THis\Z')."\r\n".
            "DTSTART:".$start->format('Ymd\THis\Z')."\r\n".
            "DTEND:".$end->format('Ymd\THis\Z')."\r\n".
            "SUMMARY:".addcslashes($summary, ",;\\")."\r\n".
            "DESCRIPTION:".addcslashes($desc, ",;\\")."\r\nEND:VEVENT\r\nEND:VCALENDAR\r\n";

        return response()->streamDownload(
            fn() => print($ics),
            'request-'.$this->req->id.'.ics',
            ['Content-Type' => 'text/calendar; charset=UTF-8']
        );
    }

    // -------- Helpers --------
    protected function fJ(null|Carbon|string $dt): string
    {
        if (!$dt) return '—';
        $c = $dt instanceof Carbon ? $dt : Carbon::parse($dt);
        return Jalalian::fromCarbon($c)->format('Y/m/d H:i');
    }

    protected function durationText(int $hours): string
    {
        if ($hours <= 0) return '—';
        $d = intdiv($hours, 24);
        $h = $hours % 24;
        $parts = [];
        if ($d) $parts[] = $d.' روز';
        if ($h) $parts[] = $h.' ساعت';
        return $parts ? implode(' و ', $parts) : 'کمتر از ۱ ساعت';
    }

    protected function approvalLabel(string $st): string
    {
        return [
            ApprovalStep::ST_PENDING  => 'در انتظار',
            ApprovalStep::ST_APPROVED => 'تأیید',
            ApprovalStep::ST_REJECTED => 'رد',
            ApprovalStep::ST_SKIPPED  => 'عبور',
        ][$st] ?? $st;
    }

    protected function approvalBadgeClass(string $st): string
    {
        return [
            ApprovalStep::ST_PENDING  => 'text-bg-warning',
            ApprovalStep::ST_APPROVED => 'text-bg-success',
            ApprovalStep::ST_REJECTED => 'text-bg-danger',
            ApprovalStep::ST_SKIPPED  => 'text-bg-secondary',
        ][$st] ?? 'text-bg-light';
    }

    /** خروجی PDF */
    public function downloadPdf()
    {
        $pdf = Pdf::loadView('livewire.admin.attendence.requests.show-pdf', [
            'req' => $this->req,
            'fJ' => fn($dt) => $this->fJ($dt),
            'durationText' => fn($hours) => $this->durationText($hours),
            'project' => $this->req->details?->project_id ? Project::find($this->req->details->project_id) : null,
            'statusMap' => $this->statusMap,
        ]);

        return response()->streamDownload(
            fn() => print($pdf->output()),
            'request-'.$this->req->id.'.pdf'
        );
    }
}
