<?php

namespace App\Livewire\Admin\Category;

use App\Models\BlogCategory;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\Attributes\Url;
use Livewire\Attributes\On;
use Livewire\Attributes\Layout;

// تعیین قالب (Layout) مدیریتی
#[Layout('admin.master')]
class Index extends Component
{
    use WithPagination;

    // --- State Properties (با استفاده از ویژگی [Url]) ---

    #[Url(history: true)]
    public string $search = '';

    #[Url(history: true)]
    public string $sortField = 'id';

    #[Url(history: true, as: 'dir')]
    public bool $sortAsc = true;

    // --- Deletion Modal ---

    public bool $showDeleteModal = false;
    public ?int $categoryIdToDelete = null;

    // ✅ گوش دادن به رویداد ذخیره یا حذف برای تازه سازی لیست
    #[On('category-saved')]
    #[On('category-deleted')]
    public function refreshList(): void
    {
        $this->resetPage();
    }

    // --- Metods ---

    // ریست کردن صفحه‌بندی هنگام تغییر جستجو
    public function updatingSearch(string $value): void
    {
        $this->resetPage();
    }

    // تغییر حالت مرتب‌سازی
    public function sortBy(string $field): void
    {
        if ($this->sortField === $field) {
            $this->sortAsc = !$this->sortAsc;
        } else {
            $this->sortAsc = true;
        }
        $this->sortField = $field;
    }

    // --- Deletion Logic ---

    public function confirmDelete(int $categoryId): void
    {
        $this->categoryIdToDelete = $categoryId;
        $this->showDeleteModal = true;
        // ارسال ایونت برای باز کردن مودال در Blade
        $this->dispatch('open-delete-modal');
    }

    public function deleteCategory(): void
    {
        if (is_null($this->categoryIdToDelete)) {
            $this->showDeleteModal = false;
            return;
        }

        $category = BlogCategory::find($this->categoryIdToDelete); // ✅ مدل BlogCategory استفاده شد

        if (!$category) {
            session()->flash('error', '⚠️ دسته‌بندی مورد نظر یافت نشد.');
        }
        // بررسی وجود رابطه 'posts' و سپس چک کردن وجود پست‌های مرتبط
        elseif (method_exists($category, 'posts') && $category->posts()->exists()) {
            session()->flash('error', '⚠️ امکان حذف وجود ندارد. ابتدا پست‌های مرتبط با این دسته‌بندی را حذف یا منتقل کنید.');
        }
        else {
            $category->delete();
            session()->flash('success', '✅ دسته‌بندی با موفقیت حذف شد.');
        }

        $this->showDeleteModal = false;
        $this->categoryIdToDelete = null;
        // ارسال ایونت برای بسته شدن مودال در Blade
        $this->dispatch('close-delete-modal');
    }

    // --- Render Method ---

    public function render()
    {
        $categories = BlogCategory::query()
            ->withCount('posts')
            ->when($this->search, function ($query) {
                $query->where('name', 'like', '%' . $this->search . '%')
                    ->orWhere('slug', 'like', '%' . $this->search . '%');
            })
            // مرتب‌سازی بر اساس فیلد انتخاب شده
            ->orderBy($this->sortField, $this->sortAsc ? 'asc' : 'desc')
            ->paginate(10);

        return view('livewire.admin.category.index', [
            'categories' => $categories,
        ]);
    }
}
