<?php

namespace App\Livewire\Admin\Category;

use App\Models\BlogCategory; // ✅ استفاده صحیح از مدل
use Livewire\Component;
use Illuminate\Support\Str;
use Livewire\Attributes\On; // ✅ ویژگی مدرن Livewire 3
use Livewire\Attributes\Validate;

class Upsert extends Component
{
    // تعریف قانون اعتبارسنجی name با استفاده از ویژگی Validate
    #[Validate('required|string|min:3|max:255')]
    public $name = '';

    public $slug = '';

    public ?int $categoryId = null;

    // ✅ جایگزینی $listeners با ویژگی On
    #[On('edit-category')]
    public function edit(int $categoryId): void
    {
        $category = BlogCategory::findOrFail($categoryId); // ✅ استفاده از BlogCategory

        $this->categoryId = $category->id;
        $this->name = $category->name;
        $this->slug = $category->slug;

        // ریست کردن ولیدیشن برای شروع مجدد فرم
        $this->resetValidation();
    }

    // متد rules() برای اعتبارسنجی کامل
    protected function rules(): array
    {
        // ❌ اصلاح مشکل اساسی: استفاده از نام جدول 'blog_categories' به جای 'categories'
        // اگر نام جدول شما 'categories' است، آن را به همان 'categories' تغییر دهید.
        $tableName = (new BlogCategory())->getTable();

        return [
            // name از طریق #[Validate] ولید می‌شود
            'slug' => 'required|string|unique:' . $tableName . ',slug' . ($this->categoryId ? ',' . $this->categoryId : ''),
        ];
    }

    // Real-time update for slug
    public function updatedName(string $value): void
    {
        // اگر در حال ایجاد باشیم یا اسلاگ قبلا تنظیم نشده باشد
        if (is_null($this->categoryId) || empty($this->slug)) {
            $this->slug = Str::slug($value);
        }
        // اگر اسلاگ دستی تغییر داده نشود، این به تولید خودکار ادامه می‌دهد.
    }

    // Hook برای اعتبارسنجی لحظه‌ای فیلد Slug
    public function updated(string $propertyName): void
    {
        if ($propertyName === 'slug') {
            $this->validateOnly('slug');
        }
    }

    // Save or Update the category
    public function save(): void
    {
        // 1. اعتبارسنجی نهایی (شامل name از ویژگی و slug از متد rules)
        $this->validate();

        $data = [
            'name' => $this->name,
            'slug' => $this->slug,
        ];

        if ($this->categoryId) {
            // Update
            BlogCategory::find($this->categoryId)->update($data); // ✅ اصلاح به BlogCategory
            $message = '✅ دسته‌بندی با موفقیت ویرایش شد.';
        } else {
            // Create
            BlogCategory::create($data); // ✅ اصلاح به BlogCategory
            $message = '✅ دسته‌بندی با موفقیت ایجاد شد.';
        }

        // 2. ارسال ایونت برای تازه‌سازی و بسته شدن مودال
        $this->dispatch('category-saved');
        $this->dispatch('refresh-list')->to(Index::class);

        session()->flash('success', $message);
        $this->resetForm();
    }

    public function resetForm(): void
    {
        $this->reset(['categoryId', 'name', 'slug']);
        $this->resetValidation();
    }

    public function render()
    {
        return view('livewire.admin.category.upsert');
    }
}
