<?php

namespace App\Livewire\Admin\Company;

use App\Models\Company;
use App\Models\Department;
use App\Models\User;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;
use Morilog\Jalali\Jalalian;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class CompanyList extends Component
{
    use WithFileUploads;
    use WithPagination;

    protected $paginationTheme = "bootstrap";

    public $search = '';
    public $createdAtFilter = '';
    public $SortId = true;
    public $companyId;
    public $name, $tel, $fax, $address, $image; // ✨ اضافه شدن $address
    public $updateMode = false;

    // Modal جزئیات شرکت
    public $showDetailsModal = false;
    public $selectedCompany = null;
    public $departments = [];
    public $companyImage = null;
    public $companyLetterForms = []; // ✨ لیست سربرگ‌ها

    // Modal اعضای دپارتمان
    public $showDepartmentUsersModal = false;
    public $selectedDepartment = null;
    public $departmentUsers = [];

    // ------------------------------------------------------------
    // Queries & Render
    // ------------------------------------------------------------

    #[Layout('admin.master')]
    public function render()
    {
        $companies = Company::query()
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('name', 'LIKE', '%' . $this->search . '%')
                        ->orWhere('tel', 'LIKE', '%' . $this->search . '%')
                        ->orWhere('fax', 'LIKE', '%' . $this->search . '%')
                        ->orWhere('address', 'LIKE', '%' . $this->search . '%');
                });
            })
            ->when($this->createdAtFilter, function ($query) {
                $query->whereDate('created_at', '=', $this->createdAtFilter);
            })
            ->orderBy('id', $this->SortId ? 'asc' : 'desc')
            ->paginate(10);

        // فرمت دهی تاریخ در اینجا انجام می‌شود
        foreach ($companies as $company) {
            $company->created_at = $this->formatShamsiDate($company->created_at, 'Y/m/d');
        }

        return view('livewire.admin.company.company-list', compact('companies'));
    }

    public function toggleSortId()
    {
        $this->SortId = !$this->SortId;
    }

    // ------------------------------------------------------------
    // CRUD Actions
    // ------------------------------------------------------------

    public function editCompany($id)
    {
        $company = Company::findOrFail($id);
        $this->companyId = $company->id;
        $this->name = $company->name;
        $this->tel = $company->tel;
        $this->fax = $company->fax;
        $this->address = $company->address; // ✨ لود آدرس
        $this->companyImage = $company->image;
        $this->updateMode = true;
    }

    public function updateCompany()
    {
        $this->validate([
            'name' => 'required|min:3',
            'tel' => 'nullable|min:6',
            'fax' => 'nullable|min:6',
            'address' => 'nullable|string|max:500', // ✨ اعتبارسنجی آدرس
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,webp|max:2048',
        ]);

        $company = Company::findOrFail($this->companyId);
        $company->name = $this->name;
        $company->tel = $this->tel;
        $company->fax = $this->fax;
        $company->address = $this->address; // ✨ ذخیره آدرس

        if ($this->image && is_object($this->image)) {
            // حذف لوگوی قدیمی
            if ($company->image) {
                Storage::disk('company')->delete('logo/' . $company->image);
            }
            $imageName = time() . '_' . $this->image->getClientOriginalName();
            $this->image->storeAs('logo/', $imageName, 'company');
            $company->image = $imageName;
        }

        $company->save();

        $this->updateMode = false;
        $this->reset(['name', 'tel', 'fax', 'image', 'address']);
        session()->flash('message', 'شرکت با موفقیت آپدیت شد!');
        $this->dispatch('CompanyUpdated');
    }

    public function deleteCompany($id)
    {
        $company = Company::findOrFail($id);
        // حذف لوگو قبل از حذف رکورد
        if ($company->image) {
            Storage::disk('company')->delete('logo/' . $company->image);
        }
        $company->delete();
        session()->flash('message', 'شرکت با موفقیت حذف شد!');
    }

    // ------------------------------------------------------------
    // Modal Logic
    // ------------------------------------------------------------

    // نمایش جزئیات شرکت
    public function showCompanyDetails($companyId)
    {
        // ✨ لود کردن letterForms و departments
        $this->selectedCompany = Company::with(['departments.users', 'letterForms'])->findOrFail($companyId);

        $this->departments = $this->selectedCompany->departments;
        $this->companyLetterForms = $this->selectedCompany->letterForms;

        $this->showDetailsModal = true;
    }

    public function closeCompanyDetailsModal()
    {
        $this->showDetailsModal = false;
        $this->selectedCompany = null;
        $this->departments = [];
        $this->companyLetterForms = [];
    }

    // نمایش اعضای دپارتمان
    public function showDepartmentUsers($departmentId)
    {
        $this->selectedDepartment = Department::with('users')->findOrFail($departmentId);
        $this->departmentUsers = $this->selectedDepartment->users;
        $this->showDepartmentUsersModal = true;
    }

    public function closeDepartmentUsersModal()
    {
        $this->showDepartmentUsersModal = false;
        $this->selectedDepartment = null;
        $this->departmentUsers = [];
    }

    // ------------------------------------------------------------
    // Helpers
    // ------------------------------------------------------------

    public function formatShamsiDate($date, $format = 'Y/m/d H:i')
    {
        if (!$date) return '—';
        try {
            return Jalalian::fromDateTime($date)->format($format);
        } catch (\Exception $e) {
            return '—';
        }
    }

    // متد کمکی برای ساخت URL لوگو (استفاده شده در Modal ویرایش)
    public function getCompanyLogoUrl($filename)
    {
        if (!$filename) return asset('images/default_logo.png'); // Placeholder
        // ⚠️ توجه: اگر دیسک company آدرس را به /storage/logo/file می‌دهد، از storage:url استفاده کنید.
        // در غیر این صورت از asset() استفاده کنید.
        return asset('app/images/company/logo/' . $filename);
    }
}
