<?php
namespace App\Livewire\Admin\Contact;

use Livewire\Component;
use App\Models\CallReminder;
use App\Models\Contact;
use Carbon\Carbon;
use Morilog\Jalali\Jalalian;
use Livewire\Attributes\Rule;
use Livewire\Attributes\Validate;
use Livewire\Attributes\On; // استفاده از On برای Event Listener ها
use function Livewire\Volt\layout;


class CallReminderComponent extends Component
{
    // === پراپرتی‌ها و اعتبارسنجی ===
    public $contacts = [];
    public $callReminders = [];

    #[Rule('nullable|string|max:255')]
    public $newReminderMessage;

    // پراپرتی برای تاریخ شمسی که از Input دریافت می‌شود
    #[Validate('required|string', message: 'تاریخ و زمان یادآوری (شمسی) الزامی است.')]
    public $reminderDateJalali;

    // پراپرتی میلادی (برای ذخیره در دیتابیس و اعتبارسنجی منطقی)
    // این فیلد در متد hydrate پر می‌شود و در اعتبارسنجی استفاده می‌شود.
    public $reminderDate;

    #[Rule('required|exists:contacts,id', message: 'انتخاب مخاطب الزامی است.')]
    public $contactId;

    // === Hook: Mount ===
    public function mount()
    {
        // 🚀 انتخاب ستون‌های لازم برای اکسسور display_name
        $this->contacts = auth()->user()->contacts()
            ->select('id', 'name', 'phone_number')
            ->orderBy('name')
            ->get();

        $this->loadReminders();
    }

    // === Hook: Hydrate - تبدیل شمسی به میلادی قبل از اجرای متدها (بهبود یافته) ===
    public function hydrate()
    {
        // اگر تاریخ شمسی پر شده و معتبر باشد، آن را تبدیل می‌کند
        if ($this->reminderDateJalali) {
            try {
                // اطمینان از اینکه تاریخ حداقل دارای فرمت Y/m/d H:i باشد
                $jDate = Jalalian::fromFormat('Y/m/d H:i', $this->reminderDateJalali);
                $this->reminderDate = $jDate->toCarbon()->format('Y-m-d H:i:s');
            } catch (\Exception $e) {
                // اگر فرمت نامعتبر بود، null قرار می‌دهیم تا اعتبارسنجی آن را بگیرد
                $this->reminderDate = null;
            }
        } else {
            $this->reminderDate = null;
        }
    }

    // === منطق: بارگذاری یادآوری‌ها (بهبود یافته) ===
    // استفاده از @On برای رفرش شدن لیست در صورت دریافت ایونت‌های دیگر
    #[On('reminder-list-updated')]
    public function loadReminders()
    {
        // انتخاب ستون‌های لازم برای نمایش display_name در ویو
        $this->callReminders = CallReminder::with('contact:id,name,phone_number')
            ->where('user_id', auth()->id())
            ->where('reminder_time', '>', now())
            ->orderBy('reminder_time', 'asc')
            ->get();
    }

    // === منطق: ذخیره یادآوری جدید ===
    public function saveReminder()
    {
        // اعتبارسنجی نهایی با قوانین میلادی (اطمینان از آینده بودن تاریخ)
        $this->validate([
            'contactId' => 'required|exists:contacts,id',
            'newReminderMessage' => 'nullable|string|max:255',
            // اعتبارسنجی روی $this->reminderDate که میلادی و استاندارد است
            'reminderDate' => 'required|date|after_or_equal:now',
        ]);

        $contact = $this->contacts->firstWhere('id', $this->contactId);

        CallReminder::create([
            'user_id' => auth()->id(),
            'contact_id' => $this->contactId,
            // استفاده از $this->reminderDate که در hydrate تبدیل شده است
            'reminder_time' => Carbon::parse($this->reminderDate),
            'message' => $this->newReminderMessage ?: "یادآوری تماس با " . ($contact->display_name ?? $contact->name ?? 'این مخاطب'),
        ]);

        // استفاده از reset() Livewire 3
        $this->reset(['newReminderMessage', 'reminderDate', 'reminderDateJalali', 'contactId']);

        $this->loadReminders();
        $this->dispatch('toast', message: 'یادآوری با موفقیت ذخیره شد.', type: 'success');
    }

    // === منطق: انجام یادآوری (جدید) ===
    public function completeReminder($reminderId)
    {
        $reminder = CallReminder::where('user_id', auth()->id())->where('id', $reminderId)->first();
        if ($reminder) {
            // به جای حذف، می‌توان وضعیت (مانند is_completed) را در دیتابیس آپدیت کرد
            // اما برای سادگی فعلی، آن را حذف می‌کنیم.
            $reminder->delete();
            $this->loadReminders();
            $this->dispatch('toast', message: 'یادآوری انجام و حذف شد.', type: 'success');
        }
    }

    // === منطق: حذف یادآوری ===
    public function deleteReminder($reminderId)
    {
        CallReminder::where('user_id', auth()->id())->where('id', $reminderId)->delete();
        $this->loadReminders();
        $this->dispatch('toast', message: 'یادآوری با موفقیت حذف شد.', type: 'info');
    }

    // === منطق: ایجاد یادآوری هوشمند (بهبود یافته) ===
    // در این متد نیازی به دریافت contactId نیست چون آن را از پراپرتی $this->contactId می‌گیرد.
    public function createSmartReminder()
    {
        if (!$this->contactId) {
            $this->dispatch('toast', message: 'لطفاً ابتدا یک مخاطب را از لیست انتخاب کنید.', type: 'warning');
            return;
        }

        $contact = Contact::select('id', 'name', 'phone_number')->find($this->contactId);

        // ... (ادامه منطق هوشمند که قبلاً داشتید)

        // فرض بر این است که $contact->interactions() وجود دارد.
        if (!method_exists($contact, 'interactions')) {
            $this->dispatch('toast', message: 'خطا: مدل Contact متد interactions() را ندارد.', type: 'error');
            return;
        }

        $lastInteraction = $contact->interactions()->latest()->first();
        $contactName = $contact->display_name ?? $contact->name ?? 'این مخاطب';
        $interactionThresholdDays = 45;
        // زمان استاندارد برای یادآوری هوشمند (فردا ساعت ۱۰ صبح)
        $reminderTime = now()->addDay()->setHour(10)->setMinute(0);
        $message = "";
        $toastMessage = "";

        if (!$lastInteraction) {
            $message = 'یادآوری هوشمند: هیچ تعاملی با ' . $contactName . ' ثبت نشده است. لازم است تماس اولیه را برقرار کنید.';
            $reminderTime = now()->addDays(2)->setHour(9)->setMinute(0);
            $toastMessage = "یادآوری هوشمند (اولین تماس) ایجاد شد.";
        } else {
            $daysSinceInteraction = Carbon::parse($lastInteraction->interaction_date)->diffInDays(now());

            if ($daysSinceInteraction <= $interactionThresholdDays) {
                $this->dispatch('toast', message: "وضعیت سبز: فقط $daysSinceInteraction روز از آخرین تعامل گذشته است. نیازی به یادآوری هوشمند نیست.", type: 'success');
                return;
            }

            $message = "یادآوری هوشمند: بیش از $interactionThresholdDays روز ($daysSinceInteraction روز) از آخرین تعامل با " . $contactName . " گذشته است.";
            $toastMessage = "یادآوری هوشمند عدم پیگیری ایجاد شد.";
        }

        CallReminder::create([
            'user_id' => auth()->id(),
            'contact_id' => $this->contactId,
            'reminder_time' => $reminderTime,
            'message' => $message,
        ]);

        $this->loadReminders();
        $this->dispatch('toast', message: $toastMessage, type: 'info');
    }

    // === متد کمکی: فرمت تاریخ شمسی ===
    public function formatReminderTime($time)
    {
        if (!$time instanceof Carbon) {
            return '—';
        }
        try {
            return Jalalian::fromCarbon($time)->format('Y/m/d ساعت H:i');
        } catch (\Exception $e) {
            return $time->format('Y-m-d H:i');
        }
    }

    public function render()
    {
        return view('livewire.admin.contact.call-reminder-component')->layout('admin.master');
    }
}
