<?php

namespace App\Livewire\Admin\Contact;

use App\Models\Contact;
use App\Models\ContactGroup;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;
use Morilog\Jalali\Jalalian;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Throwable; // ✅ برای مدیریت خطای بهتر

class ContactBookComponent extends Component
{
    use WithPagination, WithFileUploads;

    // --- Pagination ---
    public $perPage = 24;
    // 💡 public $contacts حذف شد و در متد render تعریف می‌شود.

    // --- Filters & Search ---
    public $search = '';
    public $selectedGroup = null;
    public $dateRange = []; // ['1404/01/01', '1404/12/30']
    public $sortBy = 'name_asc'; // name_asc|name_desc|created_at_desc|created_at_asc

    // --- View Mode ---
    public $viewMode = 'grid'; // grid | list

    // --- Modals ---
    public $showAddContactModal = false;
    public $showEditContactModal = false;
    public $showDeleteContactModal = false;
    public $showAddGroupModal = false;
    public $showExportModal = false;
    public $showImportModal = false;

    // --- Bulk Actions ---
    public $selectedContacts = [];
    public $showBulkActions = false;

    // --- Contact Form (Add/Edit) ---
    public $newContactName = '';
    public $newContactPhone = '';
    public $newContactFax = '';
    public $newContactEmail = '';
    public $newContactAddress = '';
    public $newContactTags = [];
    public $newContactGroup = null;
    public $newContactPhoneType = 'mobile';
    public $newContactIsPublic = false;

    public $editContactId = null;
    public $editContactName = '';
    public $editContactPhone = '';
    public $editContactFax = '';
    public $editContactEmail = '';
    public $editContactAddress = '';
    public $editContactTags = [];
    public $editContactGroup = null;
    public $editContactPhoneType = 'mobile';
    public $editContactIsPublic = false;

    // --- Group Form ---
    public $newGroupName = '';
    public $newGroupColor = '#4361EE';

    // --- Delete ---
    public $deleteContactId = null;
    public $contactToDelete = null;
    public $deleteGroupId; // 👈 این متغیر مفقود است و باید اضافه شود

    // ✅ جدید: متغیرهای فرم ویرایش گروه
    public $editGroupId = null;
    public $editGroupName = '';
    public $editGroupColor = '';
    public $groupToDelete = null; // ✅ جدید: برای نمایش نام گروه در مدال حذف

    public $showEditGroupModal = false; // ✅ جدید: مدال ویرایش گروه
    public $showDeleteGroupModal = false; // ✅ جدید: مدال حذف گروه

    // --- File Import ---
    public $importFile;

    // --- UI State ---
    public $contactGroups = [];
    public $totalContacts = 0;

    // --- Schema Flags ---
    public $hasTagsColumn = false;
    public $hasColorColumn = false;

    // --- Query String ---
    protected $queryString = [
        'search' => ['except' => ''],
        'selectedGroup' => ['except' => null],
        'sortBy' => ['except' => 'name_asc'],
        'viewMode' => ['except' => 'grid'],
        'dateRange' => ['except' => []],
        'perPage' => ['except' => 24],
    ];

    // --- Mount ---
    public function mount()
    {
        $this->hasTagsColumn = Schema::hasColumn('contacts', 'tags');
        $this->hasColorColumn = Schema::hasColumn('contact_groups', 'color');
        $this->loadContactGroups();
    }



    public function loadContactGroups()
    {
        $userId = auth()->id();
        $cacheKey = 'contact_groups_user_' . $userId;

        // ✅ رفع مشکل: بیرون کشیدن متغیرهای Livewire از Scope اصلی
        $hasColorColumn = $this->hasColorColumn;

        // ⏳ استفاده از Cache::remember
        $this->contactGroups = Cache::remember($cacheKey, now()->addMinutes(30),
            function () use ($userId, $hasColorColumn) { // ✅ $hasColorColumn به use اضافه شد

                // 1. تعریف عبارت دسترسی به مخاطب (Closure)
                $visibleContactsScope = function ($q) use ($userId) {
                    // فرض بر وجود Scope `visibleToUser` در مدل Contact است
                    // اگر وجود ندارد، باید منطق دسترسی عمومی/خصوصی را اینجا بنویسید.
                    $q->visibleToUser($userId);
                };

                // 2. شروع کوئری ContactGroup
                $query = ContactGroup::query();

                // 3. انتخاب فیلدها
                $selects = ['id', 'name', 'created_by'];
                if ($hasColorColumn) { // ✅ استفاده از متغیر منتقل شده
                    $selects[] = 'color';
                } else {
                    $selects[] = DB::raw('NULL as color');
                }

                $query->select($selects);

                // 4. شمارش مخاطبین قابل مشاهده
                $query->withCount(['contacts' => $visibleContactsScope]);

                // 5. اعمال فیلتر دسترسی گروهی
                $query->where(function ($q) use ($userId, $visibleContactsScope) {
                    // گروه‌هایی که توسط کاربر ایجاد شده‌اند
                    $q->where('created_by', $userId)
                        // یا گروه‌هایی که حداقل یک مخاطب قابل مشاهده دارند
                        ->orWhereHas('contacts', $visibleContactsScope);
                });

                // 6. اجرای کوئری و مرتب‌سازی
                return $query->orderBy('name')->get();
            }
        );
    }

//    public function loadContactGroups()
//    {
//        $userId = auth()->id();
//        $cacheKey = 'contact_groups_user_' . $userId;
//
//        $this->contactGroups = Cache::remember($cacheKey, now()->addMinutes(30), function () use ($userId) {
//
//            // 1. ساخت کوئری اصلی و انتخاب فیلدها
//            $selects = ['id', 'name', 'created_by'];
//
//            // افزودن ستون color به صورت شرطی برای جلوگیری از خطای DB
//            if ($this->hasColorColumn) {
//                $selects[] = 'color';
//            } else {
//                $selects[] = DB::raw('NULL as color');
//            }
//
//            $query = ContactGroup::select($selects);
//
//            // 2. تعریف Closure برای فیلتر کردن مخاطبینی که برای کاربر قابل مشاهده‌اند
//            $visibleContacts = function ($q) use ($userId) {
//                // فرض: اگر یک Local Scope به نام 'visibleToUser' در مدل Contact دارید، از آن استفاده کنید
//                // $q->visibleToUser($userId);
//
//                // اگر از Scope استفاده نمی‌کنید، عبارت اصلی را حفظ می‌کنیم:
//                $q->where('is_public', true)
//                    ->orWhere('created_by', $userId);
//            };
//
//            // 3. شمارش مخاطبین قابل مشاهده
//            $query->withCount(['contacts' => $visibleContacts]);
//
//            // 4. فیلتر نمایش گروه‌ها
//            $query->where(function ($q) use ($userId, $visibleContacts) {
//                $q->where('created_by', $userId) // گروه‌های ایجاد شده توسط کاربر فعلی (همیشه نمایش)
//                ->orWhereHas('contacts', $visibleContacts); // گروه‌هایی که حداقل یک مخاطب قابل مشاهده دارند
//            });
//
//            // 5. اجرا و مرتب‌سازی
//            return $query->orderBy('name')->get();
//        });
//    }

    private function getContactsQuery()
    {
        $userId = auth()->id(); // شناسه کاربر فعلی

        $query = Contact::with('group')
            ->select('id', 'name', 'phone_number', 'email', 'fax_number', 'address', 'group_id', 'created_at', 'is_public', 'created_by'); // created_by برای فیلتر حریم خصوصی لازم است

        if ($this->hasTagsColumn) {
            $query->addSelect('tags');
        }

        // 🔒 اعمال فیلتر حریم خصوصی (Privacy Filter)
        // نمایش مخاطبینی که: (عمومی هستند) OR (توسط کاربر فعلی ایجاد شده اند)
        $query->where(function ($q) use ($userId) {
            $q->where('is_public', true)
                ->orWhere('created_by', $userId);
        });

        // 🔍 جستجو
        if ($this->search) {
            $term = '%' . $this->search . '%';
            // توجه: این کوئری جستجو فقط روی مخاطبینی اعمال می‌شود که از فیلتر حریم خصوصی عبور کرده‌اند.
            $query->where(function ($q) use ($term) {
                $q->where('name', 'like', $term)
                    ->orWhere('phone_number', 'like', $term)
                    ->orWhere('email', 'like', $term);
                if ($this->hasTagsColumn) {
                    $q->orWhere('tags', 'like', $term);
                }
            });
        }

        // 🔹 فیلتر گروه
        if ($this->selectedGroup) {
            $query->where('group_id', $this->selectedGroup);
        }

        // 🔹 فیلتر تاریخ (Jalali → Gregorian)
        if (count($this->dateRange) === 2 && $this->dateRange[0] && $this->dateRange[1]) {
            try {
                [$startJalali, $endJalali] = $this->dateRange;
                $start = Jalalian::fromFormat('Y/m/d', $startJalali)->toCarbon();
                $end = Jalalian::fromFormat('Y/m/d', $endJalali)->endOfDay()->toCarbon();
                $query->whereBetween('created_at', [$start, $end]);
            } catch (\Exception $e) {
                // نادیده گرفتن تاریخ نامعتبر
            }
        }

        // 🔹 مرتب‌سازی
        match($this->sortBy) {
            'name_asc' => $query->orderBy('name', 'asc'),
            'name_desc' => $query->orderBy('name', 'desc'),
            'created_at_desc' => $query->orderBy('created_at', 'desc'),
            'created_at_asc' => $query->orderBy('created_at', 'asc'),
            default => $query->orderBy('name', 'asc')
        };

        return $query;
    }

    // --- Modals ---
    public function addContact()
    {
        $this->resetContactForm();
        $this->showAddContactModal = true;
    }

    public function openEditModal($contactId)
    {
        $contact = Contact::findOrFail($contactId);
        $this->editContactId = $contact->id;
        $this->editContactName = $contact->name;
        $this->editContactPhone = $contact->phone_number;
        $this->editContactFax = $contact->fax_number ?? '';
        $this->editContactEmail = $contact->email ?? '';
        $this->editContactAddress = $contact->address ?? '';
        $this->editContactGroup = $contact->group_id;
        $this->editContactPhoneType = $contact->phone_type ?? 'mobile';
        $this->editContactIsPublic = $contact->is_public;

        if ($this->hasTagsColumn) {
            $this->editContactTags = is_array($contact->tags) ? $contact->tags : explode(',', $contact->tags ?? '');
        }

        $this->showEditContactModal = true;
    }

    public function openDeleteModal($contactId)
    {
        $contact = Contact::findOrFail($contactId);
        $this->deleteContactId = $contactId;
        $this->contactToDelete = $contact;
        $this->showDeleteContactModal = true;
    }

    public function openAddGroupModal()
    {
        $this->resetGroupForm();
        $this->showAddGroupModal = true;
    }

    public function openExportModal()
    {
        $this->showExportModal = true;
    }

    public function openImportModal()
    {
        $this->showImportModal = true;
    }

    // --- Save & Update ---
    public function saveContact()
    {
        $this->validate([
            'newContactName' => 'nullable|string|max:255',
            'newContactPhone' => 'required|string|max:20|unique:contacts,phone_number',
            'newContactFax' => 'nullable|string|max:20',
            'newContactEmail' => 'nullable|email|max:255|unique:contacts,email',
            'newContactAddress' => 'nullable|string|max:500',
            'newContactGroup' => 'nullable|exists:contact_groups,id',
            'newContactPhoneType' => 'in:mobile,work,home,internal,fax',
        ], $this->contactMessages());

        $data = [
            'name' => trim($this->newContactName),
            'phone_number' => $this->formatPhoneNumber($this->newContactPhone),
            'fax_number' => $this->newContactFax ? $this->newContactFax : null,
            'email' => $this->newContactEmail ?: null,
            'address' => $this->newContactAddress ?: null,
            'group_id' => $this->newContactGroup,
            'phone_type' => $this->newContactPhoneType,
            'is_public' => $this->newContactIsPublic,
            'created_by' => auth()->id(),
        ];

        if ($this->hasTagsColumn) {
            $data['tags'] = $this->newContactTags;
        }

        Contact::create($data);

        $this->closeAddContactModal();
        $this->refreshData();
        $this->dispatch('contact-saved');
        $this->toast('مخاطب با موفقیت اضافه شد.', 'success');
    }

    public function updateContact()
    {
        $this->validate([
            'editContactName' => 'nullable|string|max:255',
            'editContactPhone' => 'required|string|max:20|unique:contacts,phone_number,' . $this->editContactId,
            'editContactFax' => 'nullable|string|max:20',
            'editContactEmail' => 'nullable|email|max:255|unique:contacts,email,' . $this->editContactId,
            'editContactAddress' => 'nullable|string|max:500',
            'editContactGroup' => 'nullable|exists:contact_groups,id',
            'editContactPhoneType' => 'in:mobile,work,home,internal,fax',
        ], $this->contactMessages());

        $contact = Contact::findOrFail($this->editContactId);

        $data = [
            'name' => trim($this->editContactName),
            'phone_number' => $this->formatPhoneNumber($this->editContactPhone),
            'fax_number' => $this->editContactFax ?: null,
            'email' => $this->editContactEmail ?: null,
            'address' => $this->editContactAddress ?: null,
            'group_id' => $this->editContactGroup,
            'phone_type' => $this->editContactPhoneType,
            'is_public' => $this->editContactIsPublic,
        ];

        if ($this->hasTagsColumn) {
            $data['tags'] = $this->editContactTags;
        }

        $contact->update($data);

        $this->closeEditContactModal();
        $this->refreshData();
        $this->dispatch('contact-updated');
        $this->toast('اطلاعات مخاطب با موفقیت به‌روزرسانی شد.', 'success');
    }

    // --- Groups ---
    public function saveGroup()
    {
        $this->validate([
            'newGroupName' => 'required|string|max:100|unique:contact_groups,name',
            'newGroupColor' => $this->hasColorColumn ? 'required|regex:/^#[0-9A-Fa-f]{6}$/' : 'nullable',
        ], [
            'newGroupName.required' => 'نام گروه الزامی است.',
            'newGroupName.unique' => 'این نام گروه قبلاً ثبت شده است.',
            'newGroupColor.regex' => 'فرمت رنگ باید #RRGGBB باشد.',
        ]);

        $data = [
            'name' => trim($this->newGroupName),
            'created_by' => auth()->id(),
        ];

        if ($this->hasColorColumn) {
            $data['color'] = $this->newGroupColor;
        }

        ContactGroup::create($data);

        $this->closeAddGroupModal();
        $this->refreshData();
        $this->toast('گروه جدید با موفقیت ساخته شد.', 'success');
    }

    public function openEditGroupModal($groupId)
    {
        try {
            $group = ContactGroup::findOrFail($groupId);

            // 🔒 بررسی دسترسی: فقط ایجادکننده می‌تواند ویرایش کند
            if ($group->created_by !== auth()->id()) {
                $this->toast('شما اجازه ویرایش این گروه را ندارید.', 'warning');
                return;
            }

            $this->editGroupId = $group->id;
            $this->editGroupName = $group->name;
            // اگر ستون رنگ وجود داشته باشد، مقدار آن را لود کن
            $this->editGroupColor = $this->hasColorColumn ? ($group->color ?? '#4361EE') : null;

            $this->showEditGroupModal = true;
        } catch (Throwable $e) {
            $this->toast('گروه مورد نظر یافت نشد.', 'error');
        }
    }

    // 💡 باز کردن مدال حذف گروه
    public function openDeleteGroupModal($groupId)
    {
        try {
            $group = ContactGroup::withCount('contacts')->findOrFail($groupId);

            // 🔒 بررسی دسترسی
            if ($group->created_by !== auth()->id()) {
                $this->toast('شما اجازه حذف این گروه را ندارید.', 'warning');
                return;
            }

            $this->deleteGroupId = $groupId;
            $this->groupToDelete = $group;
            $this->showDeleteGroupModal = true;
        } catch (Throwable $e) {
            $this->toast('گروه مورد نظر یافت نشد.', 'error');
        }
    }

//    public function openAddGroupModal()
//    {
//        $this->resetGroupForm();
//        $this->showAddGroupModal = true;
//    }

    // ✅ جدید: متد به‌روزرسانی گروه
    public function updateGroup()
    {
        $validationRules = [
            'editGroupName' => 'required|string|max:100|unique:contact_groups,name,' . $this->editGroupId,
        ];
        if ($this->hasColorColumn) {
            $validationRules['editGroupColor'] = 'required|regex:/^#[0-9A-Fa-f]{6}$/';
        }

        $this->validate($validationRules, [
            'editGroupName.required' => 'نام گروه الزامی است.',
            'editGroupName.unique' => 'این نام گروه قبلاً ثبت شده است.',
            'editGroupColor.regex' => 'فرمت رنگ باید #RRGGBB باشد.',
        ]);

        try {
            $group = ContactGroup::findOrFail($this->editGroupId);

            // 🔒 بررسی دسترسی
            if ($group->created_by !== auth()->id()) {
                $this->toast('شما اجازه ویرایش این گروه را ندارید.', 'warning');
                $this->closeEditGroupModal();
                return;
            }

            $data = [
                'name' => trim($this->editGroupName),
            ];

            if ($this->hasColorColumn) {
                $data['color'] = $this->editGroupColor;
            }

            $group->update($data);

            $this->closeEditGroupModal();
            $this->refreshData();
            $this->toast('گروه با موفقیت به‌روزرسانی شد.', 'success');
        } catch (Throwable $e) {
            $this->toast('خطا در به‌روزرسانی گروه.', 'error');
        }
    }


    // ✅ جدید: متد حذف گروه
    public function deleteGroup()
    {
        if ($this->deleteGroupId) {
            try {
                $group = ContactGroup::findOrFail($this->deleteGroupId);

                // 🔒 بررسی دسترسی
                if ($group->created_by !== auth()->id()) {
                    $this->toast('شما اجازه حذف این گروه را ندارید.', 'warning');
                    $this->closeDeleteGroupModal();
                    return;
                }

                // ⚠️ نکته امنیتی: اگر گروه حذف شود، باید مخاطبین آن را به گروه پیش‌فرض منتقل یا فیلد group_id آن‌ها را Null کنید.
                // فرض می‌کنیم در مدل Contact یک foreign key با onDelete('set null') وجود دارد.

                $groupName = $group->name;
                $group->delete();

                $this->closeDeleteGroupModal();
                $this->refreshData();
                $this->toast("گروه '{$groupName}' با موفقیت حذف شد.", 'success');
            } catch (Throwable $e) {
                $this->toast('خطا در حذف گروه.', 'error');
            }
        }
    }

// ✅ جدید: ریست فرم ویرایش گروه
    private function resetEditGroupForm()
    {
        $this->reset(['editGroupId', 'editGroupName', 'editGroupColor']);
    }
    // --- Delete ---
    public function deleteContact()
    {
        if ($this->deleteContactId) {
            Contact::findOrFail($this->deleteContactId)->delete();
            $this->closeDeleteContactModal();
            $this->refreshData();
            $this->toast('مخاطب با موفقیت حذف شد.', 'success');
        }
    }

    public function bulkDelete()
    {
        if (!empty($this->selectedContacts)) {
            Contact::whereIn('id', $this->selectedContacts)->delete();
            $count = count($this->selectedContacts);
            $this->selectedContacts = [];
            $this->showBulkActions = false;
            $this->refreshData();
            $this->toast("{$count} مخاطب با موفقیت حذف شدند.", 'success');
        }
    }

    // --- Import/Export ---
    public function exportContacts($format = 'csv')
    {
        try {
            $contacts = Contact::with('group')->get();

            return match ($format) {
                'csv' => $this->exportToCsv($contacts),
                'json' => $this->exportToJson($contacts),
                'vcf' => $this->exportToVcf($contacts),
                default => throw new \Exception('فرمت نامعتبر')
            };
        } catch (\Exception $e) {
            $this->toast('خطا در خروجی: ' . $e->getMessage(), 'error');
        }
    }

    public function importContacts()
    {
        $this->validate([
            'importFile' => 'required|file|mimes:csv,json,vcf|max:2048',
        ], [
            'importFile.required' => 'فایلی انتخاب نشده است.',
            'importFile.mimes' => 'فرمت مجاز: CSV، JSON یا VCF.',
            'importFile.max' => 'حداکثر حجم ۲ مگابایت.',
        ]);

        try {
            $ext = $this->importFile->getClientOriginalExtension();
            $content = file_get_contents($this->importFile->getRealPath());

            $count = match ($ext) {
                'csv' => $this->importFromCsv($content),
                'json' => $this->importFromJson($content),
                'vcf' => $this->importFromVcf($content),
                default => throw new \Exception('فرمت ناشناخته')
            };

            $this->closeImportModal();
            $this->toast("{$count} مخاطب با موفقیت وارد شدند.", 'success');

        } catch (\Exception $e) {
            $this->toast('خطا در وارد کردن: ' . $e->getMessage(), 'error');
        }
    }

    // --- Export Helpers (بدون تغییر) ---
    private function exportToCsv($contacts)
    {
        $filename = 'contacts_' . now()->format('Y-m-d_H-i-s') . '.csv';
        $csv = "\xEF\xBB\xBF"; // UTF-8 BOM
        $headers = ['نام', 'تلفن', 'فکس', 'ایمیل', 'آدرس', 'نوع تلفن', 'عمومی', 'گروه'];
        if ($this->hasTagsColumn) $headers[] = 'برچسب‌ها';
        $headers[] = 'تاریخ ایجاد (شمسی)';
        $csv .= '"' . implode('","', $headers) . '"' . "\n";

        foreach ($contacts as $contact) {
            $row = [
                $contact->name ?? '',
                $contact->phone_number ?? '',
                $contact->fax_number ?? '',
                $contact->email ?? '',
                $contact->address ?? '',
                $contact->phone_type_label ?? '',
                $contact->is_public ? 'بله' : 'خیر',
                $contact->group?->name ?? '',
            ];
            if ($this->hasTagsColumn) {
                $tags = is_array($contact->tags) ? implode('، ', $contact->tags) : ($contact->tags ?? '');
                $row[] = $tags;
            }
            $row[] = $contact->jalali_created_at;

            $csv .= '"' . implode('","', array_map(fn($v) => str_replace('"', '""', $v), $row)) . '"' . "\n";
        }

        return response()->streamDownload(fn() => print($csv), $filename, [
            'Content-Type' => 'text/csv; charset=UTF-8',
        ]);
    }

    private function exportToJson($contacts)
    {
        $filename = 'contacts_' . now()->format('Y-m-d_H-i-s') . '.json';
        $data = $contacts->map(function ($c) {
            $d = [
                'id' => $c->id,
                'name' => $c->name,
                'phone_number' => $c->phone_number,
                'fax_number' => $c->fax_number,
                'email' => $c->email,
                'address' => $c->address,
                'phone_type' => $c->phone_type,
                'is_public' => $c->is_public,
                'group' => $c->group?->name,
                'created_at_jalali' => $c->jalali_created_at,
            ];
            if ($this->hasTagsColumn) $d['tags'] = $c->tags;
            return $d;
        });

        return response()->streamDownload(fn() => print(json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT)), $filename, [
            'Content-Type' => 'application/json; charset=UTF-8',
        ]);
    }

    private function exportToVcf($contacts)
    {
        $filename = 'contacts_' . now()->format('Y-m-d_H-i-s') . '.vcf';
        $vcf = '';
        foreach ($contacts as $contact) {
            $vcf .= "BEGIN:VCARD\r\n";
            $vcf .= "VERSION:3.0\r\n";
            $vcf .= "FN:" . ($contact->display_name) . "\r\n";
            $vcf .= "TEL;TYPE=" . ($contact->phone_type ?? 'CELL') . ":" . $contact->phone_number . "\r\n";
            if ($contact->fax_number) $vcf .= "TEL;TYPE=FAX:" . $contact->fax_number . "\r\n";
            if ($contact->email) $vcf .= "EMAIL:" . $contact->email . "\r\n";
            if ($contact->address) $vcf .= "ADR:" . $contact->address . "\r\n";
            if ($contact->group) $vcf .= "CATEGORIES:" . $contact->group->name . "\r\n";
            if ($this->hasTagsColumn && $contact->tags) {
                $tags = is_array($contact->tags) ? implode('، ', $contact->tags) : $contact->tags;
                $vcf .= "NOTE:برچسب‌ها: " . $tags . "\r\n";
            }
            $vcf .= "END:VCARD\r\n";
        }

        return response()->streamDownload(fn() => print($vcf), $filename, [
            'Content-Type' => 'text/vcard; charset=UTF-8',
        ]);
    }

    // --- Import Helpers (بدون تغییر) ---
    private function importFromCsv($content)
    {
        $lines = array_filter(array_map('trim', explode("\n", $content)));
        if (empty($lines)) return 0;

        // Remove BOM
        if (substr($lines[0], 0, 3) === "\xEF\xBB\xBF") {
            $lines[0] = substr($lines[0], 3);
        }

        $headers = str_getcsv(array_shift($lines));
        $count = 0;

        foreach ($lines as $line) {
            if (empty(trim($line))) continue;
            $row = str_getcsv($line);
            if (count($row) < 2) continue;

            $data = [
                'name' => $row[0] ?? '',
                'phone_number' => $row[1] ?? '',
                'fax_number' => $row[2] ?? null,
                'email' => $row[3] ?? null,
                'address' => $row[4] ?? null,
                'phone_type' => $row[5] ?? 'mobile',
                'is_public' => ($row[6] ?? 'خیر') === 'بله',
                'group_name' => $row[7] ?? null,
            ];

            if ($this->hasTagsColumn && isset($row[8])) {
                $tags = array_filter(array_map('trim', explode(',', $row[8])));
                $data['tags'] = $tags;
            }

            if ($this->createContactFromImport($data)) $count++;
        }

        return $count;
    }

    private function importFromJson($content)
    {
        $data = json_decode($content, true);
        if (!is_array($data)) throw new \Exception('JSON نامعتبر');

        $count = 0;
        foreach ($data as $item) {
            if (isset($item['phone_number'])) {
                if ($this->createContactFromImport($item)) $count++;
            }
        }
        return $count;
    }

    private function importFromVcf($content)
    {
        $cards = preg_split('/BEGIN:VCARD\s*\r?\n?/i', $content, -1, PREG_SPLIT_NO_EMPTY);
        $count = 0;
        foreach ($cards as $card) {
            $contact = $this->parseVcard($card);
            if (!empty($contact['phone_number']) && !empty($contact['name'])) {
                if ($this->createContactFromImport($contact)) $count++;
            }
        }
        return $count;
    }

    private function parseVcard($text)
    {
        $lines = explode("\n", str_replace("\r", "", $text));
        $data = [];
        foreach ($lines as $line) {
            $line = trim($line);
            if (str_starts_with($line, 'FN:')) $data['name'] = substr($line, 3);
            elseif (preg_match('/TEL;TYPE=([^:]+):(.+)/i', $line, $m)) {
                $type = strtolower($m[1]);
                $num = $m[2];
                if (str_contains($type, 'fax')) {
                    $data['fax_number'] = $num;
                } else {
                    $data['phone_number'] = $num;
                    $data['phone_type'] = match (true) {
                        str_contains($type, 'work') => 'work',
                        str_contains($type, 'home') => 'home',
                        str_contains($type, 'cell') => 'mobile',
                        default => 'mobile'
                    };
                }
            } elseif (str_starts_with($line, 'TEL:')) {
                $data['phone_number'] = substr($line, 4);
                $data['phone_type'] = 'mobile';
            } elseif (str_starts_with($line, 'EMAIL:')) $data['email'] = substr($line, 6);
            elseif (str_starts_with($line, 'ADR:')) $data['address'] = substr($line, 4);
            elseif (str_starts_with($line, 'CATEGORIES:')) $data['group_name'] = substr($line, 11);
            elseif (str_starts_with($line, 'NOTE:') && $this->hasTagsColumn) {
                $note = substr($line, 5);
                if (str_starts_with($note, 'برچسب‌ها: ')) $note = substr($note, 11);
                $data['tags'] = array_filter(array_map('trim', explode('،', $note)));
            }
        }
        return $data;
    }

    private function createContactFromImport($data)
    {
        try {
            if (empty($data['phone_number'])) return false;
            if (Contact::where('phone_number', $data['phone_number'])->exists()) return false;

            $groupId = null;
            if (!empty($data['group_name'])) {
                $group = ContactGroup::firstOrCreate(
                    ['name' => $data['group_name']],
                    ['color' => '#4361EE', 'created_by' => auth()->id()]
                );
                $groupId = $group->id;
            }

            $contactData = [
                'name' => $data['name'] ?? null,
                'phone_number' => $this->formatPhoneNumber($data['phone_number']),
                'fax_number' => $data['fax_number'] ?? null,
                'email' => $data['email'] ?? null,
                'address' => $data['address'] ?? null,
                'phone_type' => $data['phone_type'] ?? 'mobile',
                'is_public' => $data['is_public'] ?? false,
                'group_id' => $groupId,
                'created_by' => auth()->id(),
            ];

            if ($this->hasTagsColumn) {
                $contactData['tags'] = $data['tags'] ?? [];
            }

            Contact::create($contactData);
            return true;
        } catch (\Exception $e) {
            \Log::warning('Failed to import contact', ['data' => $data, 'error' => $e->getMessage()]);
            return false;
        }
    }

    // --- UI Helpers ---
    public function toggleContactSelection($id)
    {
        if (($key = array_search($id, $this->selectedContacts)) !== false) {
            unset($this->selectedContacts[$key]);
        } else {
            $this->selectedContacts[] = $id;
        }
        $this->selectedContacts = array_values($this->selectedContacts);
        $this->showBulkActions = !empty($this->selectedContacts);
    }

    public function selectAllContacts()
    {
        // 💡 فقط مخاطبین موجود در صفحه فعلی را انتخاب کنید.
        // متغیر $contacts در render تعریف می‌شود. از items() برای دسترسی به داده‌های صفحه فعلی استفاده کنید.
        $contacts = $this->getContactsQuery()->paginate($this->perPage);
        $this->selectedContacts = $contacts->pluck('id')->toArray();
        $this->showBulkActions = true;
    }

    public function clearSelection()
    {
        $this->selectedContacts = [];
        $this->showBulkActions = false;
    }
    // ✅ جدید: ریست فرم ویرایش گروه


    // --- Formatting & Smart Logic ---
    private function formatPhoneNumber($number)
    {
        $number = preg_replace('/[^0-9+]/', '', $number);
        if (Str::startsWith($number, '+98') && strlen($number) === 13) {
            return '0' . substr($number, 3);
        }
        if (Str::startsWith($number, '98') && strlen($number) === 12) {
            return '0' . substr($number, 2);
        }
        if (Str::startsWith($number, '9') && strlen($number) === 10) {
            return '0' . $number;
        }
        return $number;
    }

    public function updatedNewContactPhone($value)
    {
        $this->newContactPhoneType = $this->autoDetectPhoneType($value);
    }

    public function updatedEditContactPhone($value)
    {
        $this->editContactPhoneType = $this->autoDetectPhoneType($value);
    }

    private function autoDetectPhoneType($number)
    {
        $n = preg_replace('/[^0-9]/', '', $number);
        if (Str::startsWith($n, '09') && strlen($n) === 11) return 'mobile';
        if (Str::startsWith($n, '021') && strlen($n) === 11) return 'work';
        if (strlen($n) === 8 && ctype_digit($n) && !Str::startsWith($n, '0')) return 'internal';
        if (Str::startsWith($n, '0')) return 'home';
        return 'mobile';
    }

    // --- Modals Closer ---
    public function closeAddContactModal() { $this->showAddContactModal = false; $this->resetContactForm(); }
    public function closeEditContactModal() { $this->showEditContactModal = false; $this->resetEditForm(); }
    public function closeDeleteContactModal() { $this->showDeleteContactModal = false; $this->deleteContactId = null; $this->contactToDelete = null; }
    public function closeAddGroupModal() { $this->showAddGroupModal = false; $this->resetGroupForm(); }
    public function closeExportModal() { $this->showExportModal = false; }
    public function closeImportModal() { $this->showImportModal = false; $this->importFile = null; }

    // 👈 این متد باید حتماً در کامپوننت PHP شما وجود داشته باشد
    public function closeEditGroupModal() {
        $this->showEditGroupModal = false;
        $this->resetEditGroupForm(); // متد private مربوطه را صدا بزنید
    }

    // 👈 این متد هم باید حتماً وجود داشته باشد
    public function closeDeleteGroupModal() {
        $this->showDeleteGroupModal = false;
        $this->deleteGroupId = null;
        $this->groupToDelete = null;
    }

    // --- Reset Forms ---
    private function resetContactForm()
    {
        $this->reset([
            'newContactName', 'newContactPhone', 'newContactFax', 'newContactEmail',
            'newContactAddress', 'newContactTags', 'newContactGroup',
            'newContactPhoneType', 'newContactIsPublic'
        ]);
        $this->newContactPhoneType = 'mobile';
        $this->newContactIsPublic = false;
    }

    private function resetEditForm()
    {
        $this->reset([
            'editContactId', 'editContactName', 'editContactPhone', 'editContactFax',
            'editContactEmail', 'editContactAddress', 'editContactTags',
            'editContactGroup', 'editContactPhoneType', 'editContactIsPublic'
        ]);
    }

    private function resetGroupForm()
    {
        $this->reset(['newGroupName', 'newGroupColor']);
        $this->newGroupColor = '#4361EE';
    }

    // --- Refresh & Helpers ---
//    private function refreshData()
//    {
//        Cache::forget('contact_groups');
//        $this->loadContactGroups();
//        $this->resetPage();
//        // نیازی به فراخوانی loadContacts نیست، render آن را انجام می‌دهد
//    }
    private function refreshData()
    {
        $userId = auth()->id();

        // 1. تعریف کلید کش پویا (همانند متد loadContactGroups)
        $cacheKey = 'contact_groups_user_' . $userId;

        // 2. پاک کردن کش گروه‌های مخاطبین برای کاربر فعلی
        Cache::forget($cacheKey);

        // 3. بارگذاری مجدد گروه‌ها (این کار $this->contactGroups را به‌روز می‌کند)
        $this->loadContactGroups();

        // 4. بازنشانی صفحه بندی به صفحه اول پس از رفرش دیتا
        $this->resetPage();

        // 💡 نکته: نیازی به فراخوانی loadContacts نیست، زیرا متد render به طور خودکار آن را واکشی می‌کند.
    }

    private function contactMessages()
    {
        return [
            'newContactName.max' => 'نام نباید بیش از ۲۵۵ کاراکتر باشد.',
            'newContactPhone.required' => 'شماره تلفن الزامی است.',
            'newContactPhone.max' => 'شماره تلفن باید کوتاه‌تر باشد.',
            'newContactPhone.unique' => 'این شماره قبلاً ثبت شده است.',
            'newContactFax.max' => 'شماره فکس باید کوتاه‌تر باشد.',
            'newContactEmail.email' => 'فرمت ایمیل نامعتبر است.',
            'newContactEmail.unique' => 'این ایمیل قبلاً ثبت شده است.',
            'newContactAddress.max' => 'آدرس نباید بیش از ۵۰۰ کاراکتر باشد.',
        ];
    }

    private function toast($message, $type = 'info')
    {
        $this->dispatch('toast', message: $message, type: $type);
    }

    // --- Livewire Hooks ---
    public function updated($property)
    {
        if (in_array($property, ['search', 'selectedGroup', 'sortBy', 'dateRange', 'perPage'])) {
            $this->resetPage();
        }

        // اگر تغییر در selectedContacts باشد، وضعیت Bulk Actions را به‌روز کن
        if ($property === 'selectedContacts') {
            $this->showBulkActions = !empty($this->selectedContacts);
        }
    }
    public function closeEditModal() {
        $this->showEditContactModal = false;
        $this->resetEditForm();
    }

    public function render()
    {
        // ✅ منطق اصلی: کوئری را بساز و صفحه‌بندی کن.
        $query = $this->getContactsQuery();

        // 💡 آبجکت Paginator را مستقیماً ایجاد می‌کنیم.
        $contacts = $query->paginate($this->perPage);

        // totalContacts را برای آمار بروز کن
        $this->totalContacts = $contacts->total();

        // 💡 آبجکت Paginator را به View ارسال کن.
        return view('livewire.admin.contact.contact-book-component', [
            'contacts' => $contacts,
        ])
            ->layout('admin.master')
            ->title('دفترچه تماس — آدرس‌کِش');
    }
}
