<?php
// FILE: App/Livewire/Admin/Departments/ManageMembers.php

namespace App\Livewire\Admin\Departments;

use App\Models\Department;
use App\Models\User;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class ManageMembers extends Component
{
    use WithPagination;

    public $departmentId;
    public $department;
    public int $companyId;

    public $searchCurrent = '';
    public $searchAvailable = '';
    public $selectedCurrent = [];
    public $selectedAvailable = [];
    public $perPage = 15;

    protected $paginationTheme = 'bootstrap';

    protected $listeners = ['notify' => 'notify'];

    public function mount($departmentId)
    {
        $this->departmentId = (int) $departmentId;
        // 💡 لود کردن دپارتمان و مدیر برای نمایش در هدر
        $this->department   = Department::with('manager')->findOrFail($departmentId);
        $this->companyId    = (int) $this->department->company_id;
    }

    public function updatingSearchCurrent()   { $this->resetPage('current'); }
    public function updatingSearchAvailable() { $this->resetPage('available'); }

    public function removeSelected()
    {
        $toRemove = array_filter($this->selectedCurrent);
        if (empty($toRemove)) {
            $this->dispatch('notify', type: 'error', message: 'حداقل یک کاربر انتخاب کنید.');
            return;
        }

        // 🛑 FIX: مدیریت حذف مدیر دپارتمان (چک کردن مقدار به صورت رشته‌ای)
        if ($this->department->manager_id && in_array((string)$this->department->manager_id, $this->selectedCurrent)) {
            $this->dispatch('notify', type: 'error', message: 'مدیر دپارتمان نمی‌تواند حذف شود.');
            return;
        }

        User::whereIn('id', $toRemove)->update(['department_id' => null]);
        $this->selectedCurrent = [];
        $this->resetPage('current');
        $this->dispatch('notify', type: 'success', message: 'اعضا با موفقیت حذف شدند.');
        $this->dispatch('members-updated');
    }

    public function addSelected()
    {
        $toAdd = array_filter($this->selectedAvailable);
        if (empty($toAdd)) {
            $this->dispatch('notify', type: 'error', message: 'حداقل یک کاربر انتخاب کنید.');
            return;
        }

        // 💡 گارد امنیتی: فقط کاربرانِ همان شرکت اجازهٔ افزودن دارند
        $allowedIds = User::whereIn('id', $toAdd)
            ->where('company_id', $this->companyId)
            ->pluck('id')->all();

        $blocked = array_values(array_diff($toAdd, $allowedIds));

        if (!empty($allowedIds)) {
            User::whereIn('id', $allowedIds)->update(['department_id' => $this->departmentId]);
        }

        $this->selectedAvailable = [];
        $this->resetPage('available');

        if (!empty($blocked)) {
            $this->dispatch('notify', type: 'warning', message: 'بعضی کاربران به‌دلیل تفاوت شرکت اضافه نشدند.');
        } else {
            $this->dispatch('notify', type: 'success', message: 'اعضا با موفقیت اضافه شدند.');
        }

        $this->dispatch('members-updated');
    }

    public function toggleSelectAllCurrentPage()
    {
        $ids = $this->currentUsers->getCollection()->pluck('id')->map(fn($i)=>(string)$i)->all();
        $currentlySelected = $this->selectedCurrent;

        if (empty($ids)) return;

        $allCurrentSelected = collect($ids)->every(fn($id)=> in_array($id, $currentlySelected));

        $this->selectedCurrent = $allCurrentSelected
            ? array_values(array_diff($currentlySelected, $ids))
            : array_values(array_unique(array_merge($currentlySelected, $ids)));
    }

    public function toggleSelectAllAvailablePage()
    {
        $ids = $this->availableUsers->getCollection()->pluck('id')->map(fn($i)=>(string)$i)->all();
        $allSelected = collect($ids)->every(fn($id)=> in_array($id, $this->selectedAvailable));

        $this->selectedAvailable = $allSelected
            ? array_values(array_diff($this->selectedAvailable, $ids))
            : array_values(array_unique(array_merge($this->selectedAvailable, $ids)));
    }

    public function getCurrentUsersProperty()
    {
        return User::where('department_id', $this->departmentId)
            ->with(['department:id,name'])
            ->when(trim($this->searchCurrent), function($q) {
                $term = '%' . trim($this->searchCurrent) . '%';
                $q->where(function($qq) use ($term) {
                    $qq->where('name', 'like', $term)
                        ->orWhere('email', 'like', $term);
                });
            })
            ->orderBy('name')
            ->paginate($this->perPage, ['*'], 'current');
    }

    public function getAvailableUsersProperty()
    {
        return User::where('company_id', $this->companyId)
            ->where(function($q) {
                $q->whereNull('department_id')
                    ->orWhere('department_id', '!=', $this->departmentId);
            })
            ->when(trim($this->searchAvailable), function($q) {
                $term = '%' . trim($this->searchAvailable) . '%';
                $q->where(function($qq) use ($term) {
                    $qq->where('name', 'like', $term)
                        ->orWhere('email', 'like', $term);
                });
            })
            ->orderBy('name')
            ->paginate($this->perPage, ['*'], 'available');
    }

    public function closeModals()
    {
        $this->selectedCurrent = [];
        $this->selectedAvailable = [];
        // 💡 dispatch event to parent to close the modal (assuming parent component manages the show state)
        $this->dispatch('close-members-modal');
    }

    public function render()
    {
        return view('livewire.admin.departments.manage-members', [
            'currentUsers'   => $this->currentUsers,
            'availableUsers' => $this->availableUsers,
        ]);
    }
}
