<?php
// FILE: App/Livewire/Admin/Departments/Management.php (کلاس PHP)

namespace App\Livewire\Admin\Departments;

use App\Models\Company;
use App\Models\Department;
use App\Models\User;
use Illuminate\Validation\Rule;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Str;

#[Layout('admin.master')]
class Management extends Component
{
    // فرم
    public ?int $departmentId = null;
    public ?int $company_id = null;
    public ?int $parent_id = null;
    public ?int $manager_id = null;
    public ?string $name = null;
    public ?string $code = null;
    public string $status = 'active';
    public ?string $color = null;
    public int $sort_order = 0;
    public ?string $description = null;

    public bool $editMode = false;

    // فیلتر/جستجو
    public string $search = '';
    public string $filter_company_id = '';

    // مدال اعضا
    public ?int $showMembersModal = null;

    // لیست‌های وابسته
    public $parentsList = [];
    public $managersList = [];
    public $companiesList = [];

    protected $listeners = [
        'members-updated'      => '$refresh',
        'user-saved'           => '$refresh',
        'close-members-modal'  => 'closeMembersModal',
    ];

    public function messages()
    {
        return [
            'name.required'       => 'نام دپارتمان الزامی است.',
            'company_id.required' => 'انتخاب شرکت الزامی است.',
            'code.unique'         => 'این کُد در این شرکت تکراری است.',
            'color.regex'         => 'فرمت رنگ باید هگز مثل #0ea5e9 باشد.',
            'parent_id.exists'    => 'دپارتمان والد انتخاب شده معتبر نیست.',
        ];
    }

    public function rules(): array
    {
        return [
            'company_id'  => ['required','exists:companies,id'],
            'name'        => ['required','string','max:100'],
            'code'        => [
                'nullable','string','max:30',
                Rule::unique('departments','code')
                    ->where(fn($q) => $q->where('company_id', $this->company_id))
                    ->ignore($this->departmentId),
            ],
            'parent_id'   => ['nullable','exists:departments,id'],
            'manager_id'  => ['nullable','exists:users,id'],
            'status'      => ['required','in:active,inactive'],
            'color'       => ['nullable','regex:/^#([A-Fa-f0-9]{3}|[A-Fa-f0-9]{6})$/'],
            'sort_order'  => ['nullable','integer','min:0','max:65535'],
            'description' => ['nullable','string','max:500'],
        ];
    }

    public function mount()
    {
        if (Gate::denies('manage-structure')) {
            abort(403, 'شما مجوز مدیریت ساختار سازمانی را ندارید.');
        }
        $this->companiesList = Company::orderBy('name')->get();
        $this->managersList = User::orderBy('name')->get();
        $this->loadParentsAndManagers();
    }

    public function loadParentsAndManagers()
    {
        $this->resetValidation(['parent_id', 'manager_id', 'code']);

        $this->parentsList = $this->company_id
            ? Department::where('company_id', $this->company_id)
                ->where('id', '!=', $this->departmentId)
                ->whereNull('parent_id')
                ->orderBy('name')
                ->get()
            : collect();

        $this->managersList = $this->company_id
            ? User::where(fn($q) => $q->whereHas('department', fn($d) => $d->where('company_id', $this->company_id))->orWhereNull('department_id'))
                ->orderBy('name')
                ->get(['id', 'name', 'email'])
            : User::orderBy('name')->get(['id', 'name', 'email']);
    }

    public function render()
    {
        $companiesQuery = Company::query();
        if ($this->filter_company_id) {
            $companiesQuery->where('id', $this->filter_company_id);
        }
        $companies = $companiesQuery->orderBy('name')->get();

        $allDepartments = Department::with(['parent', 'manager'])
            ->withCount('users')
            ->when($this->search, function($q) {
                $term = "%{$this->search}%";
                $q->where(function($qq) use ($term) {
                    $qq->where('name', 'like', $term)
                        ->orWhere('code', 'like', $term)
                        ->orWhereHas('company', fn($c) => $c->where('name', 'like', $term));
                });
            })
            ->orderBy('sort_order')
            ->get();

        $treeByCompany = $companies->map(function ($company) use ($allDepartments) {
            $companyDepts = $allDepartments->where('company_id', $company->id);
            return [
                'company' => $company,
                'tree'    => $this->buildTree($companyDepts),
            ];
        })->filter(fn($item) => $item['tree']->isNotEmpty());


        return view('livewire.admin.departments.management', [
            'treeByCompany' => $treeByCompany,
            'companiesList' => $this->companiesList,
        ]);
    }

    private function buildTree($items, $parentId = null)
    {
        return $items->where('parent_id', $parentId)
            ->sortBy('sort_order')
            ->map(function ($item) use ($items) {
                return [
                    'id'          => $item->id,
                    'name'        => $item->name,
                    'code'        => $item->code,
                    'color'       => $item->color,
                    'status'      => $item->status,
                    'manager'     => $item->manager?->name,
                    'users_count' => $item->users_count,
                    'sort_order'  => $item->sort_order,
                    'children'    => $this->buildTree($items, $item->id)->toArray(),
                ];
            })->values();
    }

    public function updatedCompanyId()
    {
        $this->loadParentsAndManagers();
        $this->parent_id = null;
        $this->manager_id = null;
    }

    public function save()
    {
        $this->validate();

        if ($this->isInvalidParent($this->parent_id)) {
            $this->addError('parent_id', 'انتخاب والد معتبر نیست (ایجاد حلقه در ساختار).');
            return;
        }

        $payload = [
            'company_id'  => $this->company_id,
            'parent_id'   => $this->parent_id,
            'manager_id'  => $this->manager_id,
            'name'        => $this->name,
            'code'        => $this->code,
            'status'      => $this->status,
            'color'       => $this->color,
            'sort_order'  => $this->sort_order ?? 0,
            'description' => $this->description,
        ];

        if ($this->editMode) {
            \App\Models\Department::findOrFail($this->departmentId)->update($payload);
            session()->flash('success', 'دپارتمان با موفقیت بروزرسانی شد.');
        } else {
            \App\Models\Department::create($payload);
            session()->flash('success', 'دپارتمان جدید ایجاد شد.');
        }

        $this->resetForm();
        $this->dispatch('user-saved');
    }

    public function edit($id)
    {
        $dep = Department::findOrFail($id);
        $this->departmentId = $dep->id;
        $this->company_id   = $dep->company_id;
        $this->parent_id    = $dep->parent_id;
        $this->manager_id   = $dep->manager_id;
        $this->name         = $dep->name;
        $this->code         = $dep->code;
        $this->status       = $dep->status;
        $this->color        = $dep->color;
        $this->sort_order   = (int) $dep->sort_order;
        $this->description  = $dep->description;
        $this->editMode     = true;

        $this->loadParentsAndManagers();
    }

    public function delete($id)
    {
        $dep = Department::withCount(['users', 'children'])->findOrFail($id);

        if ($dep->users_count > 0) {
            $this->addError('delete', "این دپارتمان دارای {$dep->users_count} کاربر است. ابتدا کاربران را منتقل کنید.");
            return;
        }

        if ($dep->children_count > 0) {
            $this->addError('delete', "این دپارتمان {$dep->children_count} زیرمجموعه دارد. ابتدا ساختار را اصلاح کنید.");
            return;
        }

        $dep->delete();
        session()->flash('success', 'دپارتمان با موفقیت حذف شد.');
    }

    public function resetForm()
    {
        $this->reset([
            'departmentId', 'company_id', 'parent_id', 'manager_id', 'name',
            'code', 'status', 'color', 'sort_order', 'description', 'editMode', 'search', 'filter_company_id'
        ]);
        $this->resetValidation();
        $this->loadParentsAndManagers();
    }

    public function openMembersModal($departmentId) { $this->showMembersModal = $departmentId; }
    public function closeMembersModal()             { $this->showMembersModal = null; }

    private function isInvalidParent(?int $candidateId): bool
    {
        if (!$candidateId || !$this->editMode || !$this->departmentId) return false;
        if ($candidateId === $this->departmentId) return true;

        $parent = Department::find($candidateId);
        while ($parent) {
            if ($parent->id === $this->departmentId) return true;
            $parent = $parent->parent;
        }
        return false;
    }
}
