<?php

namespace App\Livewire\Admin\Letter\Classification;

use App\Models\Classification;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use Livewire\Component;
use Livewire\WithPagination;

class ClassificationList extends Component
{
    use WithPagination;

    // --- مدیریت جدول ---
    public $sortField = 'id';
    public $sortDirection = 'asc';
    public $search = "";
    public $perPage = 10;

    // --- مدیریت فرم ویرایش ---
    public $showEditModal = false;
    public $classificationId;

    // این Rule در متد updateClassification بازنویسی می‌شود
    #[Rule('required|min:2|unique:classifications,name')]
    public $name;


    #[On('Classification-Created')]
    public function ClassificationCreated()
    {
        $this->resetPage();
    }

    // --- توابع جدول ---

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
        $this->resetPage();
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingPerPage()
    {
        $this->resetPage();
    }

    // --- توابع عملیات ---

    public function showEditModal($id)
    {
        $classification = Classification::findOrFail($id);
        $this->classificationId = $classification->id;
        $this->name = $classification->name;
        $this->showEditModal = true;

        // ارسال رویداد برای باز کردن Modal
        $this->dispatch('open-modal', name: 'editClassificationModal');
    }

    public function updateClassification()
    {
        // اعمال Rule ویرایش برای نادیده گرفتن رکورد فعلی
        $this->validate([
            'name' => 'required|min:2|unique:classifications,name,' . $this->classificationId,
        ]);

        $classification = Classification::findOrFail($this->classificationId);
        $classification->update(['name' => $this->name]);

        $this->showEditModal = false;
        $this->reset(['name', 'classificationId']);
        $this->dispatch('close-modal', name: 'editClassificationModal');
        $this->dispatch('toast-message', message: 'دسته‌بندی با موفقیت به‌روزرسانی شد.', type: 'success');
    }

    public function deleteClassification($id)
    {
        try {
            Classification::destroy($id);
            $this->dispatch('toast-message', message: 'دسته‌بندی با موفقیت حذف شد.', type: 'success');
            $this->resetPage();
        } catch (\Exception $e) {
            $this->dispatch('toast-message', message: 'خطا در حذف: ' . $e->getMessage(), type: 'error');
        }
    }

    #[layout('admin.master')]
    public function render()
    {
        $classifications = Classification::query()
            ->when($this->search, function ($query) {
                $query->where('name', 'LIKE', '%' . $this->search . '%');
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        return view('livewire.admin.letter.classification.classification-list', compact('classifications'));
    }
}
