<?php

namespace App\Livewire\Admin\Letter\Create;

use App\Models\Classification;
use App\Models\Letter;
use App\Models\UserPosition;
use App\Models\Project;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\Attributes\Validate; // 💡 استفاده از Rule Attribute
use Illuminate\Validation\Rule; // 💡 استفاده از Rule برای اعتبارسنجی بهتر

class CreateLetter extends Component
{
    use WithFileUploads;

    // فیلدهای اصلی و اختیاری
    public $title;
    public $content = ''; // مقدار اولیه برای تینای‌ام‌ایس‌ای
    public $from_who;
    public $to_who; // 💡 فیلد جدید
    public $subject; // 💡 فیلد جدید
    public $project_id;
    public $attachments = []; // برای فایل‌های آپلود شده جدید

    // فیلدهای طبقه‌بندی و وضعیت
    public $classification_id;
    public $type;
    public $signature_position;
    public $priority = 'normal'; // 💡 فیلد جدید با مقدار پیش‌فرض
    public $confidential = false; // 💡 فیلد جدید با مقدار پیش‌فرض
    public $due_date; // 💡 فیلد جدید

    // رونوشت‌ها
    public $cc_name = '', $cc_position = '';
    public $ccs = [];

    // لیست‌ها
    public $classifications, $positions;

    // 💡 استفاده از Attribute Validation برای خوانایی بهتر
    protected function rules()
    {
        return [
            'title' => 'required|string|max:255',
            'content' => 'required|string|min:10', // حداقل طول برای جلوگیری از محتوای خالی
            'project_id' => 'required|exists:projects,id',

            // 💡 بهبود: تنظیم الزامی بودن فیلدهای فرستنده/گیرنده بسته به نیاز
            'from_who' => 'nullable|string|max:255', // معمولاً از طریق جایگاه سازمانی پر می‌شود
            'to_who' => 'nullable|string|max:255',
            'subject' => 'nullable|string|max:255',

            'classification_id' => 'nullable|exists:classifications,id',
            'type' => ['nullable', Rule::in(['0', '1'])],
            'signature_position' => 'nullable|string|max:255',

            // 💡 اعتبارسنجی فیلدهای جدید
            'priority' => ['required', Rule::in(['low', 'normal', 'high', 'urgent'])],
            'confidential' => 'boolean',
            'due_date' => 'nullable|date',

            // 💡 اعتبارسنجی پیوست‌ها
            'attachments.*' => 'file|mimes:pdf,doc,docx,jpg,jpeg,png,zip|max:15360', // 15MB
        ];
    }

    // 💡 متد جدید برای هماهنگی با JS در Drop Zone
    public function addFiles($files)
    {
        // در Livewire، فایل‌های آپلود شده موقت باید به صورت تکی اضافه شوند
        foreach ($files as $file) {
            $this->attachments[] = $file;
        }
        $this->attachments = array_unique($this->attachments, SORT_REGULAR);
    }

    public function mount($projectId = null)
    {
        $this->classifications = Classification::all();
        // 💡 فقط نام جایگاه‌ها برای فیلدهای رونوشت
        $this->positions = UserPosition::pluck('position')->unique()->toArray();

        // اگر از داشبورد پروژه اومده باشه، project_id رو ست کن
        if ($projectId) {
            $this->project_id = $projectId;
        }

        // 💡 تنظیم مقادیر پیش‌فرض
        $this->type = '0'; // نوع داخلی یا خارجی
        // $this->from_who = auth()->user()->full_name; // مثال: تنظیم فرستنده به صورت پیش‌فرض
    }

    public function removeAttachment($index)
    {
        // 💡 بهبود: حذف بر اساس index و فیلتر کردن
        if (isset($this->attachments[$index])) {
            // حذف موقت فایل از حافظه Livewire
            if ($this->attachments[$index] instanceof \Illuminate\Http\UploadedFile) {
                // اگر فایل موقت بود، آن را حذف کن (در صورت لزوم)
                // $this->attachments[$index]->delete(); // در Livewire 3 معمولا مدیریت می‌شود
            }
            unset($this->attachments[$index]);
            $this->attachments = array_values($this->attachments);
        }
    }

    // ... (addCc و removeCc بدون تغییر) ...

    public function saveLetter()
    {
        $this->validate();

        $letter = Letter::create([
            'code' => $this->generateUniqueCode(),
            'title' => $this->title,
            'content' => $this->content,

            // 💡 ذخیره فیلدهای جدید
            'subject' => $this->subject,
            'to_who' => $this->to_who,
            'priority' => $this->priority,
            'confidential' => $this->confidential,
            'due_date' => $this->due_date,

            // فیلدهای موجود
            'from_who' => $this->from_who,
            'from_where' => $this->from_where ?? 'شرکت/دپارتمان', // اگر از from_where استفاده می‌شود
            'classification_id' => $this->classification_id,
            'type' => $this->type,
            'signature_position' => $this->signature_position,

            'status' => Letter::STATUS_DRAFT,
            'user_id' => auth()->id(),
            'project_id' => $this->project_id,
        ]);

        // ذخیره پیوست‌ها
        foreach ($this->attachments as $file) {
            // 💡 تغییر: ذخیره با نام اصلی و اضافه کردن پیشوند امنیتی
            $originalName = $file->getClientOriginalName();
            $extension = $file->getClientOriginalExtension();
            $fileName = time() . '_' . Str::random(10) . '.' . $extension;

            $path = $file->storeAs('attachments/' . $letter->id, $fileName, 'public');

            $letter->attachments()->create([
                'file_path' => $path,
                'file_name' => $originalName,
            ]);
        }

        // ذخیره رونوشت‌ها
        foreach ($this->ccs as $cc) {
            $letter->copies()->create([
                'name' => $cc['name'],
                'position' => $cc['position'],
            ]);
        }

        // تولید PDF (در این مرحله بهتر است به صورت Asynchronous یا با متد جدا انجام شود، اما برای سادگی در اینجا نگه می‌داریم)
        $this->generatePdf($letter->id);

        session()->flash('message', 'نامه با موفقیت ثبت شد و به پیش‌نویس‌ها اضافه شد.');
        $this->resetFields();
        // 💡 یکپارچه‌سازی: ارسال ایونت برای ریست کامل تینای‌ام‌ایس‌ای
        $this->dispatch('resetTinyMCE');
    }

    // ... (generateUniqueCode و generatePdf بدون تغییر) ...

    public function resetFields()
    {
        $this->title = '';
        $this->content = '';
        $this->from_who = auth()->user()->full_name ?? ''; // پر کردن از اطلاعات کاربر فعلی
        $this->from_where = auth()->user()->position->department->name ?? '';
        $this->classification_id = null;
        $this->type = '0';
        $this->signature_position = auth()->user()->position->position ?? '';
        $this->project_id = null;
        $this->attachments = [];
        $this->ccs = [];
        $this->cc_name = '';
        $this->cc_position = '';

        $this->to_who = '';
        $this->subject = '';
        $this->priority = 'normal';
        $this->confidential = false;
        $this->due_date = null;
    }

    #[Layout('admin.master')]
    public function render()
    {
        return view('livewire.admin.letter.create.create-letter');
    }
}
