<?php

namespace App\Livewire\Admin\Letter;

use App\Models\Letter;
use App\Models\Project;
use App\Models\LetterForm;
use App\Models\User;
use App\Models\Classification;
use App\Models\UserPosition;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Collection;
use Carbon\Carbon;
use Morilog\Jalali\Jalalian;

class CreateEditLetter extends Component
{
    use WithFileUploads;

    public Project $project;
    public ?Letter $letter = null;
    public ?int $letterId = null;

    // Form Fields
    public $title = '';
    public $content = '';
    public $from_who = '';
    public $to_who = '';
    public $classification_id = null;
    public $background_id = null;
    public $priority = 'normal';
    public $confidential = false;

    public $letter_date_shamsi = '';
    public $letter_date = null; // Gregorian date for DB

    public $attachments = []; // For new uploads
    public $type = 'internal';

    // Signer/CC Fields
    public $signer_id = null;
    public $signature_position = null;
    public $signerPositions = [];
    public $ccs = [], $cc_name = '', $cc_position = '';
    public $cc_user_id = null;
    public $cc_users_suggestions = [];

    // Status Flags
    public $isEditing = false;
    public $isCreatingNewRevision = false;
    public $autoSaveEnabled = true;
    public $lastAutoSavedAt = null;

    // Const Data (for dropdowns)
    public Collection $allUsers;
    public Collection $forms;
    public Collection $classifications;
    public array $priorityOptions = [
        'low' => 'کم',
        'normal' => 'عادی',
        'high' => 'بالا',
        'urgent' => 'فوری',
    ];

    // --- قوانین اعتبارسنجی ---
    protected function rules()
    {
        return [
            'title' => 'required|string|max:255',
            'content' => 'required|string|min:10',
            'from_who' => 'required|string|max:255',
            'to_who' => 'required|string|max:255',
            'priority' => 'required|in:low,normal,high,urgent',
            'attachments.*' => 'nullable|file|max:10240|mimes:pdf,doc,docx,xls,xlsx,jpg,jpeg,png,zip,rar',
            'letter_date' => 'required|date',
            'letter_date_shamsi' => 'required|string|max:10',
            'classification_id' => 'nullable|exists:classifications,id',
            'background_id' => 'required|exists:letter_forms,id',
            'signer_id' => 'required|exists:users,id',
            'signature_position' => 'required|string|max:100',
            'ccs' => 'array',
            'cc_name' => 'nullable|string|max:100', // Only validated in addCc
        ];
    }

    // ------------------------------------------------------------
    // Lifecycle & Mount Logic (مدیریت ویرایش و بازنگری)
    // ------------------------------------------------------------

    public function mount(Project $project, Letter $letter = null)
    {
        $this->project = $project;
        // Initialize static data
        $this->allUsers = User::where('is_active', true)->select('id', 'name')->get();
        $this->forms = LetterForm::all();
        $this->classifications = \App\Models\Classification::all(); // Assuming Classification model exists

        $this->signer_id = Auth::id();

        // 1. Handle Edit/Revision
        if ($letter) {
            $this->isEditing = true;
            $isSigned = $letter->status === Letter::STATUS_APPROVED;

            if ($isSigned) {
                // Scenario 1: New Revision (Copy data but clear sensitive status/code)
                $this->isCreatingNewRevision = true;
                $this->letterId = null; // Forces creation of a new record
                $this->autoSaveEnabled = true;

                $this->title = 'نسخه بازنگری: ' . $letter->title;
                $this->dispatch('show-toast', type: 'warning', message: '💡 نامه اصلی امضا شده است. در حال ایجاد نسخه بازنگری.');
            } else {
                // Scenario 2: Simple Edit
                $this->letterId = $letter->id;
                $this->autoSaveEnabled = false;
                $this->title = $letter->title;
            }

            // Fill common data
            $this->fill([
                'title' => $this->title,
                'content' => $letter->content,
                'from_who' => $letter->from_who,
                'to_who' => $letter->to_who,
                'classification_id' => $letter->classification_id,
                'background_id' => $letter->background_id,
                'priority' => $letter->priority,
                'confidential' => $letter->confidential ?? false,
                'type' => $letter->type,
                'signer_id' => $letter->user_id,
                'signature_position' => $letter->signature_position,
            ]);

            // Set Dates
            $letterDate = $letter->letter_date ?? $letter->created_at;
            if ($letterDate) {
                $this->letter_date = $letterDate->format('Y-m-d');
                $this->letter_date_shamsi = Jalalian::fromCarbon($letterDate)->format('Y/m/d');
            }

            // Retrieve Copies
            if ($letter->relationLoaded('copies')) {
                $this->ccs = $letter->copies->map(fn($copy) => ['name' => $copy->name, 'position' => $copy->position])->toArray();
            }

        } else {
            // Handle Create New
            $now = now();
            $this->letter_date = $now->format('Y-m-d');
            $this->letter_date_shamsi = Jalalian::fromCarbon($now)->format('Y/m/d');
        }

        $this->updateSignerPosition();
    }

    // ------------------------------------------------------------
    // Date & Signer Logic
    // ------------------------------------------------------------

    public function updatedLetterDateShamsi($value)
    {
        try {
            $carbonDate = Jalalian::fromFormat('Y/m/d', $value)->toCarbon();
            $this->letter_date = $carbonDate->format('Y-m-d');
            // Validate the date input after successful conversion
            $this->validateOnly('letter_date');
        } catch (\Exception $e) {
            $this->letter_date = null; // Clear Gregorian date on invalid input
        }
    }

    public function updatedSignerId($value)
    {
        $this->updateSignerPosition();
    }

    public function updateSignerPosition()
    {
        $this->signerPositions = [];
        $this->signature_position = null;

        if ($this->signer_id) {
            // Assuming UserPosition model is imported
            $positions = UserPosition::where('user_id', $this->signer_id)->pluck('position')->toArray();
            $this->signerPositions = array_unique(array_filter($positions));

            if (count($this->signerPositions) === 1) {
                $this->signature_position = $this->signerPositions[0];
            }
        }
    }

    // ------------------------------------------------------------
    // CC/Rónevis Logic
    // ------------------------------------------------------------

    public function updatedCcName($value)
    {
        if (strlen($value) < 2) {
            $this->cc_users_suggestions = [];
            return;
        }
        // Assuming User model is imported
        $this->cc_users_suggestions = User::where('is_active', true)
            ->where('name', 'like', '%' . $value . '%')
            ->select('id', 'name')
            ->limit(5)
            ->get();
    }

    public function selectCcUser($userId, $userName)
    {
        $this->cc_name = $userName;
        $this->cc_user_id = $userId;
        $this->cc_users_suggestions = [];

        $positions = UserPosition::where('user_id', $userId)->pluck('position')->toArray();
        $uniquePositions = array_unique(array_filter($positions));

        if (count($uniquePositions) === 1) {
            $this->cc_position = $uniquePositions[0];
        } else {
            $this->cc_position = '';
        }
    }

    public function addCc()
    {
        try {
            $this->validate(['cc_name' => 'required|string|max:100', 'cc_position' => 'nullable|string|max:100']);
        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('show-toast', type: 'error', message: 'نام رونوشت نمی‌تواند خالی باشد.');
            throw $e;
        }

        $this->ccs[] = [
            'name' => $this->cc_name,
            'position' => $this->cc_position,
        ];
        $this->cc_name = '';
        $this->cc_position = '';
        $this->cc_user_id = null;
        $this->resetValidation(['cc_name', 'cc_position']);
    }

    public function removeCc($index)
    {
        unset($this->ccs[$index]);
        $this->ccs = array_values($this->ccs);
    }

    // ------------------------------------------------------------
    // Save/Update Logic
    // ------------------------------------------------------------

    public function save()
    {
        $this->validate();

        if (! $this->letter_date) {
            $this->dispatch('show-toast', type: 'error', message: 'تاریخ نامه صحیح نیست.');
            return;
        }

        $data = $this->prepareData();

        try {
            if ($this->isEditing && !$this->isCreatingNewRevision) {
                // UPDATE: ویرایش نامه موجود (پیش نویس یا در انتظار)
                $this->letter->update($data);
                $this->handleAttachments();
                $this->handleCopies($this->letter, true);
                $successMessage = '✅ نامه با موفقیت به‌روزرسانی شد.';
            } else {
                // CREATE / NEW REVISION: ایجاد نامه جدید یا بازنگری
                $code = $this->generateLetterCode();

                $letter = $this->project->letters()->create(array_merge($data, [
                    'user_id' => $this->signer_id,
                    'sender_id' => Auth::id(),
                    'status' => Letter::STATUS_DRAFT,
                    'code' => $code,
                    'unique_code' => Str::uuid(), // Assuming unique_code field exists
                ]));

                $this->handleAttachments($letter);
                $this->handleCopies($letter);
                $successMessage = $this->isCreatingNewRevision
                    ? "✅ نسخه بازنگری با کد {$code} ایجاد شد."
                    : "✅ نامه جدید با کد {$code} ثبت شد.";
            }

            $this->resetForm();
            $this->dispatch('close-modal');
            $this->dispatch('refresh-letters');
            $this->dispatch('show-toast', type: 'success', message: $successMessage);

            return $this->redirect(route('admin.user.letters'), navigate: true);

        } catch (\Exception $e) {
            Log::error("Failed to save letter: " . $e->getMessage());
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در ثبت نامه.');
        }
    }

    private function prepareData()
    {
        // Extracts data from Livewire properties to a database-ready array
        return [
            'title' => $this->title,
            'content' => $this->content,
            'from_who' => $this->from_who,
            'to_who' => $this->to_who,
            'classification_id' => $this->classification_id,
            'background_id' => $this->background_id,
            'priority' => $this->priority,
            'confidential' => $this->confidential,
            'due_date' => $this->letter_date,
            'type' => $this->type,
            'user_id' => $this->signer_id,
            'signature_position' => $this->signature_position,
        ];
    }

    private function handleCopies(Letter $letter, $isEditing = false)
    {
        if ($isEditing) {
            // Delete old copies before inserting new ones
            $letter->copies()->delete();
        }

        foreach ($this->ccs as $copy) {
            // Assuming LetterCopy model relationship
            $letter->copies()->create([
                'name' => $copy['name'],
                'position' => $copy['position'] ?? null,
            ]);
        }
    }

    private function handleAttachments($letter = null)
    {
        if (empty($this->attachments)) return;

        $targetLetter = $letter ?? $this->letter;
        if (!$targetLetter) return;

        // Logic to move files from temporary storage and create Attachment records
    }

    private function generateLetterCode()
    {
        // Generates a simple time-based code for new drafts/revisions
        $projectCode = $this->project->code ?: 'NOCODE';
        $datePart = now()->format('Ymd');
        $timePart = now()->format('His');
        return "LTR/{$projectCode}/{$datePart}/{$timePart}";
    }

    private function resetForm()
    {
        $this->reset([
            'title', 'content', 'from_who', 'to_who', 'classification_id', 'background_id',
            'priority', 'letter_date', 'letter_date_shamsi', 'attachments',
            'cc_name', 'cc_position', 'ccs', 'signature_position', 'cc_user_id', 'cc_users_suggestions'
        ]);
        $this->letter = null;
        $this->isEditing = false;
        $this->isCreatingNewRevision = false;
        $this->autoSaveEnabled = true;
        $this->signer_id = Auth::id();
        $this->updateSignerPosition();
        $now = now();
        $this->letter_date = $now->format('Y-m-d');
        $this->letter_date_shamsi = Jalalian::fromCarbon($now)->format('Y/m/d');
    }

    public function previewLetter()
    {
        // Simple validation for preview
        $this->validate(['title' => 'required', 'content' => 'required', 'from_who' => 'required', 'to_who' => 'required', 'letter_date_shamsi' => 'required|string']);

        $previewData = [
            'title' => $this->title,
            'content' => $this->content,
            'project' => $this->project,
            'background' => $this->background_id ? LetterForm::find($this->background_id) : null,
            'sender' => Auth::user(),
            'priority' => $this->priority,
            'confidential' => $this->confidential,
            'letter_date' => $this->letter_date_shamsi,
            'from_who' => $this->from_who,
            'to_who' => $this->to_who,
            'priorityLabel' => $this->priorityLabel(),
        ];
        // Dispatch event to open a modal or new tab for preview
        $this->dispatch('open-preview', data: $previewData);
    }

    public function priorityLabel()
    {
        $labels = [
            'low' => 'کم',
            'normal' => 'عادی',
            'high' => 'بالا',
            'urgent' => 'فوری'
        ];
        return $labels[$this->priority] ?? 'نامشخص';
    }


    public function render()
    {
        $pageTitle = $this->isCreatingNewRevision ? 'ایجاد بازنگری جدید' :
            ($this->isEditing ? 'ویرایش نامه' : 'ایجاد نامه جدید');

        return view('livewire.admin.letter.create-edit-letter', [
            'pageTitle' => $pageTitle,
            'classifications' => $this->classifications,
            'allUsers' => $this->allUsers,
            'forms' => $this->forms,
        ])->layout('admin.master');;
    }
}
