<?php

namespace App\Livewire\Admin\Letter;

use App\Models\Classification;
use App\Models\Letter;
use App\Models\LetterForm;
use App\Models\User;
use App\Models\Project;
use App\Models\UserPosition;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Collection;
use Livewire\Attributes\On; // 💡 استفاده از Attribute برای Listener ها
use Illuminate\Validation\ValidationException; // 💡 استفاده مستقیم برای Catch بهتر

class CreateLetter extends Component
{
    use WithFileUploads;

    public $title = '', $from_who = '', $from_where = '', $content = '';
    public $user_id, $sender_id, $classification_id, $letterFormId, $type = Letter::PRIORITY_NORMAL, $attachments = [];
    public $signature_position = null;
    public $project_id;

    // فیلدهای جدید امضاکننده
    public $signer_user_id;
    public $signerPositions = []; // سمت‌های موجود برای امضاکننده انتخاب شده (آرایه رشته‌ها)

    public Collection $users;
    public Collection $classifications;
    public Collection $projects;
    public $currentStep = 1, $totalSteps = 4;
    public $selectedForm = null;
    public $summary = [];

    // افزوده برای رونوشت‌ها
    public $ccs = [], $cc_name = '', $cc_position = '';

    // متغیر کمکی برای حفظ وضعیت لودینگ در UI
    public $isSignerLoading = false;


    public function mount($projectId = null, $templateId = null)
    {
        // لود داده‌های ثابت
        $this->users = User::select('id', 'name')->get();
        $this->classifications = Classification::select('id', 'name')->get();
        $this->projects = Project::select('id', 'name', 'code')->get();
        $this->user_id = $this->sender_id = auth()->id();

        // --- شناسایی هوشمند project_id و templateId از URL ---
        if ($projectId && is_numeric($projectId) && $projectId > 0) {
            $this->project_id = $projectId;
        }

        if ($templateId && is_numeric($templateId) && $templateId > 0) {
            $this->letterFormId = $templateId;
        }

        // --- هوشمندی: اعمال قالب پیش‌فرض ---
        if (!$this->letterFormId) {
            $defaultForm = LetterForm::latest()->first(); // 💡 گرفتن آخرین/اولین قالب موجود
            if ($defaultForm) {
                $this->letterFormId = $defaultForm->id;
            }
        }

        // لود کردن قالب انتخاب شده
        if ($this->letterFormId) {
            $this->selectedForm = LetterForm::find($this->letterFormId);
        }

        // تنظیم امضاکننده پیش‌فرض و لود سمت‌ها
        $this->signer_user_id = $this->user_id;
        $this->updateSignerPosition();
    }

    /**
     * منطق هوشمند برای لود کردن سمت‌های فعال امضاکننده انتخاب شده
     */
    public function updateSignerPosition()
    {
        $this->isSignerLoading = true;
        $this->signerPositions = [];
        $this->signature_position = null;

        if ($this->signer_user_id) {
            $positions = UserPosition::where('user_id', $this->signer_user_id)->pluck('position')->toArray();
            $this->signerPositions = array_unique($positions);

            // اگر تنها یک سمت دارد، آن را به صورت پیش‌فرض انتخاب کن
            if (count($this->signerPositions) === 1) {
                $this->signature_position = $this->signerPositions[0];
            } else {
                $this->signature_position = null;
            }
        }
        $this->isSignerLoading = false;
    }

    /**
     * هوک Livewire: بروزرسانی سمت‌ها هنگام تغییر امضاکننده
     */
    public function updatedSignerUserId($value)
    {
        $this->updateSignerPosition();
    }


    protected function getStepRules()
    {
        switch ($this->currentStep) {
            case 1:
                return [
                    'from_who' => 'required|string|max:100',
                    'from_where' => 'required|string|max:100',
                    'project_id' => 'required|exists:projects,id',
                    'letterFormId' => ['nullable', Rule::exists('letter_forms', 'id')],
                ];
            case 2:
                return [
                    'title' => 'required|string|max:255',
                    'classification_id' => ['required', Rule::exists('classifications', 'id')],
                    // 💡 استفاده از ثابت‌های مدل
                    'type' => ['required', Rule::in([Letter::PRIORITY_NORMAL, Letter::PRIORITY_URGENT])],
                    'signer_user_id' => 'required|exists:users,id',
                    'signature_position' => ['required', 'string', 'max:100'],
                ];
            case 3:
                return [
                    'content' => 'required|string|min:4',
                ];
            case 4:
                return [
                    // 💡 حجم پیوست به 15MB افزایش یافت
                    'attachments.*' => 'nullable|file|max:15360|mimes:pdf,doc,docx,xlsx,xls,jpg,jpeg,png,gif,zip,rar',
                    'ccs' => 'array',
                    'ccs.*.name' => 'required|string|max:100',
                    'ccs.*.position' => 'nullable|string|max:100',
                ];
            default:
                return [];
        }
    }

    // 💡 قوانین نهایی برای اعتبارسنجی نهایی
    protected $rules = [
        'project_id' => 'required|exists:projects,id',
        'letterFormId' => 'nullable|exists:letter_forms,id',
        'title' => 'required|string|max:255',
        'from_who' => 'required|string|max:100',
        'from_where' => 'required|string|max:100',
        'classification_id' => 'required|exists:classifications,id',
        'type' => 'required|in:' . Letter::PRIORITY_NORMAL . ',' . Letter::PRIORITY_URGENT,
        'signer_user_id' => 'required|exists:users,id',
        'signature_position' => 'required|string|max:100',
        'content' => 'required|string|min:4',
        'attachments.*' => 'nullable|file|max:15360|mimes:pdf,doc,docx,xlsx,xls,jpg,jpeg,png,gif,zip,rar',
        'ccs' => 'array',
        'ccs.*.name' => 'required|string|max:100',
        'ccs.*.position' => 'nullable|string|max:100',
        'cc_name' => 'nullable|string|max:100',
        'cc_position' => 'nullable|string|max:100',
    ];

    public function updatedLetterFormId()
    {
        $this->selectedForm = LetterForm::find($this->letterFormId);
    }

    public function updatedAttachments()
    {
        $this->validateOnly('attachments.*');
    }

    public function nextStep()
    {
        // اجرای اعتبارسنجی مرحله قبل
        try {
            $this->validate($this->getStepRules());
        } catch (ValidationException $e) { // 💡 استفاده از ValidationException
            $this->dispatch('scroll-to-top');
            $this->dispatch('show-toast', message: 'خطا در اعتبارسنجی. لطفا فیلدهای مشخص شده را تکمیل کنید.', type: 'error');
            throw $e;
        }

        if ($this->currentStep < $this->totalSteps) {
            $this->currentStep++;
            $this->dispatch('scroll-to-top');
            $this->resetValidation();
        }
    }

    public function prevStep()
    {
        if ($this->currentStep > 1) {
            $this->currentStep--;
            $this->dispatch('scroll-to-top');
            $this->resetValidation();
        }
    }

    // مدیریت رونوشت‌ها
    public function addCc()
    {
        $this->validate([
            'cc_name' => 'required|string|max:100',
            'cc_position' => 'nullable|string|max:100',
        ]);

        $this->ccs[] = [
            'name' => $this->cc_name,
            'position' => $this->cc_position,
        ];
        $this->cc_name = '';
        $this->cc_position = '';
        $this->resetValidation(['cc_name', 'cc_position']);
    }

    public function removeCc($index)
    {
        if (isset($this->ccs[$index])) {
            unset($this->ccs[$index]);
            $this->ccs = array_values($this->ccs);
        }
    }

    /**
     * ثبت نهایی نامه در دیتابیس
     */
    public function saveLetter()
    {
        // اجرای اعتبارسنجی جامع
        try {
            $this->validate();
        } catch (ValidationException $e) {
            $this->dispatch('scroll-to-top');
            $this->dispatch('show-toast', message: 'خطا در اعتبارسنجی نهایی. لطفا فیلدهای مشخص شده را تکمیل کنید.', type: 'error');
            throw $e;
        }

        try {
            // 💡 تعیین کد نامه (برای تولید کد می‌توانید یک متد جدا بنویسید)
            $letter = Letter::create([
                'project_id' => $this->project_id,
                'title' => $this->title,
                'content' => $this->content,
                'sender_id' => $this->sender_id,
                'user_id' => $this->signer_user_id,
                'background_id' => $this->letterFormId,
                'classification_id' => $this->classification_id,
                'type' => Letter::TYPE_INTERNAL, // 💡 نوع نامه (داخلی/خارجی)
                'priority' => $this->type, // 'type' در این کامپوننت به عنوان priority استفاده شده است
                'from_where' => $this->from_where,
                'from_who' => $this->from_who,
                'signature_position' => $this->signature_position,
                'status' => Letter::STATUS_DRAFT,
                'code' => 'LTR-' . now()->timestamp, // 💡 placeholder: باید با لاجیک تولید کد واقعی جایگزین شود
            ]);

            // ذخیره پیوست‌ها
            if ($this->attachments && count($this->attachments)) {
                foreach ($this->attachments as $file) {
                    // 💡 بهبود: ذخیره در دایرکتوری منحصر به فرد نامه
                    $path = $file->store('attachments/' . $letter->id);
                    $letter->attachments()->create([
                        'file_path' => $path,
                        'original_name' => $file->getClientOriginalName(),
                    ]);
                }
            }

            // ذخیره رونوشت‌ها
            foreach ($this->ccs as $copy) {
                $letter->copies()->create([
                    'name' => $copy['name'],
                    'position' => $copy['position'] ?? null,
                ]);
            }

            // پیام موفقیت هوشمند
            $successMessage = "نامه شما با موفقیت و با شماره {$letter->code} ثبت شد و آماده امضا است.";

            $this->resetFields();
            $this->resetValidation();
            $this->currentStep = 1;
            // 💡 فراخوانی ایونت برای پاکسازی ادیتور (در ویو تعریف شده است)
            $this->dispatch('clearEditor');
            $this->dispatch('show-toast', message: $successMessage, type: 'success');
            session()->flash('message', $successMessage);


        } catch (\Exception $e) {
            $this->addError('system', 'خطا در ثبت نامه: ' . $e->getMessage());
            $this->dispatch('scroll-to-top');
            Log::error('Letter Save Error: ' . $e->getMessage());
            $this->dispatch('show-toast', message: 'خطای سیستمی در ثبت نامه رخ داد. جزئیات در کنسول ثبت شده است.', type: 'error');
        }
    }

    public function removeAttachment($index)
    {
        if (isset($this->attachments[$index])) {
            unset($this->attachments[$index]);
            $this->attachments = array_values($this->attachments);
        }
        $this->resetValidation('attachments');
    }

    public function resetFields()
    {
        $this->title = '';
        $this->content = '';
        $this->from_where = '';
        $this->from_who = '';
        $this->classification_id = null;
        $this->letterFormId = null;
        $this->type = Letter::PRIORITY_NORMAL;
        $this->signature_position = null;
        $this->attachments = [];
        $this->selectedForm = null;
        $this->ccs = [];
        $this->cc_name = '';
        $this->cc_position = '';

        if (!request()->route('projectId')) {
            $this->project_id = null;
        }
        $this->signer_user_id = auth()->id();
        $this->updateSignerPosition();
        $this->resetValidation();
    }

    #[\Livewire\Attributes\Layout('admin.master')]
    public function render()
    {
        $selectedProject = $this->projects->firstWhere('id', $this->project_id);

        $this->summary = [
            'project' => $selectedProject ? $selectedProject->name : 'تعیین نشده',
            'title' => $this->title,
            'from_who' => $this->from_who,
            'from_where' => $this->from_where,
            'classification' => $this->classifications->find($this->classification_id)?->name,
            'type' => $this->type === Letter::PRIORITY_URGENT ? 'فوری' : ($this->type === Letter::PRIORITY_NORMAL ? 'عادی' : 'نامشخص'),
            'signer' => $this->users->firstWhere('id', $this->signer_user_id)?->name ?: 'تعیین نشده',
            'signature_position' => $this->signature_position,
            'attachments_count' => count($this->attachments),
            'ccs' => $this->ccs,
        ];

        $allUsers = $this->users;


        return view('livewire.admin.letter.create-letter', [
            'summary' => $this->summary,
            'selectedForm' => $this->selectedForm,
            'allUsers' => $allUsers,
        ]);
    }
}
