<?php

namespace App\Livewire\Admin\Letter\Form;

use App\Models\LetterForm;
use App\Models\Company; // ✨ مدل شرکت
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Collection;
use Illuminate\Validation\Rule; // برای Rule::requiredIf

class LetterFormCreate extends Component
{
    use WithFileUploads;

    public string $name = '';
    public $background; // Temporary uploaded file
    public $previewForm = null;
    public string $search = '';
    public $editFormId = null;
    public ?string $currentBackgroundPath = null; // مسیر فایل فعلی برای نمایش در ویرایش

    // ✨ فیلدهای کنترل دسترسی
    public bool $is_global = false;
    public array $selectedCompanies = [];

    // ✨ داده‌های ثابت برای dropdown شرکت‌ها
    public Collection $companies;

    private const TEMPLATE_DISK = 'letter_template';
    private const TEMPLATE_PATH = ''; // مسیر خالی، چون ریشه در filesystems.php تعریف شده است

    public function mount()
    {
        if (Gate::denies('manage-structure')) {
            abort(403, 'شما مجوز مدیریت کاربران را ندارید.');
        }
        $this->companies = Company::select('id', 'name')->orderBy('name')->get();
    }

    protected function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'background' => [
                // در حالت ایجاد، background اجباری است. در حالت ویرایش، اختیاری.
                Rule::requiredIf(function () {
                    return is_null($this->editFormId) && is_null($this->currentBackgroundPath);
                }),
                'nullable', 'file', 'mimes:jpeg,jpg,png,svg', 'max:2048',
            ],
            'is_global' => 'boolean',
            // اگر is_global فالس باشد، حداقل یک شرکت باید انتخاب شود.
            'selectedCompanies' => [
                Rule::requiredIf(function () {
                    return $this->is_global === false;
                }),
                'array',
            ],
            'selectedCompanies.*' => 'exists:companies,id', // هر آیتم باید یک ID معتبر شرکت باشد
        ];
    }

    public function getBackgroundUrl(?string $filename): ?string
    {
        if (!$filename) return null;
        return Storage::disk(self::TEMPLATE_DISK)->url($filename);
    }

    public function getLetterFormsProperty(): Collection
    {
        return LetterForm::when($this->search, function($q) {
            $q->where('name', 'like', '%' . $this->search . '%');
        })->orderByDesc('id')->get();
    }

    public function saveLetterForm()
    {
        // اعتبار سنجی نهایی
        $this->validate();

        $imgname = null;
        if ($this->background) {
            $path = $this->background->store(self::TEMPLATE_PATH, self::TEMPLATE_DISK);
            $imgname = basename($path);
        }

        $letterForm = LetterForm::create([
            'name' => $this->name,
            'background' => $imgname,
            'is_global' => $this->is_global,
        ]);

        // ذخیره رابطه Many-to-Many
        if (!$this->is_global && !empty($this->selectedCompanies)) {
            $letterForm->companies()->sync($this->selectedCompanies);
        }

        session()->flash('message', 'سربرگ جدید با موفقیت ذخیره شد.');
        $this->resetFields();
        $this->dispatch('close-modal', modal: 'formModal');
    }

    public function editLetterForm($id)
    {
        $this->resetFields(); // ✨ ریست کردن همه فیلدها قبل از ویرایش
        $this->editFormId = $id;
        $form = LetterForm::find($id);

        if (!$form) {
            session()->flash('error', 'فرم سربرگ یافت نشد.');
            $this->dispatch('close-modal', modal: 'formModal');
            return;
        }

        $this->name = $form->name;
        $this->background = null; // فایل آپلود شده را ریست می‌کنیم
        $this->is_global = $form->is_global;

        // لود کردن شرکت‌های مرتبط
        if ($form->is_global === false) {
            $this->selectedCompanies = $form->companies()->pluck('company_id')->toArray();
        } else {
            $this->selectedCompanies = [];
        }

        $this->currentBackgroundPath = $this->getBackgroundUrl($form->background);

        $this->dispatch('open-modal', modal: 'formModal');
    }

    public function updateLetterForm()
    {
        // اعتبار سنجی نهایی
        $this->validate();

        $form = LetterForm::find($this->editFormId);

        if (!$form) {
            session()->flash('error', 'فرم سربرگ برای ویرایش یافت نشد.');
            $this->dispatch('close-modal', modal: 'formModal');
            return;
        }

        $imgname = $form->background; // نگه داشتن نام فایل قدیمی

        if ($this->background) { // اگر فایل جدیدی آپلود شد
            if ($imgname) { // اگر فایل قدیمی وجود داشت، آن را حذف کن
                Storage::disk(self::TEMPLATE_DISK)->delete($imgname);
            }
            $path = $this->background->store(self::TEMPLATE_PATH, self::TEMPLATE_DISK);
            $imgname = basename($path);
        } elseif (is_null($this->background) && is_null($this->currentBackgroundPath)) {
            // اگر فایل جدیدی آپلود نشد و مسیر فایل فعلی هم خالی شد (یعنی کاربر حذفش کرده)
            if ($imgname) {
                Storage::disk(self::TEMPLATE_DISK)->delete($imgname);
            }
            $imgname = null;
        }

        $form->update([
            'name' => $this->name,
            'background' => $imgname,
            'is_global' => $this->is_global,
        ]);

        // به‌روزرسانی رابطه Many-to-Many
        if (!$this->is_global) {
            $form->companies()->sync($this->selectedCompanies);
        } else {
            $form->companies()->sync([]); // اگر عمومی شد، روابط قبلی را پاک کن
        }

        session()->flash('message', 'فرم با موفقیت ویرایش شد.');
        $this->resetFields();
        $this->dispatch('close-modal', modal: 'formModal');
    }

    public function resetBackground()
    {
        $this->background = null;
        $this->currentBackgroundPath = null;
    }

    public function previewLetterForm($id)
    {
        $this->previewForm = LetterForm::find($id);
        $this->dispatch('open-modal', modal: 'previewModal');
    }

    public function deleteLetterForm($id)
    {
        $letterForm = LetterForm::find($id);
        if ($letterForm) {
            if ($letterForm->background) {
                Storage::disk(self::TEMPLATE_DISK)->delete($letterForm->background);
            }
            $letterForm->delete();
            session()->flash('message', 'فرم سربرگ با موفقیت حذف شد.');
        } else {
            session()->flash('error', 'فرم سربرگ برای حذف یافت نشد.');
        }
        $this->resetFields();
    }

    public function resetFields()
    {
        $this->reset(['name', 'background', 'editFormId', 'previewForm', 'currentBackgroundPath', 'is_global', 'selectedCompanies']);
        $this->resetErrorBag(); // پاک کردن خطاهای اعتبارسنجی
        $this->resetValidation(); // ✨ این هم برای پاک کردن تمام خطاهای اعتبارسنجی است
    }

    #[Layout('admin.master')]
    public function render()
    {
        return view('livewire.admin.letter.form.letter-form-create', [
            'letterForms' => $this->getLetterFormsProperty(),
        ]);
    }
}
