<?php

namespace App\Livewire\Admin\Letter;

use App\Models\Letter;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Title;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Livewire\Attributes\Computed; // استفاده از Computed Property

#[Title('لیست نامه‌های وارده')]
#[Layout('admin.master')]
class IncomingLettersList extends Component
{
    use WithPagination;

    // از bootstrap به tailwind تغییر داده شد تا با طراحی مدرن‌تر Blade سازگارتر باشد
    // اگر بوت‌استرپ لازم است، آن را به 'bootstrap' برگردانید.
    protected $paginationTheme = 'tailwind';

    public string $search = '';
    public string $searchCode = '';
    public string $filterStatus = 'all';
    public int $perPage = 10;

    // مدیریت مودال‌ها
    public bool $showPreviewModal = false;
    public ?Letter $previewLetter = null;
    public bool $showAttachmentModal = false;
    public array $attachmentFiles = [];
    public int $currentAttachmentIndex = 0;
    public string $attachmentPreviewText = '';

    // Count Properties (نیازی به تعریف به صورت عمومی نیست اگر در Computed استفاده شوند،
    // اما برای دسترسی مستقیم در Blade حفظ می‌کنیم)
    public int $totalCount = 0;
    public int $pendingCount = 0;
    public int $approvedCount = 0;
    public int $rejectedCount = 0;

    // Status Constants (برای استفاده بهتر در PHP و Blade، بهتر است در مدل تعریف شوند)
    const STATUS_MAP = [
        'pending' => ['label' => 'در انتظار اقدام', 'color' => 'warning'],
        'approved' => ['label' => 'بایگانی شده', 'color' => 'success'],
        'rejected' => ['label' => 'رد شده', 'color' => 'danger'],
    ];

    // به جای mount، از این متد استفاده می‌کنیم تا هر بار که صفات تغییر می‌کنند آمار هم به‌روز شود
    public function boot()
    {
        $this->loadCounts();
    }

    // متد بهینه شده برای بارگذاری آمار
    public function loadCounts(): void
    {
        $base = Letter::where('type', Letter::TYPE_INCOMING);
        $this->totalCount = (clone $base)->count();
        $this->pendingCount = (clone $base)->where('status', 'pending')->count();
        $this->approvedCount = (clone $base)->where('status', 'approved')->count();
        $this->rejectedCount = (clone $base)->where('status', 'rejected')->count();
    }

    public function updatingSearch() { $this->resetPage(); }
    public function updatingSearchCode() { $this->resetPage(); }
    public function updatingFilterStatus() { $this->resetPage(); }

    public function showPreview(int $letterId): void
    {
        // استفاده از TYPE_INCOMING
        $this->previewLetter = Letter::where('type', Letter::TYPE_INCOMING)
            ->with(['classification', 'project', 'attachments'])
            ->findOrFail($letterId);
        $this->showPreviewModal = true;
    }

    public function closePreview(): void
    {
        $this->reset('showPreviewModal', 'previewLetter');
    }

    public function showAttachmentPreview(int $letterId, ?int $index = 0): void
    {
        $letter = Letter::with('attachments')->findOrFail($letterId);

        // اطمینان از وجود فایل‌ها و دسترسی (باید متد formatBytes در کلاس موجود باشد)
        $attachments = $letter->attachments->map(function ($att) {
            $filePath = storage_path('app/public/' . $att->file_path);

            return [
                'path' => $att->file_path,
                'name' => $att->original_name,
                'url' => Storage::url($att->file_path),
                'mime' => $att->mime_type ?? (file_exists($filePath) ? Storage::mimeType($att->file_path) : 'application/octet-stream'),
                'size' => file_exists($filePath) ? $this->formatBytes(filesize($filePath)) : 'نامشخص',
                'created_at' => $att->created_at ? jdate($att->created_at)->format('Y/m/d H:i') : '—',
            ];
        })->filter()->values(); // فیلتر کردن پیوست‌های نامعتبر

        if ($attachments->isEmpty()) return;

        $this->attachmentFiles = $attachments->toArray();
        $this->currentAttachmentIndex = $index ?? 0;
        $this->loadPreviewText();
        $this->showAttachmentModal = true;
    }

    private function formatBytes($size, $precision = 2): string
    {
        if ($size <= 0) return '0 B';
        $units = ['B', 'KB', 'MB', 'GB'];
        $i = floor(log($size, 1024));
        return round($size / (1024 ** $i), $precision) . ' ' . $units[$i];
    }

    public function loadPreviewText(): void
    {
        $this->attachmentPreviewText = '';
        $current = $this->attachmentFiles[$this->currentAttachmentIndex] ?? null;
        if (!$current) return;

        $mime = $current['mime'];

        if (str_starts_with($mime, 'text/')) {
            // محتوای فایل متنی
            try {
                $this->attachmentPreviewText = Str::limit(Storage::get($current['path']), 5000);
            } catch (\Exception $e) {
                $this->attachmentPreviewText = "خطا در بارگذاری محتوای متنی.";
            }
        } elseif (in_array($mime, [
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document', // docx
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', // xlsx
            'application/vnd.openxmlformats-officedocument.presentationml.presentation', // pptx
            'application/msword', // doc
            'application/vnd.ms-excel', // xls
        ])) {
            // فایل‌های آفیس
            $ext = pathinfo($current['name'], PATHINFO_EXTENSION);
            $type = match(strtolower($ext)) {
                'docx', 'doc' => 'سند ورد',
                'xlsx', 'xls' => 'صفحه گسترده',
                'pptx', 'ppt' => 'ارائه پاورپوینت',
                default => 'فایل Office'
            };
            $this->attachmentPreviewText = "⚠️ این یک فایل $type است. برای مشاهدهٔ کامل محتوا یا پیش‌نمایش رسمی، آن را دانلود کنید یا در تب جدید باز کنید.";
        } else {
            // انواع دیگر فایل (فقط دانلود)
            $this->attachmentPreviewText = ''; // اطمینان از خالی بودن برای انواع دیگر
        }
    }

    public function nextAttachment(): void
    {
        if ($this->currentAttachmentIndex < count($this->attachmentFiles) - 1) {
            $this->currentAttachmentIndex++;
            $this->loadPreviewText();
        }
    }

    public function prevAttachment(): void
    {
        if ($this->currentAttachmentIndex > 0) {
            $this->currentAttachmentIndex--;
            $this->loadPreviewText();
        }
    }

    public function setCurrentAttachmentIndex(int $index): void
    {
        if (isset($this->attachmentFiles[$index])) {
            $this->currentAttachmentIndex = $index;
            $this->loadPreviewText();
        }
    }

    public function closeAttachmentPreview(): void
    {
        $this->reset(['showAttachmentModal', 'attachmentFiles', 'currentAttachmentIndex', 'attachmentPreviewText']);
    }

    public function resetFilters(): void
    {
        $this->reset(['search', 'searchCode', 'filterStatus', 'perPage']);
        $this->perPage = 10; // اطمینان از مقداردهی مجدد perPage
        $this->resetPage();
    }

    protected function baseBuilder()
    {
        // بهتر است از ثابت‌های تعریف شده در مدل استفاده شود.
        return Letter::where('type', Letter::TYPE_INCOMING)
            ->with(['classification', 'project', 'attachments']) // attachments برای چک کردن وجود پیوست
            ->when($this->search, fn($q) => $q->where(function ($query) {
                $query->where('title', 'like', "%{$this->search}%")
                    ->orWhere('external_sender', 'like', "%{$this->search}%");
            }))
            ->when($this->searchCode, fn($q) => $q->where(function ($query) {
                $query->where('code', 'like', "%{$this->searchCode}%")
                    ->orWhere('external_number', 'like', "%{$this->searchCode}%");
            }))
            ->when($this->filterStatus !== 'all', fn($q) => $q->where('status', $this->filterStatus))
            ->orderBy('created_at', 'desc');
    }

    #[Computed]
    public function letters()
    {
        return $this->baseBuilder()->paginate($this->perPage);
    }

    #[Computed]
    public function statuses()
    {
        // برای استفاده در select filter
        return self::STATUS_MAP;
    }

    public function render()
    {
        // loadCounts به boot منتقل شد.
        return view('livewire.admin.letter.incoming-letters-list');
    }
}
