<?php

namespace App\Livewire\Admin\Letter;

use App\Models\Letter;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;

#[Layout('admin.master')]
class LetterEdit extends Component
{
    use WithFileUploads;

    public Letter $letter;
    public $title;
    public $content;
    public $from_who;
    public $from_where;
    public $classification_id;
    public $type;
    public $signature_position;
    public $attachments = [];
    public $newAttachments = [];

    public function mount(Letter $letter)
    {
        // بررسی مالکیت نامه
        if ($letter->user_id != auth()->id()) {
            abort(403, 'شما اجازه ویرایش این نامه را ندارید.');
        }

        $this->letter = $letter;
        $this->title = $letter->title;
        $this->content = $letter->content;
        $this->from_who = $letter->from_who;
        $this->from_where = $letter->from_where;
        $this->classification_id = $letter->classification_id;
        $this->type = $letter->type;
        $this->signature_position = $letter->signature_position;
    }

    public function updateLetter()
    {
        $this->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string|min:10',
            'from_who' => 'required|string|max:100',
            'from_where' => 'required|string|max:100',
            'classification_id' => 'required|exists:classifications,id',
            'type' => 'required|in:0,1',
            'signature_position' => 'nullable|string|max:100',
            'newAttachments.*' => 'nullable|file|max:10240|mimes:pdf,doc,docx,xlsx,xls,jpg,jpeg,png,gif,zip,rar',
        ]);

        try {
            $this->letter->update([
                'title' => $this->title,
                'content' => $this->content,
                'from_who' => $this->from_who,
                'from_where' => $this->from_where,
                'classification_id' => $this->classification_id,
                'type' => $this->type,
                'signature_position' => $this->signature_position,
            ]);

            // ذخیره پیوست‌های جدید
            if ($this->newAttachments) {
                foreach ($this->newAttachments as $file) {
                    $path = $file->store('attachments', 'public');
                    $this->letter->attachments()->create([
                        'file_path' => $path,
                        'original_name' => $file->getClientOriginalName(),
                        'mime_type' => $file->getMimeType(),
                        'size' => $file->getSize(),
                    ]);
                }
            }

            $this->dispatch('show-toast', type: 'success', message: '✅ نامه با موفقیت به‌روزرسانی شد.');
            session()->flash('success', 'نامه با موفقیت به‌روزرسانی شد.');

        } catch (\Exception $e) {
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در به‌روزرسانی نامه: ' . $e->getMessage());
            \Log::error('Error updating letter: ' . $e->getMessage());
        }
    }

    public function removeAttachment($attachmentId)
    {
        try {
            $attachment = $this->letter->attachments()->find($attachmentId);
            if ($attachment) {
                \Storage::disk('public')->delete($attachment->file_path);
                $attachment->delete();
                $this->dispatch('show-toast', type: 'success', message: '✅ پیوست با موفقیت حذف شد.');
            }
        } catch (\Exception $e) {
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در حذف پیوست: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.admin.letter.letter-edit', [
            'classifications' => \App\Models\Classification::select('id', 'name')->get(),
        ]);
    }
}
