<?php

namespace App\Livewire\Admin\Letter;

use App\Models\Letter;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Url;
use Livewire\Component;
use Illuminate\Database\Eloquent\Builder;
use Carbon\Carbon;
use Illuminate\Support\Collection;

// اگر از پکیج Morilog/Jalali استفاده می‌کنید، ممکن است نیاز باشد آن را در اینجا Import کنید (اختیاری)
// use Morilog\Jalali\Jalalian;

#[Layout('admin.master', [
    'title' => 'گزارش نامه‌ها',
    'heading' => 'گزارش و تحلیل نامه‌ها',
])]
class LetterReport extends Component
{
    // --- State Properties ---
    #[Url]
    public $fromDate;

    #[Url]
    public $toDate;

    #[Url]
    public $status = '';

    #[Url]
    public $search = '';

    #[Url]
    public $reportType = 'all'; // all, sent, received, archived, signed

    public array $reportData = [];
    public Collection $details;

    // --- Constant Data ---
    public array $statuses = [
        Letter::STATUS_PENDING => 'در انتظار',
        Letter::STATUS_APPROVED => 'تأیید شده',
        Letter::STATUS_REJECTED => 'رد شده',
    ];

    // ** اصلاح: استفاده از نام ستون‌های واقعی مدل Letter **
    private const SENDER_ID_COLUMN = 'sender_id'; // فرستنده داخلی نامه
    private const RECIPIENT_ID_COLUMN = 'recipient_user_id'; // گیرنده اولیه نامه وارده

    public function mount()
    {
        // تنظیم تاریخ‌های پیش‌فرض
        if (!$this->toDate) {
            $this->toDate = Carbon::today()->toDateString();
        }
        if (!$this->fromDate) {
            $this->fromDate = Carbon::today()->subDays(6)->toDateString();
        }

        $this->details = collect([]);
        $this->generateReport();
    }

    // --- Computed Properties for Chart and Trends ---

    #[Computed]
    public function chartData(): array
    {
        $endDate = $this->toDate ? Carbon::parse($this->toDate) : Carbon::today();
        $startDate = $this->fromDate ? Carbon::parse($this->fromDate) : $endDate->copy()->subDays(6);

        $dateRange = collect();
        $currentDate = $startDate->copy();
        while ($currentDate->lte($endDate)) {
            $dateRange->push($currentDate->toDateString());
            $currentDate->addDay();
        }

        // اصلاح منطق محاسبه آمار برای سازگاری با مدل Letter
        $stats = Letter::query()
            ->whereDate('created_at', '>=', $startDate->toDateString())
            ->whereDate('created_at', '<=', $endDate->toDateString())
            ->selectRaw('DATE(created_at) as date,
                        COUNT(*) as total_count,
                        -- نامه‌های ارسالی (به طور کلی نامه‌های داخلی یا خارجی که مبدأ از سازمان است)
                        COUNT(CASE WHEN type = ? THEN 1 END) as sent_count,
                        -- نامه‌های وارده (به طور کلی نامه‌هایی که وارد سازمان می‌شوند)
                        COUNT(CASE WHEN type = ? THEN 1 END) as received_count',
                [Letter::TYPE_EXTERNAL, Letter::TYPE_INCOMING])
            ->groupBy('date')
            ->orderBy('date')
            ->get()
            ->keyBy('date');

        $labels = [];
        $total = [];
        $sent = [];
        $received = [];

        foreach ($dateRange as $date) {
            // اگر jdate در دسترس نیست، باید از Morilog\Jalali\Jalalian استفاده شود.
            // فرض بر دسترسی سراسری:
            $jDate = function_exists('jdate') ? jdate($date)->format('Y/m/d') : $date;

            $labels[] = $jDate;
            $total[] = $stats->get($date)['total_count'] ?? 0;
            $sent[] = $stats->get($date)['sent_count'] ?? 0;
            $received[] = $stats->get($date)['received_count'] ?? 0;
        }

        return [
            'labels' => $labels,
            'total' => $total,
            'sent' => $sent,
            'received' => $received,
        ];
    }

    #[Computed]
    public function letterTrends(): array
    {
        $today = Carbon::today()->toDateString();
        $yesterday = Carbon::yesterday()->toDateString();

        // 1. آمار کلی امروز
        $todayStats = Letter::query()
            ->whereDate('created_at', $today)
            ->selectRaw("COUNT(CASE WHEN type = ? THEN 1 END) as total_sent_today", [Letter::TYPE_EXTERNAL])
            ->selectRaw("COUNT(CASE WHEN type = ? THEN 1 END) as total_received_today", [Letter::TYPE_INCOMING])
            ->first();

        // 2. آمار کلی دیروز
        $yesterdayStats = Letter::query()
            ->whereDate('created_at', $yesterday)
            ->selectRaw("COUNT(CASE WHEN type = ? THEN 1 END) as yesterday_sent", [Letter::TYPE_EXTERNAL])
            ->selectRaw("COUNT(CASE WHEN type = ? THEN 1 END) as yesterday_received", [Letter::TYPE_INCOMING])
            ->first();

        // آمار کل
        $totalSent = Letter::where('type', Letter::TYPE_EXTERNAL)->count();
        $totalReceived = Letter::where('type', Letter::TYPE_INCOMING)->count();

        // محاسبه Trend
        $todaySent = $todayStats->total_sent_today ?? 0;
        $yesterdaySent = $yesterdayStats->yesterday_sent ?? 0;
        // جلوگیری از تقسیم بر صفر
        $sentTrend = $yesterdaySent > 0 ? round((($todaySent - $yesterdaySent) / $yesterdaySent) * 100, 1) : 0;

        $todayReceived = $todayStats->total_received_today ?? 0;
        $yesterdayReceived = $yesterdayStats->yesterday_received ?? 0;
        $receivedTrend = $yesterdayReceived > 0 ? round((($todayReceived - $yesterdayReceived) / $yesterdayReceived) * 100, 1) : 0;

        // نامه‌های در انتظار بیش از ۷ روز (SLA)
        $expiredCount = Letter::where('status', Letter::STATUS_PENDING)
            ->whereDate('created_at', '<=', now()->subDays(7)->toDateString())->count();

        return [
            'totalSent' => $totalSent,
            'sentTrend' => $sentTrend,
            'totalReceived' => $totalReceived,
            'receivedTrend' => $receivedTrend,
            'expiredCount' => $expiredCount,
        ];
    }

    // --- Main Method ---

    public function generateReport(): void
    {
        $query = Letter::query();

        // --- اعمال فیلترهای تاریخ، وضعیت و جستجو ---
        if ($this->fromDate) { $query->whereDate('created_at', '>=', $this->fromDate); }
        if ($this->toDate) { $query->whereDate('created_at', '<=', $this->toDate); }
        // استفاده از ثوابت مدل Letter
        if ($this->status) { $query->where('status', $this->status); }
        if ($this->search) {
            $query->where(function(Builder $q){
                $q->where('title', 'like', "%{$this->search}%")
                    ->orWhere('code', 'like', "%{$this->search}%")
                    // جستجو بر اساس فرستنده (استفاده از ستون متنی نام فرستنده)
                    ->orWhere('from_who', 'like', "%{$this->search}%")
                    ->orWhere('from_where', 'like', "%{$this->search}%");
            });
        }

        // --- اعمال فیلتر نوع گزارش (اصلاح بر اساس فیلدهای مدل) ---
        $title = 'همه نامه‌ها';
        switch ($this->reportType) {
            case 'sent':
                // نامه‌های ارسالی: شامل نامه‌های داخلی و خارجی صادر شده
                $query->whereIn('type', [Letter::TYPE_INTERNAL, Letter::TYPE_EXTERNAL]);
                $title = 'نامه‌های ارسالی';
                break;
            case 'received':
                // نامه‌های دریافتی: نامه‌های وارده به سازمان
                $query->where('type', Letter::TYPE_INCOMING);
                $title = 'نامه‌های دریافتی';
                break;
            case 'archived':
                // نامه‌های بایگانی شده
                $query->where('status', Letter::STATUS_ARCHIVED);
                $title = 'نامه‌های بایگانی‌شده';
                break;
            case 'signed':
                // اصلاح: بررسی ستون signed_at
                $query->whereNotNull('signed_at');
                $title = 'نامه‌های امضا شده';
                break;
        }

        // --- استخراج داده‌ها ---
        $this->reportData = [
            'count' => $query->count(),
            'title' => $title
        ];

        // استخراج جزئیات برای جدول گزارش
        $this->details = $query->latest()->take(20)->get([
            'id','title','code',
            'from_who', // فرستنده
            'from_where', // مبدا
            'external_sender', // گیرنده خارجی (اختیاری)
            'created_at','status'
        ]);

        $this->dispatch('report-updated');
    }

    public function resetFilters(): void
    {
        $this->reset(['fromDate', 'toDate', 'status', 'search', 'reportType']);
        // پس از ریست، باید مقادیر پیش‌فرض mount دوباره تنظیم و گزارش تولید شود
        $this->mount();
    }

    public function render()
    {
        return view('livewire.admin.letter.letter-report');
    }
}
