<?php

namespace App\Livewire\Admin\Letter;

use App\Models\Letter;
use App\Models\Referral;
use App\Models\ReferralComment;
use App\Models\User; // اضافه شد
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Morilog\Jalali\Jalalian;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Mpdf\Config\ConfigVariables;
use Mpdf\Config\FontVariables;
use Symfony\Component\HttpFoundation\StreamedResponse as BaseStreamedResponse;
use Mpdf\Mpdf;
use Illuminate\Support\Facades\Response;
use App\Models\LetterForm;
use Illuminate\Support\Facades\DB; // اضافه شد
use Illuminate\Support\Facades\Storage; // اضافه شد

use Illuminate\Support\Facades\File; // اضافه شد

// ✨ DTO برای داده‌های ارجاع فعال (برای جلوگیری از خطای Undefined property)
class ActiveReferralData
{
    public function __construct(
        public int $userId,
        public string $name,
        public string $status,
        public string $type,
        public string $photoUrl,
        public string $statusLabel,
        public string $typeLabel,
        public ?string $internalNumber,
    ) {}
}

#[Layout('admin.master')]
class LetterShow extends Component
{
    public Letter $letter;
    public ?Referral $referral = null;

    // Action Modal
    public bool $showActionModal = false;
    public string $actionType = '';
    public string $opinion = '';

    // New Handoff/Re-referral Fields
    public bool $showHandoffModal = false;
    public ?int $handoffUserId = null;
    public string $handoffPurpose = 'اقدام';
    public array $purposesOptions = ['امضا', 'ثبت صادر', 'اقدام', 'نظر'];

    // Comments
    public string $newComment = '';
    public ?int $replyParentId = null;

    // Tree/History Modal
    public bool $showReferralTreeModal = false;
    public array $referralTree = [];
    public ?Collection $historyTimeline = null;

    public string $sortOrder = 'latest'; // ✨ متغیر جدید: 'latest' یا 'oldest'

    // ------------------------------------------------------------
    // Lifecycle
    // ------------------------------------------------------------

    public function mount(Letter $letter)
    {
        $this->letter = $letter->load([
            'user', 'sender', 'classification', 'background',
            'attachments',
            'referrals.user', 'referrals.creator',
            'events.user',
        ]);

        $this->referral = Referral::with([
            'user', 'creator',
            'comments' => fn($q) => $q->with(['user', 'replies.user']),
        ])
            ->where('letter_id', $letter->id)
            ->where('user_id', Auth::id())
            ->first();

        $this->checkAccess();

        // علامت‌گذاری به‌عنوان دیده‌شده در اولین مشاهده
        if ($this->referral && $this->referral->status === 'pending') {
            $this->referral->update(['status' => 'viewed']);
        }
    }

    protected function checkAccess()
    {
        $isSender = $this->letter->sender_id === Auth::id();
        $isRecipient = $this->referral !== null && $this->referral->user_id === Auth::id();

        if (! ($isSender || $isRecipient)) {
            abort(403, '⚠️ دسترسی غیرمجاز');
        }
    }

    // ------------------------------------------------------------
    // Computed Properties
    // ------------------------------------------------------------

    #[Computed]
    public function allComments(): Collection
    {
        $referralIds = $this->letter->referrals()->pluck('id');

        if ($referralIds->isEmpty()) {
            return collect();
        }

        // ✨ منطق مرتب‌سازی
        $order = $this->sortOrder === 'oldest' ? 'asc' : 'desc';

        return \App\Models\ReferralComment::whereIn('referral_id', $referralIds)
            ->with(['user', 'replies.user'])
            ->whereNull('parent_id')
            ->orderBy('created_at', $order) // استفاده از $order
            ->get();
    }
    #[Computed]
    public function shamsiCreatedDate(): string
    {
        return $this->formatShamsiDate($this->letter->created_at, 'Y/m/d');
    }

    #[Computed]
    public function shamsiDueDate(): ?string
    {
        return $this->letter->due_date
            ? $this->formatShamsiDate($this->letter->due_date, 'Y/m/d')
            : null;
    }

    #[Computed]
    public function shamsiReferralDate(): ?string
    {
        return $this->referral?->created_at
            ? $this->formatShamsiDate($this->referral->created_at)
            : null;
    }

    #[Computed]
    public function canEdit(): bool
    {
        return $this->letter->sender_id === Auth::id()
            && in_array($this->letter->status, [
                Letter::STATUS_DRAFT,
                Letter::STATUS_PENDING,
            ]);
    }

    #[Computed]
    public function canAct(): bool
    {
        return $this->referral !== null
            && in_array($this->referral->status, ['pending', 'viewed']);
    }

    #[Computed]
    public function availableUsers(): Collection
    {
        return User::select('id', 'name')->where('is_active', true)
            ->where('id', '!=', Auth::id())
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function activeReferrals(): Collection
    {
        return Referral::where('letter_id', $this->letter->id)
            ->with(['user:id,name,email,image,internal_number'])
            ->orderBy('created_at')
            ->get()
            ->map(fn($ref) => new ActiveReferralData(
                userId: $ref->user_id,
                name: $ref->user->name ?? 'سیستم',
                status: $ref->status,
                type: $ref->type,
                photoUrl: $this->getUserProfilePhotoUrl($ref->user),
                statusLabel: $ref->statusLabel(),
                typeLabel: $ref->typeLabel(),
                internalNumber: $ref->user->internal_number ?? '—',
            ));
    }

    // ------------------------------------------------------------
    // Actions (Modal & Submission)
    // ------------------------------------------------------------

    public function openActionModal(string $type): void
    {
        if (! $this->canAct) {
            $this->dispatch('show-toast', type: 'error', message: '❌ این نامه قابل اقدام نیست.');
            return;
        }

        $this->actionType = $type;
        $this->opinion = '';
        $this->resetErrorBag();
        $this->showActionModal = true;
    }

    public function submitAction(): void
    {
        // ۱. اعتبارسنجی اولیه دسترسی
        if (!$this->referral || !$this->canAct) {
            $this->dispatch('show-toast', type: 'error', message: '❌ خطای دسترسی یا ارجاع غیرفعال.');
            return;
        }

        // ۲. اعتبارسنجی فیلد نظر/علت (اجباری برای رد یا تصحیح)
        $validationRules = ['opinion' => 'nullable|string|max:500'];
        if (in_array($this->actionType, ['correction', 'reject'])) {
            $validationRules['opinion'] = 'required|string|max:500';
            $this->validate($validationRules, ['opinion.required' => 'لطفاً علت اقدام را وارد کنید.']);
        } else {
            $this->validateOnly('opinion', $validationRules);
        }

        DB::beginTransaction();
        try {
            // ۳. تعیین وضعیت جدید ارجاع
            $newReferralStatus = match ($this->actionType) {
                'view'       => \App\Models\Referral::STATUS_VIEWED,
                'approve'    => \App\Models\Referral::STATUS_COMPLETED,
                'reject'     => \App\Models\Referral::STATUS_REJECTED,
                'correction' => \App\Models\Referral::STATUS_COMPLETED, // در گردش کار شما تکمیل مرحله فعلی محسوب می‌شود
                default      => \App\Models\Referral::STATUS_VIEWED,
            };

            // ۴. به‌روزرسانی ارجاع فعلی
            $this->referral->update([
                'status'             => $newReferralStatus,
                'correction_message' => in_array($this->actionType, ['correction', 'reject']) ? $this->opinion : null,
            ]);

            $signer = Auth::user();
            $logDescription = 'اقدام ثبت شد.';
            $letterStatusUpdated = false;

            // ۵. اعمال منطق تغییر وضعیت کلی نامه (Global Status)
            if ($this->actionType === 'approve') {
                if ($this->referral->type === \App\Models\Referral::TYPE_APPROVAL) {
                    // الف) امضای نهایی و تایید
                    $this->letter->update([
                        'status'          => \App\Models\Letter::STATUS_APPROVED,
                        'final_signer_id' => $signer->id,
                        'signature_path'  => $signer->sign ?? null,
                        'signed_at'       => now(),
                    ]);
                    $logDescription = "نامه تأیید و توسط {$signer->name} امضا شد.";
                    $letterStatusUpdated = true;
                } else {
                    $logDescription = 'کاربر ارجاع را مشاهده و اقدام را تکمیل کرد.';
                }
            } elseif ($this->actionType === 'reject') {
                // ب) رد کلی نامه
                $this->letter->update(['status' => \App\Models\Letter::STATUS_REJECTED]);
                $logDescription = "نامه توسط {$signer->name} رد شد. دلیل: {$this->opinion}";
                $letterStatusUpdated = true;
            } elseif ($this->actionType === 'correction') {
                // ج) بازگشت جهت تصحیح (وضعیت نامه به در حال انتظار برمی‌گردد)
                $this->letter->update(['status' => \App\Models\Letter::STATUS_PENDING]);
                $logDescription = "نامه توسط {$signer->name} جهت تصحیح بازگردانده شد.";
                $letterStatusUpdated = true;
            }

            // ۶. چک کردن اتمام گردش کار (اگر ارجاع فعال دیگری نمانده باشد)
            $this->letter->refresh();
            if ($newReferralStatus === \App\Models\Referral::STATUS_COMPLETED && !$letterStatusUpdated) {
                $hasActiveReferrals = $this->letter->referrals()
                    ->where('status', \App\Models\Referral::STATUS_PENDING)
                    ->exists();

                if (!$hasActiveReferrals && !in_array($this->letter->status, [\App\Models\Letter::STATUS_APPROVED, \App\Models\Letter::STATUS_REJECTED])) {
                    $this->letter->update(['status' => \App\Models\Letter::STATUS_APPROVED]);
                    $logDescription .= ' | 🚀 سیستم: کلیه ارجاعات پایان یافت و نامه نهایی شد.';
                    $letterStatusUpdated = true;
                }
            }

            // ۷. ثبت در تاریخچه (Events)
            if (class_exists(\App\Models\LetterEvent::class)) {
                \App\Models\LetterEvent::create([
                    'letter_id'   => $this->letter->id,
                    'user_id'     => Auth::id(),
                    'type'        => $this->actionType,
                    'description' => $logDescription,
                ]);
            }

            // ۸. ارسال نوتیفیکیشن (اصلاح شده برای رفع خطای Named Parameter)
            $sender = $this->referral->creator;
            if ($sender && $sender->id !== Auth::id() && class_exists(\App\Notifications\ReferralActionTaken::class)) {
                // پارامترها مطابق Constructor: letterId, senderId, purpose
                Notification::send($sender, new \App\Notifications\ReferralActionTaken(
                    $this->letter->id,  // پارامتر اول
                    Auth::id(),        // پارامتر دوم
                    $this->actionType  // پارامتر سوم
                ));
            }

            DB::commit();

            // ۹. بستن مودال و نمایش پیام موفقیت
            $this->showActionModal = false;

            $toastMessage = match ($this->actionType) {
                'view'       => '✅ وضعیت به «دیده‌شده» تغییر یافت.',
                'reject'     => '❌ نامه با موفقیت رد شد.',
                'correction' => '🔄 نامه جهت تصحیح بازگردانده شد.',
                default      => $letterStatusUpdated ? '✅ نامه تأیید و نهایی شد.' : '✅ اقدام شما با موفقیت ثبت شد.'
            };

            $this->dispatch('show-toast', type: 'success', message: $toastMessage);

            $this->referral->refresh();
            $this->letter->refresh();

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Referral action failed', ['error' => $e->getMessage()]);
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در ثبت اقدام: ' . $e->getMessage());
        }
    }


    public function openHandoffModal(): void
    {
        if (! $this->referral || $this->referral->status === 'completed') {
            $this->dispatch('show-toast', type: 'warning', message: '⚠️ ارجاع فعلی شما تکمیل شده است.');
            return;
        }

        $this->showHandoffModal = true;
        $this->handoffUserId = null;
        $this->handoffPurpose = 'اقدام';
        $this->resetErrorBag();
    }

    public function submitHandoff(): void
    {
        if (! $this->referral || $this->referral->status === 'completed') {
            $this->dispatch('show-toast', type: 'error', message: '❌ ارجاع قبلی تکمیل شده است.');
            return;
        }

        $this->validate([
            'handoffUserId' => 'required|exists:users,id',
            'handoffPurpose' => 'required|string',
        ]);

        $newRecipient = User::find($this->handoffUserId);

        DB::beginTransaction();
        try {
            // 1. تکمیل ارجاع فعلی
            $this->referral->update(['status' => 'completed']);

            // 2. ایجاد ارجاع جدید (Handoff)
            $referralType = match ($this->handoffPurpose) {
                'امضا' => Referral::TYPE_APPROVAL ?? 'approval',
                'ثبت صادر' => Referral::TYPE_INFORMATION ?? 'information',
                'اقدام' => Referral::TYPE_ACTION ?? 'action',
                'نظر' => Referral::TYPE_INFORMATION ?? 'information',
                default => Referral::TYPE_ACTION ?? 'action'
            };

            Referral::create([
                'letter_id' => $this->letter->id,
                'user_id' => $this->handoffUserId,
                'type' => $referralType,
                'status' => 'pending',
                'created_by' => Auth::id(),
                'parent_id' => $this->referral->id,
            ]);

            // ثبت رویداد سیستمی
            if (class_exists(\App\Models\LetterEvent::class)) {
                \App\Models\LetterEvent::create([
                    'letter_id' => $this->letter->id,
                    'user_id' => Auth::id(),
                    'type' => 'handoff',
                    'description' => "نامه به کاربر «{$newRecipient->name}» جهت «{$this->handoffPurpose}» واگذار شد.",
                ]);
            }

            // ارسال نوتیفیکیشن به گیرنده جدید
            if (class_exists(\App\Notifications\LetterReferralSent::class)) {
                Notification::send($newRecipient, new \App\Notifications\LetterReferralSent(
                    letterId: $this->letter->id,
                    senderId: Auth::id(),
                    purpose: $this->handoffPurpose
                ));
            }

            DB::commit();
            $this->showHandoffModal = false;
            $this->referral->refresh();
            $this->dispatch('show-toast', type: 'success', message: '✅ واگذاری کار به «' . $newRecipient->name . '» با موفقیت انجام شد.');
            $this->redirect(request()->header('Referer')); // رفرش کامل برای به‌روزرسانی لیست ارجاعات

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Handoff failed', ['error' => $e->getMessage()]);
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در واگذاری کار.');
        }
    }


    // ------------------------------------------------------------
    // Comments
    // ------------------------------------------------------------

    public function postComment(): void
    {
        $isSender = $this->letter->sender_id === Auth::id();
        $hasActiveReferral = $this->referral !== null;

        // 1. تعیین دسترسی
        if (! $hasActiveReferral && ! $isSender) {
            $this->dispatch('show-toast', type: 'error', message: '❌ شما مجاز به ارسال نظر نیستید. (فقط فرستنده یا گیرنده ارجاع می‌تواند پیام ارسال کند)');
            return;
        }

        $this->validate(['newComment' => 'required|string|min:1|max:1000'], ['newComment.required' => 'پیام نمی‌تواند خالی باشد.']);

        // 2. تعیین Referral ID برای ذخیره کامنت
        $commentReferralId = null;

        if ($hasActiveReferral) {
            // اگر گیرنده ارجاع است، از ارجاع فعال خود استفاده کند.
            $commentReferralId = $this->referral->id;
        } elseif ($isSender) {
            // اگر فرستنده است (و ارجاع فعالی ندارد)، کامنت را به اولین ارجاع ثبت شده برای نامه متصل می‌کنیم
            // تا کامنت در دیتابیس در جدول ReferralComment ذخیره شود.
            $firstReferral = $this->letter->referrals()->oldest()->first();
            if ($firstReferral) {
                $commentReferralId = $firstReferral->id;
            } else {
                // اگر نامه هنوز هیچ ارجاعی ندارد، اجازه کامنت دادن به فرستنده را می‌دهیم اما باید یک رکورد ارجاع سیستمی برایش ایجاد کنیم (پیچیده است).
                // برای سادگی، فعلاً اگر ارجاعی وجود ندارد، جلوی ارسال کامنت را می‌گیریم (یا باید منطق خود را برای ثبت کامنت بدون referral_id بازنویسی کنید).
                $this->dispatch('show-toast', type: 'error', message: '❌ نامه هنوز هیچ ارجاع فعالی برای اتصال پیام ندارد.');
                return;
            }
        }

        // 3. ایجاد کامنت
        \App\Models\ReferralComment::create([
            'referral_id' => $commentReferralId,
            'user_id' => Auth::id(),
            'parent_id' => $this->replyParentId,
            'body' => $this->newComment,
        ]);

        $this->newComment = '';
        $this->replyParentId = null;

        $this->letter->refresh();

        $this->dispatch('show-toast', type: 'success', message: '✅ نظر شما ثبت شد.');
        $this->dispatch('focus-comment-input');
    }
    public function replyTo(int $parentId): void
    {
        $this->replyParentId = $parentId;
        $this->newComment = '';
        // ✨ اضافه کردن JS برای فوکوس روی تکست‌آریا پس از کلیک
        $this->dispatch('focus-comment-input');
    }

    public function deleteComment(int $commentId): void
    {
        if (! $this->referral) {
            $this->dispatch('show-toast', type: 'error', message: '❌ خطای دسترسی.');
            return;
        }

        try {
            $comment = \App\Models\ReferralComment::where('id', $commentId)
                ->where('referral_id', $this->referral->id)
                ->first();

            if ($comment && $comment->user_id === Auth::id()) {
                // حذف پاسخ‌های زیرمجموعه قبل از حذف والد
                \App\Models\ReferralComment::where('parent_id', $commentId)->delete();
                $comment->delete();

                $this->referral->refresh();
                $this->dispatch('show-toast', type: 'success', message: '✅ نظر با موفقیت حذف شد.');
            } else {
                $this->dispatch('show-toast', type: 'error', message: '❌ فقط نویسنده نظر می‌تواند آن را حذف کند.');
            }
        } catch (\Exception $e) {
            \Log::error('Comment deletion failed', ['error' => $e->getMessage()]);
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در حذف نظر.');
        }
    }




//    public function downloadPdf(): BaseStreamedResponse
//    {
//        $this->checkAccess();
//
//        $letterForm = $this->letter->background;
//        $filename = $letterForm?->background;
//
//
//
//        // 1. آماده‌سازی مسیر فیزیکی تصویر قالب برای mPDF
//        $watermarkImagePath = null;
//        if ($letterForm && $filename) {
//            $physicalPath = public_path('app/template/' . $filename);
//            if (File::exists($physicalPath)) {
//                $watermarkImagePath = str_replace('\\', '/', $physicalPath);
//            }
//        }
//
//        // 2. آماده‌سازی مسیر امضا برای mPDF
//        $signatureImagePath = null;
//        $signatureUrl = $this->getSignerSignatureUrl();
//
//        if ($signatureUrl) {
//            // اگر امضا URL آنلاین است، دانلود کنیم
//            if (Str::startsWith($signatureUrl, ['http://', 'https://'])) {
//                try {
//                    $tempPath = storage_path('app/temp/signature_' . time() . '.png');
//                    file_put_contents($tempPath, file_get_contents($signatureUrl));
//                    $signatureImagePath = $tempPath;
//                } catch (\Exception $e) {
//                    \Log::error('Failed to download signature image: ' . $e->getMessage());
//                }
//            }
//            // اگر مسیر لوکال است
//            elseif (Str::startsWith($signatureUrl, '/')) {
//                $localPath = public_path($signatureUrl);
//                if (File::exists($localPath)) {
//                    $signatureImagePath = $localPath;
//                }
//            }
//            // اگر مسیر storage است
//            else {
//                $storagePath = str_replace(Storage::url(''), '', $signatureUrl);
//                $fullPath = Storage::path($storagePath);
//                if (File::exists($fullPath)) {
//                    $signatureImagePath = $fullPath;
//                }
//            }
//        }
//
//
//        // 2. تنظیمات mPDF با پشتیبانی از فونت فارسی
//        $mpdfConfig = $this->getMpdfConfiguration();
//
//        // اضافه کردن فونت‌های فارسی به mPDF
//        $mpdfConfig['fontDir'] = array_merge(
//            $mpdfConfig['fontDir'] ?? [],
//            [public_path('panel/dist/fonts/')]
//        );
//
//        $mpdfConfig['fontdata'] = array_merge(
//            $mpdfConfig['fontdata'] ?? [],
//            [
//               'vazir' => [
//                   'R' => 'Vazir.ttf',
//                   'B' => 'Vazir-Bold.ttf',
//                   'I' => 'Vazir.ttf',
//                   'BI' => 'Vazir-Bold.ttf',
//                    'useOTL' => 0xFF,
//                   'useKashida' => 75,
//                ],
//                'IranNastaliq' => [
//                    'R' => 'IranNastaliq.ttf',
////                    'B' => 'IRANSansWeb_Bold.ttf',
//                    'useOTL' => 0xFF,
//                    'useKashida' => 75,
//                ],
//
//                'iransans' => [
//                    'R' => 'IRANSansWeb.ttf',
////                    'B' => 'IRANSansWeb_Bold.ttf',
//                    'useOTL' => 0xFF,
//                    'useKashida' => 75,
//                ],
//                'nazanin' => [
//                    'R' => 'B-NAZANIN.ttf',
//                    'B' => 'B-NAZANIN-Bold.ttf',
//                    'useOTL' => 0xFF,
//                    'useKashida' => 75,
//                ],
//                'lotus' => [
//                    'R' => 'B-Lotus.ttf',
//                    'useOTL' => 0xFF,
//                    'useKashida' => 75,
//                ],
//                'mitra' => [
//                    'R' => 'BMitra.ttf',
//                    'useOTL' => 0xFF,
//                    'useKashida' => 75,
//                ],
//                'yekan' => [
//                    'R' => 'BYekan.ttf',
//                    'useOTL' => 0xFF,
//                    'useKashida' => 75,
//                ]
//            ]
//        );
//
//        // تنظیم فونت پیش‌فرض
//        $mpdfConfig['default_font'] = $this->letter->font_family ?? 'vazir';
//
//        try {
//            $mpdf = new Mpdf($mpdfConfig);
//
//            // تنظیم حاشیه‌ها
//            $mpdf->SetMargins(16, 16, 48); // راست، چپ، بالا، پایین
//
//            // 3. تنظیم واترمارک (تصویر سربرگ در پس‌زمینه)
//            if ($watermarkImagePath) {
//                $mpdf->SetWatermarkImage($watermarkImagePath, 0.9, 'P', 'P', false, 0.1);
//                $mpdf->showWatermarkImage = true;
//            }
//
//            // 5. بارگیری داده‌های امضا برای ارسال به ویو
//            $signatureData = null;
//            if ($signatureImagePath && File::exists($signatureImagePath)) {
//                // تبدیل تصویر امضا به base64 برای نمایش در HTML
//                $signatureData = 'data:image/png;base64,' . base64_encode(file_get_contents($signatureImagePath));
//            }
//
//            // 4. ساخت HTML جزئیات نامه
//            $letterDetailsHtml = $this->buildPdfDetailsHtml();
//
//            // 5. رندر HTML محتوای اصلی
//            $html = view('livewire.admin.letter.pdf-show', [
//                'letter' => $this->letter,
//                'referral' => $this->referral,
//                'signatureData' => $signatureData,
//                'letterDetailsHtml' => $letterDetailsHtml,
//                'templateImageUrl' => null,
//                // ارسال تنظیمات فونت از دیتابیس
//                'font_family' => $this->letter->font_family ?? 'vazirmatn',
//                'font_size' => $this->letter->font_size ?? 14,
//                'line_height' => $this->letter->line_height ?? 1.6,
//                'content_font_size' => $this->letter->content_font_size ?? 1.05,
//                'content_line_height' => $this->letter->content_line_height ?? 1.8,
//                'content_align' => $this->letter->content_align ?? 'justify',
//            ])->render();
//
//            // نوشتن HTML به mPDF با تنظیمات فونت
//            $mpdf->WriteHTML('<div dir="rtl" style="font-family: ' . ($this->letter->font_family ?? 'vazirmatn') . ';">');
//            $mpdf->WriteHTML($html);
//            $mpdf->WriteHTML('</div>');
//
//            // تنظیم فونت برای تمام صفحات
//            $mpdf->SetDefaultBodyCSS('font-family', $this->letter->font_family ?? 'vazirmatn');
//
//            $safeCode = Str::slug($this->letter->code ?? 'letter-' . $this->letter->id);
//            $fileName = "letter-{$safeCode}.pdf";
//
//            return Response::streamDownload(function () use ($mpdf) {
//                echo $mpdf->Output('', 'S');
//            }, $fileName, ['Content-Type' => 'application/pdf']);
//
//        } catch (\Mpdf\MpdfException $e) {
//            \Log::error('mPDF Generation Failed: ' . $e->getMessage());
//            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در تولید PDF: ' . $e->getMessage());
//            return Response::streamDownload(function () {}, 'error.txt');
//        }
//    }

    public function downloadPdf(): BaseStreamedResponse
    {
        $this->checkAccess();

        $letterForm = $this->letter->background;
        $filename = $letterForm?->background;

        // 1. آماده‌سازی مسیر فیزیکی تصویر قالب
        $watermarkImagePath = null;
        if ($letterForm && $filename) {
            $physicalPath = public_path('app/template/' . $filename);
            if (File::exists($physicalPath)) {
                $watermarkImagePath = str_replace('\\', '/', $physicalPath);
            }
        }

        // 2. آماده‌سازی امضا (تبدیل به Base64 برای اطمینان از نمایش)
        $signatureData = null;
        $signatureUrl = $this->getSignerSignatureUrl();
        if ($signatureUrl) {
            $signatureImagePath = null;

            // تبدیل URL به مسیر فیزیکی سیستم
            if (Str::startsWith($signatureUrl, ['http://', 'https://'])) {
                // استخراج مسیر نسبی از URL
                $relativeUrl = str_replace(url('/'), '', $signatureUrl);
                $signatureImagePath = public_path(ltrim($relativeUrl, '/'));
            } else {
                // اگر از قبل مسیر بود
                $signatureImagePath = public_path(ltrim($signatureUrl, '/'));
            }

            // اگر در public پیدا نشد، در storage چک شود
            if (!File::exists($signatureImagePath)) {
                $storagePath = str_replace('/storage/', '', $signatureUrl);
                $signatureImagePath = storage_path('app/public/' . ltrim($storagePath, '/'));
            }

            // تبدیل نهایی به Base64 برای نمایش قطعی در PDF
            if (File::exists($signatureImagePath)) {
                $extension = pathinfo($signatureImagePath, PATHINFO_EXTENSION);
                $imgData = File::get($signatureImagePath);
                $signatureData = 'data:image/' . $extension . ';base64,' . base64_encode($imgData);
            }
        }

        // 3. تنظیمات mPDF
        $mpdfConfig = $this->getMpdfConfiguration();
        $mpdfConfig['fontDir'] = array_merge($mpdfConfig['fontDir'] ?? [], [public_path('panel/dist/fonts/')]);
        $mpdfConfig['fontdata'] = array_merge($mpdfConfig['fontdata'] ?? [], [
            'vazir' => ['R' => 'Vazir.ttf', 'B' => 'Vazir-Bold.ttf', 'useOTL' => 0xFF],
            'peyda' => ['R' => 'PeydaWebNoEn-Regular.ttf', 'B' => 'PeydaWebNoEn-Bold.ttf', 'useOTL' => 0xFF],
            'irannastaliq' => ['R' => 'IranNastaliq.ttf', 'useOTL' => 0xFF],
            'iransans' => ['R' => 'IRANSansWeb.ttf', 'useOTL' => 0xFF],
            'nazanin' => ['R' => 'B-NAZANIN.ttf', 'B' => 'B-NAZANIN-Bold.ttf', 'useOTL' => 0xFF],
            'lotus' => ['R' => 'B-Lotus.ttf', 'useOTL' => 0xFF],
            'mitra' => ['R' => 'BMitra.ttf', 'useOTL' => 0xFF],
            'yekan' => ['R' => 'BYekan.ttf', 'useOTL' => 0xFF],
        ]);

        $mpdfConfig['default_font'] = $this->letter->font_family ?? 'vazir';

        try {
            $mpdf = new \Mpdf\Mpdf($mpdfConfig);
            $mpdf->SetMargins(16, 16, 48);
            $mpdf->SetDirectionality('rtl');

            if ($watermarkImagePath) {
                $mpdf->SetWatermarkImage($watermarkImagePath, 0.9, 'P', 'P', false);
                $mpdf->showWatermarkImage = true;
            }

            // 4. رندر HTML و حل مشکل مسیر تصاویر
            $html = view('livewire.admin.letter.pdf-show', [
                'letter' => $this->letter,
                'signatureData' => $signatureData,
                'font_family' => $this->letter->font_family ?? 'vazir',
                // سایر متغیرها...
            ])->render();

            // نکته بسیار مهم: تبدیل آدرس‌های وب تصاویر به مسیر فیزیکی برای mPDF
            $html = str_replace(url('storage'), public_path('storage'), $html);

            // 5. خروجی نهایی
            // محصور کردن کل محتوا در یک div با فونت اصلی
            $mpdf->WriteHTML('<div dir="rtl" style="font-family: ' . ($this->letter->font_family ?? 'vazir') . ';">' . $html . '</div>');

            $fileName = "letter-" . Str::slug($this->letter->code ?? $this->letter->id) . ".pdf";
            return Response::streamDownload(function () use ($mpdf) {
                echo $mpdf->Output('', 'S');
            }, $fileName, ['Content-Type' => 'application/pdf']);

        } catch (\Exception $e) {
            \Log::error($e->getMessage());
            return response('خطا در تولید فایل', 500);
        }
    }

    /**
     * ساخت HTML جزئیات نامه
     */
    protected function buildPdfDetailsHtml(): string
    {
        $priorityClasses = [
            'urgent' => 'color: #E91E63; font-weight: bold;',
            'high' => 'color: #FF9800; font-weight: bold;',
            'normal' => 'color: #4CAF50;',
            'low' => 'color: #2196F3;'
        ];

        $priorityStyle = $priorityClasses[$this->letter->priority] ?? '';

        return '
    <div class="detail-row">
        <span class="detail-label">موضوع:</span>
        <span class="detail-value fw-bold">' . ($this->letter->title ?? '—') . '</span>
    </div>
    <div class="detail-row">
        <span class="detail-label">از طرف:</span>
        <span class="detail-value">' . ($this->letter->from_who ?? '—') . '</span>
    </div>
    <div class="detail-row">
        <span class="detail-label">به:</span>
        <span class="detail-value">' . ($this->letter->from_where ?? '—') . '</span>
    </div>
    <div class="detail-row">
        <span class="detail-label">تاریخ:</span>
        <span class="detail-value">' . ($this->letter->letter_date ? \Carbon\Carbon::parse($this->letter->letter_date)->format('Y/m/d') : '—') . '</span>
    </div>
    <div class="detail-row">
        <span class="detail-label">کد:</span>
        <span class="detail-value fw-bold">' . ($this->letter->code ?? '—') . '</span>
    </div>
    ' . ($this->letter->priority ? '
    <div class="detail-row">
        <span class="detail-label">اولویت:</span>
        <span class="detail-value">
            <span style="' . $priorityStyle . '">
                ' . ($this->letter->priority_label ?? $this->letter->priority) . '
            </span>
        </span>
    </div>
    ' : '');
    }

    /**
     * تنظیمات mPDF
     */
    protected function getMpdfConfiguration(): array
    {
        return [
            'mode' => 'utf-8',
            'format' => 'A4',
            'orientation' => 'P',
            'margin_left' => 17,
            'margin_right' => 17,
            'margin_top' => 25,
            'margin_bottom' => 45,
            'margin_header' => 0,
            'margin_footer' => 0,
            'tempDir' => storage_path('app/mpdf/tmp'),
            'default_font_size' => $this->letter->font_size ?? 12,
            'default_font' => $this->letter->font_family ?? 'vazirmatn',
            'direction' => 'rtl',
            'autoScriptToLang' => true,
            'autoLangToFont' => true,
            'autoArabic' => true,
            'useSubstitutions' => true,
            'showWatermarkText' => false,
            'watermarkImgBehind' => true,
        ];
    }

    /**
     * ✨ متد جدید: ساخت HTML برای جزئیات نامه (برای قرارگیری در بدنه محتوا)
     */

    // ✨ متد کمکی تبدیل تاریخ شمسی (رفع خطا)
    public function formatShamsiDate(mixed $date, string $format = 'Y/m/d H:i'): string
    {
        if (! $date) { return '—'; }
        try {
            $carbonDate = (Carbon::parse($date));
            return Jalalian::fromCarbon($carbonDate)->format($format);
        } catch (\Exception $e) {
            return (string) $date;
        }
    }

    // ✨ متد کمکی آواتار (رفع خطا)
    public function getUserProfilePhotoUrl(mixed $user): string
    {
        if (! is_object($user) || ! isset($user->name)) {
            $initial = mb_substr($user?->name ?? 'ک', 0, 1);
        } else {
            $initial = mb_substr($user->name, 0, 1);
        }

        $encodedInitial = urlencode($initial);
        $default = "https://ui-avatars.com/api/?name={$encodedInitial}&background=EBF4FF&color=2563EB&size=64";

        if (! $user) { return $default; }

        if (isset($user->image) && $user->image) {
            if (str_starts_with($user->image, 'http') || str_starts_with($user->image, 'https')) {
                return $user->image;
            } elseif (str_starts_with($user->image, '/')) {
                return url($user->image);
            } else {
                $path = ltrim($user->image, '/');
                if (Storage::disk('public')->exists($path)) {
                    return Storage::url($path);
                }
            }
        }
        return $default;
    }

    // ------------------------------------------------------------
    // Render & Tree
    // ------------------------------------------------------------

    public function openReferralTreeModal(): void
    {
        try {
            $this->letter->load(['referrals.user', 'events.user']);

            // منطق ساخت تایم‌لاین (برای نمایش شیک در مودال)
            $timelineItems = collect();
            $this->letter->referrals->each(function ($ref) use ($timelineItems) {
                $timelineItems->push((object)[
                    'type' => 'referral_sent',
                    'date' => $ref->created_at,
                    'user' => $ref->creator,
                    'description' => 'ارجاع به ' . ($ref->user->name ?? '—') . ' جهت ' . $ref->typeLabel(),
                ]);
            });
            $this->letter->events->each(function ($event) use ($timelineItems) {
                $timelineItems->push((object)[
                    'type' => $event->type,
                    'date' => $event->created_at,
                    'user' => $event->user,
                    'description' => $event->description,
                ]);
            });

            $this->historyTimeline = $timelineItems->sortByDesc('date')->unique('description')->take(20);

            $this->referralTree = $this->buildReferralTree($this->letter->referrals->sortBy('created_at'));
            $this->showReferralTreeModal = true;
        } catch (\Exception $e) {
            \Log::error('Referral Tree Failed: ' . $e->getMessage());
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در بارگذاری تاریخچه گردش');
        }
    }

    protected function buildReferralTree(Collection $referrals): array
    {
        $grouped = $referrals->groupBy('parent_id');
        $build = function ($parentId = null) use (&$build, $grouped) {
            return $grouped->get($parentId, collect())->map(function ($referral) use (&$build) {
                return [
                    'referral' => $referral,
                    'children' => $build($referral->id),
                ];
            })->all();
        };
        return $build();
    }
    public function getSignerSignatureUrl(): ?string
    {
        // اولویت 1: امضای ذخیره شده در نامه
        if ($this->letter->signature_path) {
            $path = $this->letter->signature_path;
            if (Str::startsWith($path, ['http', '/'])) {
                return $path;
            }
            if (Storage::disk('public')->exists($path)) {
                return Storage::url($path);
            }
        }

        // اولویت 2: امضای امضاکننده نهایی
        if ($this->letter->finalSigner && $this->letter->finalSigner->sign) {
            $path = $this->letter->finalSigner->sign;
            if (Str::startsWith($path, ['http', '/'])) {
                return $path;
            }
            if (Storage::disk('public')->exists($path)) {
                return Storage::url($path);
            }
        }

        // اولویت 3: امضای کاربر اصلی
        if ($this->letter->user && $this->letter->user->sign) {
            $path = $this->letter->user->sign;
            if (Str::startsWith($path, ['http', '/'])) {
                return $path;
            }
            if (Storage::disk('public')->exists($path)) {
                return Storage::url($path);
            }
        }

        return null;
    }
//    public function getSignerSignatureUrl(): ?string
//    {
//        // امضاکننده اصلی نامه (user_id)
//        $signer = $this->letter->user;
//
//        if ($signer && $signer->sign) {
//            $path = $signer->sign;
//            // بررسی کنید آیا مسیر کامل است یا نیاز به Storage URL دارد
//            if (Str::startsWith($path, 'http') || Str::startsWith($path, '/')) {
//                return $path;
//            }
//            if (Storage::disk('public')->exists($path)) {
//                return Storage::url($path);
//            }
//        }
//        return null;
//    }


    public function closeReferralTreeModal(): void
    {
        $this->showReferralTreeModal = false;
        $this->referralTree = [];
        $this->historyTimeline = collect();
    }

    public function render()
    {
        return view('livewire.admin.letter.letter-show');
    }
}
