<?php

namespace App\Livewire\Admin\Letter;

use App\Models\Letter;
use App\Models\Referral;
use App\Models\User;
use App\Models\LetterEvent; // فرض می‌کنیم LetterEvent وجود دارد
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Title;
use Livewire\Attributes\Computed;
use Livewire\Component;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

#[Title('مرکز عملیات نامه‌نگاری من')]
#[Layout('admin.master')]
class MyCorrespondenceHub extends Component
{
    // ✨ حذف متغیرهای عمومی آماری (استفاده از Computed)

    // ✨ مدیریت لودینگ دستی (Manual Refresh)
    public bool $isRefreshing = false;

    // ------------------------------------------------------------
    // Computed Properties (آمار و واکشی داده‌ها)
    // ------------------------------------------------------------

    #[Computed]
    public function stats(): array
    {
        $userId = Auth::id();
        $today = Carbon::today();

        // کوئری‌های پایه
        $sentQuery = Letter::where('sender_id', $userId);
        $pendingRefQuery = Referral::where('user_id', $userId)->where('status', Referral::STATUS_PENDING ?? 'pending');
        $allRefQuery = Referral::where('user_id', $userId);
        $completedRefQuery = Referral::where('user_id', $userId)->where('status', Referral::STATUS_COMPLETED ?? 'completed');

        // محاسبات
        $totalReceived = $allRefQuery->count();
        $completedCount = $completedRefQuery->count();
        $completionRate = $totalReceived > 0 ? round(($completedCount / $totalReceived) * 100) : 100;

        $urgentPendingCount = (clone $pendingRefQuery)
            ->whereHas('letter', fn($q) => $q->where('priority', 'urgent'))
            ->count();

        return [
            'totalSent' => $sentQuery->count(),
            'incomingRegisteredCount' => Letter::where('user_id', $userId)
                ->where('type', Letter::TYPE_INCOMING ?? 'incoming')
                ->count(),
            'totalReceived' => $totalReceived,
            'totalPendingReferrals' => $pendingRefQuery->count(),
            'pendingToday' => (clone $pendingRefQuery)->whereDate('created_at', $today)->count(),
            'urgentPendingCount' => $urgentPendingCount,
            'completionRate' => $completionRate,
            'completedCount' => $completedCount,
        ];
    }

    #[Computed]
    public function topPendingRefs(): Collection
    {
        return Referral::where('user_id', Auth::id())
            ->where('status', Referral::STATUS_PENDING ?? 'pending')
            ->with(['letter:id,title,code,priority'])
            ->orderBy('created_at', 'asc')
            ->take(5)
            ->get();
    }

    #[Computed]
    public function recentSent(): Collection
    {
        return Letter::where('sender_id', Auth::id())
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->get();
    }

    #[Computed]
    public function activityTimeline(): Collection
    {
        $userId = Auth::id();

        // ترکیبی از رویدادهای ارجاع/تایید و ایجاد نامه (برای تایم‌لاین)
        // این بخش نیاز به تعریف دقیق مدل LetterEvent دارد.
        // شبیه‌سازی برای نمایش بهتر:
        return collect([
            // ارجاعات دریافتی (Referral)
            Referral::where('user_id', $userId)->orderByDesc('created_at')->limit(3)->get()->map(fn($r) => [
                'type' => 'ارجاع دریافتی',
                'subject' => $r->letter->title ?? 'نامه ارجاعی',
                'time' => $r->created_at,
                'status' => $r->status,
                'icon' => 'fas fa-arrow-down',
                'color' => '#f59e0b', // warning
            ]),
            // رویدادهای نامه (LetterEvent - مثلا تایید یا ارسال)
            LetterEvent::where('user_id', $userId)->orderByDesc('created_at')->limit(3)->get()->map(fn($e) => [
                'type' => $e->type === 'approval' ? 'تایید' : 'عملیات',
                'subject' => $e->description,
                'time' => $e->created_at,
                'status' => $e->type,
                'icon' => 'fas fa-check',
                'color' => '#10b981', // success
            ]),
        ])->flatten(1)->sortByDesc('time')->take(6);
    }

    // ------------------------------------------------------------
    // Actions & Core Logic
    // ------------------------------------------------------------

    public function loadData(): void
    {
        // از آنجایی که از Computed Properties استفاده می‌کنیم،
        // فقط کافی است Livewire را مجبور به رفرش کنیم.
        // متدهای computed به صورت خودکار فراخوانی می‌شوند.
        $this->isRefreshing = true;

        // این dispatch پس از محاسبات خودکار، DOM را به‌روز می‌کند
        $this->dispatch('refresh');

        // زمان‌بندی برای خاموش کردن حالت رفرش
        DB::afterCommit(fn() => $this->isRefreshing = false);
    }

    public function performActionOnReferral(int $referralId): void
    {
        // در برنامه واقعی: باید Modal اقدام باز شود.
        $this->dispatch('show-toast', type: 'info', message: "وظیفه ارجاع $referralId برای اقدام باز شد.");
        // Redirect or open modal here
        // $this->loadData();
    }

    public function render()
    {
        return view('livewire.admin.letter.my-correspondence-hub');
    }
}
