<?php

namespace App\Livewire\Admin\Letter;

use App\Models\Letter;
use App\Models\Referral;
use App\Models\User;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;
use Morilog\Jalali\Jalalian;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

#[Layout('admin.master')]
class ReceivedLetters extends Component
{
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    // 🔍 Filters
    #[Url(except: '')]
    public string $searchText = '';

    #[Url(except: '')]
    public string $filterReferralStatus = '';

    #[Url(except: '')]
    public ?int $filterReferrerId = null;

    #[Url(except: '')]
    public string $filterFromDate = '';

    #[Url(except: '')]
    public string $filterToDate = '';

    #[Url]
    public int $perPage = 10;

    // 💬 Modals
    public bool $showPreviewModal = false;
    public bool $showActionModal = false;
    public bool $showReferralTreeModal = false;

    // 📄 Modal Data
    public ?object $previewLetter = null; // ✅ اکنون یک stdClass است
    public ?Referral $currentReferral = null;
    public array $referralTree = [];
    public ?Collection $historyTimeline = null;

    // ⚙️ Action Modal
    public string $actionType = '';
    public string $correctionMessage = '';

    // 👥 Top Referrers for Smart Filter
    public Collection $topReferrers;

    // ------------------------------------------------------------
    // Lifecycle & Hooks
    // ------------------------------------------------------------

    public function mount(): void
    {
        $this->topReferrers = $this->getTopReferrers();
    }

    public function updating(string $property): void
    {
        if (in_array($property, ['searchText', 'filterReferralStatus', 'filterReferrerId', 'perPage', 'filterFromDate', 'filterToDate'])) {
            $this->resetPage();
        }
    }

    #[On('refreshReceivedLetters')]
    public function refreshLetters(): void
    {
        $this->resetPage();
    }

    // ------------------------------------------------------------
    // Helpers
    // ------------------------------------------------------------

    public function formatShamsiDate(mixed $date, string $format = 'Y/m/d H:i'): string
    {
        if (!$date) return '—';
        try {
            $carbonDate = ($date instanceof \Carbon\Carbon) ? $date : \Carbon\Carbon::parse($date);
            return Jalalian::fromCarbon($carbonDate)->format($format);
        } catch (\Exception $e) {
            \Log::warning('Failed to format Shamsi date', ['input' => $date, 'error' => $e->getMessage()]);
            return (string) $date;
        }
    }

    public function getUserProfilePhotoUrl(mixed $user): string
    {
        $name = is_object($user) && isset($user->name) ? $user->name : 'Unknown';
        $initial = mb_substr($name, 0, 1);
        $encodedInitial = urlencode($initial);
        $default = "https://ui-avatars.com/api/?name={$encodedInitial}&background=EBF4FF&color=2563EB&size=128";

        if (!is_object($user)) {
            return $default;
        }

        if (isset($user->image) && $user->image) {
            if (str_starts_with($user->image, 'http') || str_starts_with($user->image, 'https')) {
                return $user->image;
            } elseif (str_starts_with($user->image, '/')) {
                return url($user->image);
            } else {
                $path = ltrim($user->image, '/');
                if (Storage::disk('public')->exists($path)) {
                    return Storage::url($path);
                }
            }
        }

        return $default;
    }

    protected function getTopReferrers(): Collection
    {
        $topReferrerIds = Referral::where('user_id', Auth::id())
            ->select('created_by')
            ->groupBy('created_by')
            ->orderByRaw('COUNT(*) DESC')
            ->limit(5)
            ->pluck('created_by');

        if ($topReferrerIds->isEmpty()) {
            return collect();
        }

        $referrers = User::whereIn('id', $topReferrerIds)
            ->select('id', 'name', 'image')
            ->get();

        $counts = Referral::where('user_id', Auth::id())
            ->whereIn('created_by', $topReferrerIds)
            ->groupBy('created_by')
            ->select('created_by', DB::raw('COUNT(*) as count'))
            ->pluck('count', 'created_by');

        return $referrers->map(function ($user) use ($counts) {
            return (object) [
                'id' => $user->id,
                'name' => $user->name ?? 'سیستم',
                'count' => $counts[$user->id] ?? 0,
                'photo_url' => $this->getUserProfilePhotoUrl($user),
            ];
        });
    }

    public function cloneLetterContent(int $letterId): void
    {
        try {
            $originalLetter = Letter::with('copies', 'project')->findOrFail($letterId);
        } catch (ModelNotFoundException) {
            $this->dispatch('show-toast', type: 'error', message: '❌ نامه اصلی برای کپی یافت نشد.');
            return;
        } catch (\Exception $e) {
            \Log::error("Failed to retrieve letter for cloning: " . $e->getMessage());
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در دسترسی به نامه.');
            return;
        }

        try {
            $clonedData = [
                'title' => 'کپی از ' . $originalLetter->title,
                'content' => $originalLetter->content,
                'from_who' => $originalLetter->from_who,
                'from_where' => $originalLetter->from_where,
                'classification_id' => $originalLetter->classification_id,
                'background_id' => $originalLetter->background_id,
                'priority' => $originalLetter->priority,
                'type' => $originalLetter->type,
                'confidential' => $originalLetter->confidential,
                'signer_id' => $originalLetter->user_id,
                'signature_position' => $originalLetter->signature_position,
                'ccs' => $originalLetter->copies->map(fn($copy) => [
                    'name' => $copy->name,
                    'position' => $copy->position
                ])->toArray(),
            ];

            session()->put('cloned_letter_data', $clonedData);
            session()->put('cloned_project_id', $originalLetter->project_id ?? null);

        } catch (\Exception $e) {
            \Log::error("Cloning data preparation failed: " . $e->getMessage());
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در آماده‌سازی داده‌های کپی.');
            return;
        }

        $this->dispatch('show-toast', type: 'info', message: '💡 محتوای نامه کپی شد. هدایت به صفحه ایجاد نامه...');
        $this->redirect(route('admin.projects.letters.create', [
            'project' => $originalLetter->project_id
        ]), navigate: true);
    }

    public function resetFilters(): void
    {
        $this->searchText = '';
        $this->filterReferralStatus = '';
        $this->filterReferrerId = null;
        $this->filterFromDate = '';
        $this->filterToDate = '';
        $this->resetPage();
    }

    // ------------------------------------------------------------
    // Actions
    // ------------------------------------------------------------

    public function markAsRead(int $referralId): void
    {
        $referral = Referral::where('id', $referralId)
            ->where('user_id', Auth::id())
            ->where('status', Referral::STATUS_PENDING)
            ->first();

        if ($referral) {
            $referral->update(['status' => Referral::STATUS_VIEWED]);
            $this->dispatch('refreshReceivedLetters');
            $this->dispatch('show-toast', type: 'success', message: '✅ وضعیت به «دیده شده» تغییر یافت.');
        } else {
            $this->dispatch('show-toast', type: 'info', message: '💡 نامه از قبل دیده‌شده یا یافت نشد.');
        }
    }

    public function openActionModal(int $referralId, string $type): void
    {
        $referral = Referral::with(['letter', 'creator'])
            ->where('id', $referralId)
            ->where('user_id', Auth::id())
            ->whereIn('status', [Referral::STATUS_PENDING, Referral::STATUS_VIEWED])
            ->first();

        if (!$referral) {
            $this->dispatch('show-toast', type: 'error', message: '❌ ارجاع مربوطه یافت نشد.');
            return;
        }

        $this->currentReferral = $referral;
        $this->actionType = $type;
        $this->correctionMessage = '';
        $this->resetErrorBag();
        $this->showActionModal = true;
    }

    public function submitAction(): void
    {
        if (!$this->currentReferral) return;

        \DB::beginTransaction();
        try {
            $data = ['status' => match ($this->actionType) {
                'completed' => Referral::STATUS_COMPLETED,
                'correction' => Referral::STATUS_CANCELED,
                default => Referral::STATUS_VIEWED,
            }];

            if ($this->actionType === 'correction') {
                $this->validate(['correctionMessage' => 'required|string|max:500']);
                $data['correction_message'] = $this->correctionMessage;
            }

            $this->currentReferral->update($data);

            // ثبت رویداد سیستمی
            if (class_exists(\App\Models\LetterEvent::class)) {
                \App\Models\LetterEvent::create([
                    'letter_id' => $this->currentReferral->letter_id,
                    'user_id' => Auth::id(),
                    'type' => $this->actionType,
                    'description' => match ($this->actionType) {
                        'completed' => 'اقدام با موفقیت تکمیل شد.',
                        'correction' => 'نامه جهت تصحیح بازگردانده شد: ' . $this->correctionMessage,
                        default => 'وضعیت به‌روزرسانی شد.',
                    },
                ]);
            }

            // ارسال نوتیفیکیشن به ارجاع‌دهنده
            $sender = $this->currentReferral->creator;
            if ($sender && $sender->id !== Auth::id() && class_exists(\App\Notifications\ReferralActionTaken::class)) {
                Notification::send($sender, new \App\Notifications\ReferralActionTaken(
                    $this->currentReferral->letter_id,
                    Auth::id(),
                    $this->actionType
                ));
            }

            \DB::commit();
            $this->dispatch('refreshReceivedLetters');
            $this->closeActionModal();
            $this->dispatch('show-toast', type: 'success', message: match ($this->actionType) {
                'completed' => '✅ اقدام با موفقیت تکمیل شد.',
                'correction' => '🔄 نامه برای تصحیح بازگردانده شد.',
                default => '✅ وضعیت به‌روزرسانی شد.',
            });

        } catch (\Exception $e) {
            \DB::rollBack();
            \Log::error('Action submission failed', [
                'user' => Auth::id(),
                'referral_id' => optional($this->currentReferral)->id,
                'action' => $this->actionType,
                'error' => $e->getMessage()
            ]);
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در ثبت اقدام. لطفاً مجدداً تلاش کنید.');
        }
    }

    public function closeActionModal(): void
    {
        $this->showActionModal = false;
        $this->currentReferral = null;
        $this->actionType = '';
        $this->correctionMessage = '';
        $this->resetErrorBag();
    }

    public function openReferralTreeModal(int $letterId): void
    {
        try {
            $letter = Letter::with(['referrals.user', 'events.user'])->findOrFail($letterId);

            $timelineItems = collect();
            $letter->referrals->each(function ($ref) use ($timelineItems) {
                $timelineItems->push((object) [
                    'type' => 'ارجاع',
                    'date' => $ref->created_at,
                    'user' => $ref->creator,
                    'description' => 'ارجاع به ' . ($ref->user->name ?? '—') . ' جهت ' . ($ref->typeLabel() ?? 'اقدام'),
                ]);
            });
            $letter->events->each(function ($event) use ($timelineItems) {
                $timelineItems->push((object) [
                    'type' => $event->type,
                    'date' => $event->created_at,
                    'user' => $event->user,
                    'description' => $event->description,
                ]);
            });

            $this->historyTimeline = $timelineItems->sortByDesc('date')->take(30);
            $this->referralTree = $this->buildReferralTree($letter->referrals->sortBy('created_at'));
            $this->showReferralTreeModal = true;

        } catch (\Exception $e) {
            \Log::error('Referral tree load error', ['letter_id' => $letterId, 'error' => $e->getMessage()]);
            $this->dispatch('show-toast', type: 'error', message: '❌ خطایی در بارگذاری درخت ارجاع رخ داد.');
        }
    }

    protected function buildReferralTree(Collection $referrals): array
    {
        $grouped = $referrals->groupBy('parent_id');

        $build = function ($parentId = null) use (&$build, $grouped) {
            return $grouped->get($parentId, collect())->map(function ($referral) use (&$build) {
                return [
                    'referral' => $referral,
                    'children' => $build($referral->id),
                ];
            })->all();
        };

        return $build();
    }

    public function closeReferralTreeModal(): void
    {
        $this->showReferralTreeModal = false;
        $this->referralTree = [];
        $this->historyTimeline = collect();
    }

    public function closePreviewModal(): void
    {
        $this->showPreviewModal = false;
        $this->previewLetter = null;
    }

    // ✅ این متد اصلاح شده — اکنون از stdClass استفاده می‌کند
    public function showPreview(int $letterId): void
    {
        try {
            $letter = Letter::with([
                'user', 'sender', 'classification', 'background',
                'referrals' => fn($q) => $q->where('user_id', Auth::id())->with(['creator']),
                'attachments'
            ])->findOrFail($letterId);

            $referral = $letter->referrals->first();

            // ✅ تبدیل به stdClass — کلید رفع خطا
            $this->previewLetter = (object) [
                'letter' => $letter,
                'referral' => $referral,
                'shamsiCreatedDate' => $this->formatShamsiDate($letter->created_at, 'Y/m/d'),
            ];

            $this->showPreviewModal = true;

        } catch (ModelNotFoundException) {
            $this->dispatch('show-toast', type: 'error', message: '❌ نامه مورد نظر یافت نشد.');
        } catch (\Exception $e) {
            \Log::error('Preview load error for letter ID: ' . $letterId, [
                'user_id' => Auth::id(),
                'error' => $e->getMessage(),
            ]);
            $this->dispatch('show-toast', type: 'error', message: '❌ خطایی در بارگذاری پیش‌نمایش رخ داد.');
        }
    }

    // ------------------------------------------------------------
    // Computed Properties
    // ------------------------------------------------------------

    #[Computed]
    public function referrals()
    {
        return Referral::with([
            'letter:id,title,code,created_at,due_date,sender_id,status',
            'letter.sender:id,name,image',
            'creator:id,name,image'
        ])
            ->where('user_id', Auth::id())
            ->when($this->searchText, function ($q) {
                $q->whereHas('letter', function ($sub) {
                    $sub->where('title', 'like', "%{$this->searchText}%")
                        ->orWhere('content', 'like', "%{$this->searchText}%")
                        ->orWhere('code', 'like', "%{$this->searchText}%");
                });
            })
            ->when($this->filterReferralStatus, function ($q) {
                $q->where('status', $this->filterReferralStatus);
            })
            ->when($this->filterReferrerId, function ($q) {
                $q->where('created_by', $this->filterReferrerId);
            })
            ->when($this->filterFromDate, function ($q) {
                $q->whereDate('created_at', '>=', $this->filterFromDate);
            })
            ->when($this->filterToDate, function ($q) {
                $q->whereDate('created_at', '<=', $this->filterToDate);
            })
            ->orderByDesc('created_at')
            ->paginate($this->perPage);
    }

    #[Computed]
    public function statusCounts(): array
    {
        $baseQuery = Referral::where('user_id', Auth::id());
        return [
            'pending'    => $baseQuery->clone()->where('status', Referral::STATUS_PENDING)->count(),
            'viewed'     => $baseQuery->clone()->where('status', Referral::STATUS_VIEWED)->count(),
            'completed'  => $baseQuery->clone()->where('status', Referral::STATUS_COMPLETED)->count(),
            'canceled'   => $baseQuery->clone()->where('status', Referral::STATUS_CANCELED)->count(),
            'total'      => $baseQuery->clone()->count(),
        ];
    }

    // ------------------------------------------------------------
    // Render
    // ------------------------------------------------------------

    public function render()
    {
        return view('livewire.admin.letter.received-letters', [
            'referrals'      => $this->referrals,
            'statusCounts'   => $this->statusCounts,
            'topReferrers'   => $this->topReferrers,
        ]);
    }
}
