<?php

namespace App\Livewire\Admin\Letter;

use App\Models\Letter;
use App\Models\Project;
use App\Models\User;
use App\Models\Classification;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Morilog\Jalali\Jalalian;

class RegisterIncomingLetter extends Component
{
    use WithFileUploads;

    // --- فیلدهای اطلاعات خارجی و اصلی ---
    public $external_sender = '';
    public $external_number = '';
    public $external_date = null;

    // --- فیلدهای داخلی و ارجاع ---
    public $subject = '';
    public $project_id = null;
    public $classification_id = null;
    public $recipient_user_id = null;
    public $attachments = [];
    public $confidential = false;

    // --- داده‌های کمکی ---
    public $projects;
    public $classifications;
    public $users;

    public $is_submitting = false;

    protected $rules = [
        'external_sender' => 'required|string|max:255',
        'external_number' => 'required|string|max:50',
        'external_date' => 'required|date',
        'subject' => 'required|string|max:500',
        'project_id' => 'nullable|exists:projects,id',
        'classification_id' => 'required|exists:classifications,id',
        'recipient_user_id' => 'required|exists:users,id',
        'attachments' => 'required|array|min:1|max:5',
        'attachments.*' => 'file|max:20480|mimes:pdf,jpg,jpeg,png',
    ];

    public function mount()
    {
        $this->projects = Project::select('id', 'name', 'code')->get();
        $this->classifications = Classification::select('id', 'name')->get();
        $this->users = User::select('id', 'name')->where('is_active', true)->get();

        // ✅ تنظیم تاریخ میلادی صحیح (بدون فرمت اضافه)
        $this->external_date = now()->toDateString(); // مثلاً: 2025-10-26

        $this->recipient_user_id = Auth::id();
    }

    public function removeAttachment($index)
    {
        unset($this->attachments[$index]);
        $this->attachments = array_values($this->attachments);
    }

    public function saveIncoming()
    {
        $this->is_submitting = true;
        $this->validate();

        try {
            $letter = Letter::create([
                'title' => 'نامه وارده: ' . $this->subject,
                'content' => 'نامه وارده با شماره ' . $this->external_number . ' از ' . $this->external_sender . ' ثبت شد.',
                'from_who' => $this->external_sender,
                'user_id' => Auth::id(),
                'sender_id' => Auth::id(),
                'type' => Letter::TYPE_INCOMING,
                'status' => Letter::STATUS_PENDING,
                'external_number' => $this->external_number,
                'external_date' => $this->external_date,
                'project_id' => $this->project_id,
                'classification_id' => $this->classification_id,
                'recipient_user_id' => $this->recipient_user_id,
                'code' => 'IN/' . jdate('Y') . '/' . (Letter::where('type', 'incoming')->whereYear('created_at', now()->year)->count() + 1),
                'unique_code' => Str::uuid(),
            ]);

            $this->handleAttachments($letter);

            $this->dispatch('show-toast', type: 'success', message: "نامه وارده ({$letter->code}) با موفقیت ثبت و ارجاع شد.");
            $this->resetForm();

        } catch (\Exception $e) {
            \Log::error('Incoming Letter Save Error: ' . $e->getMessage());
            $this->dispatch('show-toast', type: 'error', message: 'خطای سیستمی در ثبت نامه رخ داد.');
        } finally {
            $this->is_submitting = false;
        }
    }

    private function handleAttachments(Letter $letter)
    {
        if (empty($this->attachments)) return;

        foreach ($this->attachments as $file) {
            $path = $file->store('incoming_attachments/' . $letter->id, 'public');
            $letter->attachments()->create([
                'file_path' => $path,
                'original_name' => $file->getClientOriginalName(),
            ]);
        }
    }

    private function resetForm()
    {
        $this->reset([
            'external_sender', 'external_number', 'subject',
            'project_id', 'classification_id', 'attachments',
            'recipient_user_id', 'confidential'
        ]);
        $this->external_date = now()->toDateString();
        $this->recipient_user_id = Auth::id();
    }

    public function render()
    {
        $externalDateShamsi = $this->external_date
            ? jdate($this->external_date)->format('Y/m/d')
            : null;

        return view('livewire.admin.letter.register-incoming-letter', [
            'externalDateShamsi' => $externalDateShamsi,
        ])->layout('admin.master');
    }
}
