<?php

namespace App\Livewire\Admin\Letter\Template;

use App\Models\LetterTemplate;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class TemplateManager extends Component
{
    use WithPagination;

    public $search = '';
    public $sortField = 'created_at';
    public $sortDirection = 'desc';
    public $showModal = false;

    // Form fields
    public $templateId = null;
    public $name = '';
    public $content = '';
    public $description = '';
    public $is_public = true;
    public $type = 'Formal';

    public $templateTypes = [
        'Formal' => 'رسمی',
        'Introduction' => 'معرفی',
        'Memo' => 'یادداشت',
        'Contract' => 'قرارداد',
        'Internal' => 'داخلی',
        'HR' => 'منابع انسانی'
    ];

    public $templateTypeColors = [
        'Formal' => '#3b82f6',
        'Introduction' => '#10b981',
        'Memo' => '#f59e0b',
        'Contract' => '#ef4444',
        'Internal' => '#6b7280',
        'HR' => '#8b5cf6'
    ];

    protected function rules()
    {
        return [
            'name' => [
                'required',
                'string',
                'max:150',
                Rule::unique('letter_templates', 'name')->ignore($this->templateId)
            ],
            'content' => 'required|string|min:10',
            'type' => 'required|string|in:Formal,Introduction,Memo,Contract,Internal,HR',
            'description' => 'nullable|string|max:500',
            'is_public' => 'boolean'
        ];
    }

    protected $queryString = [
        'search' => ['except' => ''],
        'sortField' => ['except' => 'created_at'],
        'sortDirection' => ['except' => 'desc']
    ];

    public function createTemplate()
    {
        $this->resetForm();
        $this->showModal = true;
        $this->dispatch('open-modal');
    }

    public function editTemplate($id)
    {
        $template = LetterTemplate::findOrFail($id);

        if (!$this->canModifyTemplate($template)) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'شما اجازه ویرایش این قالب را ندارید.'
            ]);
            return;
        }

        $this->templateId = $template->id;
        $this->name = $template->name;
        $this->content = $template->content;
        $this->description = $template->description;
        $this->is_public = $template->is_public;
        $this->type = $template->type;

        $this->showModal = true;
        $this->dispatch('open-modal', content: $this->content);
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->resetForm();
        $this->dispatch('close-modal');
    }

    public function saveTemplate()
    {
        $this->validate();

        try {
            $data = [
                'name' => $this->name,
                'content' => $this->content,
                'description' => $this->description,
                'is_public' => $this->is_public,
                'type' => $this->type,
            ];

            if ($this->templateId) {
                $template = LetterTemplate::findOrFail($this->templateId);

                if (!$this->canModifyTemplate($template)) {
                    throw new \Exception('شما اجازه ویرایش این قالب را ندارید.');
                }

                $template->update($data);
                $message = 'قالب با موفقیت به‌روزرسانی شد.';
            } else {
                LetterTemplate::create(array_merge($data, [
                    'user_id' => Auth::id()
                ]));
                $message = 'قالب جدید با موفقیت ایجاد شد.';
            }

            $this->closeModal();
            $this->dispatch('show-toast', [
                'type' => 'success',
                'message' => $message
            ]);

        } catch (\Exception $e) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'خطا: ' . $e->getMessage()
            ]);
        }
    }

    public function deleteTemplate($id)
    {
        try {
            $template = LetterTemplate::findOrFail($id);

            if (!$this->canModifyTemplate($template)) {
                throw new \Exception('شما اجازه حذف این قالب را ندارید.');
            }

            $templateName = $template->name;
            $template->delete();

            $this->dispatch('show-toast', [
                'type' => 'success',
                'message' => "قالب \"{$templateName}\" با موفقیت حذف شد."
            ]);

        } catch (\Exception $e) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'خطا در حذف قالب: ' . $e->getMessage()
            ]);
        }
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    private function resetForm()
    {
        $this->reset([
            'templateId', 'name', 'content', 'description',
            'is_public', 'type'
        ]);
        $this->resetErrorBag();
    }

    private function canModifyTemplate(LetterTemplate $template): bool
    {
        $user = Auth::user();

        if ($user && method_exists($user, 'isAdmin') && $user->isAdmin()) {
            return true;
        }

        return $template->user_id === $user->id;
    }

    public function render()
    {
        $templates = LetterTemplate::query()
            ->where(function($query) {
                $query->where('is_public', true)
                    ->orWhere('user_id', Auth::id());
            })
            ->when($this->search, function ($query) {
                $query->where(function($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                        ->orWhere('content', 'like', '%' . $this->search . '%')
                        ->orWhere('type', 'like', '%' . $this->search . '%')
                        ->orWhere('description', 'like', '%' . $this->search . '%');
                });
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate(12);

        return view('livewire.admin.letter.template.template-manager', [
            'templates' => $templates,
        ])->layout('admin.master');
    }
}
