<?php

namespace App\Livewire\Admin\Letter;

use App\Models\Letter;
use App\Models\LetterEvent;
use App\Models\Referral;
use App\Models\User;
use App\Models\Project; // ✨ اضافه شد
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Attributes\On;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;
use Morilog\Jalali\Jalalian;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

#[Layout('admin.master')]
class UserLetters extends Component
{
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    #[Url(except: '')]
    public string $searchLetter = '';

    #[Url(except: '')]
    public string $searchCode = '';

    #[Url(except: '')]
    public string $filterStatus = '';

    #[Url(except: '')]
    public string $filterProject = ''; // ✨ جدید: فیلتر پروژه

    #[Url(except: false)]
    public bool $filterUrgent = false; // ✨ جدید: فیلتر فوری

    #[Url(except: '')]
    public string $filterFromDate = '';

    #[Url(except: '')]
    public string $filterToDate = '';

    #[Url]
    public int $perPage = 10;

    public bool $showPreviewModal = false;
    public bool $showSendPurposeModal = false;
    public bool $showReferralTreeModal = false;

    // داده‌های Modal
    public ?Letter $previewLetter = null;
    public ?int $previewLetterId = null;
    public array $referralTree = [];
    public ?Collection $historyTimeline = null;
    public ?int $letter_id = null;

    // ارجاعات
    public array $selectedUsers = [];
    public array $selectedPurposes = [];
    public string $searchUser = '';

    public Collection $users;

    protected array $notificationsToSend = [];

    // ------------------------------------------------------------
    // Lifecycle & Hooks
    // ------------------------------------------------------------

    public function mount(): void
    {
        $this->users = collect();
        $this->loadUsers();
    }

    #[On('refreshLetters')]
    public function refreshLetters(): void
    {
        $this->resetPage();
    }

    #[On('setFilterFromDate')]
    public function setFilterFromDate(string $date): void
    {
        $this->filterFromDate = $date;
    }

    #[On('setFilterToDate')]
    public function setFilterToDate(string $date): void
    {
        $this->filterToDate = $date;
    }

    // ✨ متد جدید برای دریافت تأیید از Blade (برای بایگانی)
    #[On('confirmed')]
    public function executeConfirmedAction($method, ...$params)
    {
        if (method_exists($this, $method)) {
            $this->{$method}(...$params);
        }
    }

    #[On('filters-reset')]
    public function filtersReset(): void
    {
        // در صورت نیاز به اجرای جاوااسکریپت پس از ریست کامل
    }

    #[On('onboarding-dismissed')]
    public function markOnboardingAsSeen(): void
    {
        session()->put('onboarding_shown', true);
    }

    // ------------------------------------------------------------
    // Filters & Search
    // ------------------------------------------------------------

    public function updating(string $property): void
    {
        if (in_array($property, ['searchLetter', 'searchCode', 'filterStatus', 'filterProject', 'filterUrgent', 'perPage'])) {
            $this->resetPage();
        }
    }

    public function resetFilters(): void
    {
        $this->searchLetter = '';
        $this->searchCode = '';
        $this->filterStatus = '';
        $this->filterFromDate = '';
        $this->filterToDate = '';
        $this->filterProject = ''; // ✨ ریست فیلتر پروژه
        $this->filterUrgent = false; // ✨ ریست فیلتر فوری
        $this->resetPage();
        $this->dispatch('filters-reset');
    }


    // ------------------------------------------------------------
    // Helpers & Formatting
    // ------------------------------------------------------------

    public function formatShamsiDate(mixed $date, string $format = 'Y/m/d H:i'): string
    {
        if (!$date) return '—';
        try {
            $carbonDate = ($date instanceof \Carbon\Carbon) ? $date : new \Carbon\Carbon($date);
            return Jalalian::fromCarbon($carbonDate)->format($format);
        } catch (\Exception $e) {
            return $date instanceof \Carbon\Carbon ? $date->format('Y-m-d H:i') : (string) $date;
        }
    }

    public function getUserProfilePhotoUrl(mixed $user): string
    {
        $name = is_object($user) && isset($user->name) ? $user->name : 'Unknown';
        $initial = mb_substr($name, 0, 1);
        $default = "https://ui-avatars.com/api/?name={$initial}&background=EBF4FF&color=2563EB&size=128";

        if (!is_object($user)) return $default;

        if (isset($user->image) && $user->image) {
            if (str_starts_with($user->image, 'http://') || str_starts_with($user->image, 'https://')) {
                return $user->image;
            } elseif (str_starts_with($user->image, '/')) {
                return url($user->image);
            } else {
                $path = ltrim($user->image, '/');
                if (\Storage::disk('public')->exists($path)) {
                    return \Storage::url($path);
                }
            }
        }

        return $default;
    }

    // ------------------------------------------------------------
    // User & Smart Suggestions
    // ------------------------------------------------------------

    public function loadUsers(): void
    {
        $topUserIds = Referral::whereHas('letter', fn($q) => $q->where('sender_id', Auth::id()))
            ->select('user_id')
            ->groupBy('user_id')
            ->orderByRaw('COUNT(*) DESC')
            ->limit(3)
            ->pluck('user_id');

        $query = User::where('is_active', true)
            ->where('id', '!=', Auth::id());

        if ($this->searchUser) {
            $query->where(function ($q) {
                $q->where('name', 'like', "%{$this->searchUser}%")
                    ->orWhere('email', 'like', "%{$this->searchUser}%");
            });
        }

        if ($topUserIds->isNotEmpty()) {
            $query->orderByRaw('FIELD(id, ' . $topUserIds->implode(',') . ') DESC');
        }

        $this->users = $query
            ->orderBy('name')
            ->get()
            ->map(function (User $user) use ($topUserIds) {
                $initial = mb_substr($user->name, 0, 1);
                return (object)[
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'initial' => $initial,
                    'photo_url' => $this->getUserProfilePhotoUrl($user),
                    'is_recommended' => $topUserIds->contains($user->id),
                ];
            });
    }

    public function updatedSearchUser(): void
    {
        $this->loadUsers();
    }

    // ✨ لیست پروژه‌های مرتبط با کاربر (برای فیلتر)
    #[Computed]
    public function userProjects(): Collection
    {
        $userId = Auth::id();
        return Project::whereHas('members', function ($q) use ($userId) {
            $q->where('user_id', $userId);
        })->orWhereHas('letters', function ($q) use ($userId) {
            $q->where('sender_id', $userId);
        })->select('id', 'name')->distinct()->get();
    }


    // ------------------------------------------------------------
    // Cloning Logic
    // ------------------------------------------------------------

    public function cloneLetterContent(int $letterId): void
    {
        try {
            $originalLetter = Letter::with('copies', 'project')->where('sender_id', Auth::id())->findOrFail($letterId);

        } catch (ModelNotFoundException $e) {
            $this->dispatch('show-toast', type: 'error', message: '❌ نامه اصلی برای کپی یافت نشد.');
            return;
        } catch (\Exception $e) {
            \Log::error("Failed to retrieve letter for cloning: " . $e->getMessage());
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در دسترسی به نامه.');
            return;
        }

        try {
            $clonedData = [
                'title' => 'کپی از ' . $originalLetter->title,
                'content' => $originalLetter->content,
                'from_who' => $originalLetter->from_who,
                'to_who' => $originalLetter->to_who,
                'classification_id' => $originalLetter->classification_id,
                'background_id' => $originalLetter->background_id,
                'priority' => $originalLetter->priority,
                'type' => $originalLetter->type,
                'confidential' => $originalLetter->confidential,

                'signer_id' => $originalLetter->user_id,
                'signature_position' => $originalLetter->signature_position,

                'ccs' => $originalLetter->copies->map(fn($copy) => [
                    'name' => $copy->name,
                    'position' => $copy->position
                ])->toArray(),
            ];

            session()->put('cloned_letter_data', $clonedData);
            session()->put('cloned_project_id', $originalLetter->project_id);

        } catch (\Exception $e) {
            \Log::error("Cloning data preparation failed: " . $e->getMessage());
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در آماده‌سازی داده‌های کپی.');
            return;
        }

        $this->dispatch('show-toast', type: 'info', message: '💡 محتوای نامه کپی شد و به صفحه ایجاد نامه هدایت می‌شوید.');

        $this->redirect(route('admin.projects.letters.create', [
            'project' => $originalLetter->project_id
        ]), navigate: true);
    }


    // ------------------------------------------------------------
    // Referral Logic - Computed Property for Purposes
    // ------------------------------------------------------------


    #[Computed]
    public function purposesOptions(): array
    {
        return [
            'امضا' => ['icon' => 'fas fa-signature', 'color' => 'success', 'description' => 'تایید نهایی و امضای نامه'],
            'اقدام' => ['icon' => 'fas fa-cogs', 'color' => 'warning', 'description' => 'انجام کار یا اقدام مشخص'],
            'ثبت صادر' => ['icon' => 'fas fa-file-export', 'color' => 'info', 'description' => 'ثبت در دبیرخانه و ارسال به خارج از مجموعه'],
            'مشاوره/نظر' => ['icon' => 'fas fa-comment', 'color' => 'primary', 'description' => 'ارائه نظر، مشاوره یا راهنمایی فنی'],
            'سایر اعمال' => ['icon' => 'fas fa-ellipsis-h', 'color' => 'secondary', 'description' => 'سایر عملیات متفرقه و خاص'],
        ];
    }



    // ------------------------------------------------------------
    // Referral Logic - Handling Selected Users
    // ------------------------------------------------------------

    public function updatedSelectedUsers($value): void
    {
        $value = is_array($value) ? $value : [$value];
        $value = array_map('intval', $value);

        $letterSignerId = null;
        if ($this->letter_id) {
            $letter = Letter::select('user_id')->find($this->letter_id);
            $letterSignerId = $letter->user_id ?? null;
        }

        $old = $this->selectedPurposes;
        $new = [];
        $allPurposeKeys = array_keys($this->purposesOptions);

        foreach ($value as $userId) {
            $userIdStr = (string)$userId;
            $isSigner = ($userId === $letterSignerId);

            if (isset($old[$userIdStr])) {
                $new[$userIdStr] = $old[$userIdStr];
            } else {
                $new[$userIdStr] = array_fill_keys($allPurposeKeys, false);
                $new[$userIdStr]['اقدام'] = true;
            }

            if ($isSigner) {
                // Keep the 'امضا' option available
            } else {
                // Remove 'امضا' option if the user is not the designated signer
                unset($new[$userIdStr]['امضا']);
                if (empty(array_filter($new[$userIdStr]))) {
                    $new[$userIdStr]['اقدام'] = true;
                }
            }
        }

        $this->selectedPurposes = $new;
    }

    public function toggleSelectAllUsers(): void
    {
        $all = $this->users->pluck('id')->toArray();
        $allPurposeKeys = array_keys($this->purposesOptions);

        if (count($this->selectedUsers) === count($all) && $all) {
            $this->selectedUsers = [];
            $this->selectedPurposes = [];
        } else {
            $this->selectedUsers = $all;

            $defaultPurposes = array_fill_keys($allPurposeKeys, false);
            $defaultPurposes['اقدام'] = true;

            $this->selectedPurposes = array_combine(
                array_map('strval', $all),
                array_fill(0, count($all), $defaultPurposes)
            );
        }
    }

    public function submitSendPurpose(): void
    {
        $this->validate(['selectedUsers' => 'required|array|min:1']);

        $hasErrors = false;
        $letterSignerId = null;

        if ($this->letter_id) {
            $letter = Letter::select('user_id')->find($this->letter_id);
            $letterSignerId = $letter->user_id ?? null;
        }

        // اعتبارسنجی تکمیل اهداف
        foreach ($this->selectedUsers as $userId) {
            $userIdStr = (string)$userId;
            if (empty(array_filter($this->selectedPurposes[$userIdStr] ?? []))) {
                $userName = $this->users->firstWhere('id', $userId)?->name ?? 'کاربر ناشناس';
                $this->addError("selectedPurposes.{$userIdStr}", "برای «{$userName}» حداقل یک هدف انتخاب کنید.");
                $hasErrors = true;
            }
        }

        if ($hasErrors) {
            $this->dispatch('show-toast', type: 'error', message: '❌ لطفاً هدف‌های ارجاع را تکمیل کنید.');
            return;
        }

        try {
            $letter = Letter::with(['sender', 'project', 'user'])->findOrFail($this->letter_id);
        } catch (ModelNotFoundException $e) {
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا: نامه مورد نظر یافت نشد.');
            \Log::error('Letter not found for referral', ['letter_id' => $this->letter_id]);
            return;
        }

        $this->notificationsToSend = [];

        DB::beginTransaction();
        try {
            $referralData = [];
            $now = now();
            $currentUserId = Auth::id();

            foreach ($this->selectedUsers as $userId) {
                $user = User::find($userId);
                if (!$user) continue;

                $userIdStr = (string)$userId;
                $selectedPurposes = array_keys(array_filter($this->selectedPurposes[$userIdStr]));

                // حذف هوشمند "امضا" از غیرامضاکنندگان در لحظه ثبت نهایی
                if ($userId !== $letterSignerId && in_array('امضا', $selectedPurposes)) {
                    $selectedPurposes = array_diff($selectedPurposes, ['امضا']);
                }

                if (empty($selectedPurposes)) continue;


                foreach ($selectedPurposes as $purpose) {

                    $referralType = match ($purpose) {
                        'امضا' => Referral::TYPE_APPROVAL,
                        'ثبت صادر' => Referral::TYPE_INFORMATION,
                        'اقدام' => Referral::TYPE_ACTION,
                        'مشاوره/نظر' => Referral::TYPE_CONSULTATION,
                        'سایر اعمال' => Referral::TYPE_OTHER,
                        default => Referral::TYPE_ACTION
                    };

                    $referralData[] = [
                        'letter_id' => $this->letter_id,
                        'user_id' => $userId,
                        'type' => $referralType,
                        'status' => Referral::STATUS_PENDING,
                        'created_by' => $currentUserId,
                        'parent_id' => null,
                        'created_at' => $now,
                        'updated_at' => $now,
                    ];

                    if (class_exists(LetterEvent::class)) {
                        \App\Models\LetterEvent::create([
                            'letter_id' => $this->letter_id,
                            'user_id' => $currentUserId,
                            'type' => 'referral_sent',
                            'description' => "نامه «{$letter->title}» به کاربر «{$user->name}» جهت «{$purpose}» ارجاع داده شد.",
                        ]);
                    }

                    if (class_exists(\App\Notifications\LetterReferralSent::class)) {
                        $this->notificationsToSend[] = [
                            'user' => $user,
                            'notification' => new \App\Notifications\LetterReferralSent(
                                letterId: $this->letter_id,
                                senderId: Auth::id(),
                                purpose: $purpose
                            )
                        ];
                    }
                }
            }

            if (!empty($referralData)) {
                DB::table('referrals')->insert($referralData);
            }

            if ($letter->status === Letter::STATUS_DRAFT) {
                $letter->update(['status' => Letter::STATUS_PENDING]);
            }

            DB::commit();

            $this->sendQueuedNotificationsAfterCommit();

            $this->dispatch('refreshLetters');
            $this->resetSendPurposeData();
            $this->showSendPurposeModal = false;
            $this->dispatch('show-toast', type: 'success', message: '✅ ارجاعات با موفقیت ارسال شدند.');

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Referral submission failed', ['error' => $e->getMessage()]);
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در ثبت ارجاعات. لطفاً مجدداً تلاش کنید.');
        }
    }

    // --- New Quick Action Method ---
    public function quickSendForSign(int $letterId): void
    {
        try {
            $letter = Letter::where('sender_id', Auth::id())
                ->where('status', Letter::STATUS_DRAFT)
                ->findOrFail($letterId);

            if (empty($letter->user_id)) {
                $this->dispatch('show-toast', type: 'error', message: '❌ ابتدا باید امضاکننده نهایی در نامه مشخص شده باشد.');
                return;
            }

            // ✨ جلوگیری از ارجاع به خود
            if ($letter->user_id === Auth::id()) {
                $this->dispatch('show-toast', type: 'error', message: '❌ شما نمی‌توانید نامه خود را برای امضای خودتان ارجاع دهید.');
                return;
            }

            $signerUser = User::find($letter->user_id);
            if (!$signerUser) {
                $this->dispatch('show-toast', type: 'error', message: '❌ امضاکننده نامه یافت نشد.');
                return;
            }

            DB::beginTransaction();

            Referral::create([
                'letter_id' => $letterId,
                'user_id' => $letter->user_id,
                'type' => Referral::TYPE_APPROVAL,
                'status' => Referral::STATUS_PENDING,
                'created_by' => Auth::id(),
                'parent_id' => null,
            ]);

            $letter->update(['status' => Letter::STATUS_PENDING]);

            // Add Event
            if (class_exists(LetterEvent::class)) {
                \App\Models\LetterEvent::create([
                    'letter_id' => $letter->id,
                    'user_id' => Auth::id(),
                    'type' => 'referral_sent',
                    'description' => "نامه با قابلیت ارسال سریع برای امضاکننده ({$signerUser->name}) ارجاع داده شد.",
                ]);
            }

            // Add Notification to queue (assuming LetterReferralSent accepts purpose)
            if (class_exists(\App\Notifications\LetterReferralSent::class)) {
                $this->notificationsToSend[] = [
                    'user' => $signerUser,
                    'notification' => new \App\Notifications\LetterReferralSent(
                        letterId: $letter->id,
                        senderId: Auth::id(),
                        purpose: 'امضا'
                    )
                ];
            }

            DB::commit();
            $this->sendQueuedNotificationsAfterCommit();

            $this->dispatch('refreshLetters');
            $this->dispatch('show-toast', type: 'success', message: '🚀 نامه با موفقیت برای امضا ارجاع داده شد.');

        } catch (ModelNotFoundException $e) {
            $this->dispatch('show-toast', type: 'warning', message: 'نامه فقط در وضعیت پیش‌نویس قابل ارسال سریع است.');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error("Quick Send failed: " . $e->getMessage());
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در ارسال سریع نامه.');
        }
    }
    // --- End New Quick Action Method ---

    // ✨ متدهای بایگانی
    public function archiveLetter(int $letterId): void
    {
        try {
            $letter = Letter::where('sender_id', Auth::id())->findOrFail($letterId);

            if ($letter->archive()) {
                // ایجاد رویداد
                if (class_exists(LetterEvent::class)) {
                    \App\Models\LetterEvent::create([
                        'letter_id' => $letter->id,
                        'user_id' => Auth::id(),
                        'type' => 'archived',
                        'description' => "نامه توسط فرستنده بایگانی شد.",
                    ]);
                }

                $this->dispatch('refreshLetters');
                $this->dispatch('show-toast', type: 'success', message: '✅ نامه با موفقیت بایگانی شد.');
            } else {
                $this->dispatch('show-toast', type: 'error', message: '❌ نامه در وضعیت فعلی قابل بایگانی نیست (باید درفت، تایید یا رد شده باشد).');
            }

        } catch (ModelNotFoundException $e) {
            $this->dispatch('show-toast', type: 'error', message: '❌ نامه یافت نشد.');
        } catch (\Exception $e) {
            \Log::error("Archiving failed: " . $e->getMessage());
            $this->dispatch('show-toast', type: 'error', message: '❌ خطای سیستمی در بایگانی.');
        }
    }

    public function confirmArchive(int $letterId): void
    {
        $this->dispatch('show-confirmation', [
            'title' => 'بایگانی نامه',
            'text' => 'آیا مطمئنید که می‌خواهید این نامه را بایگانی کنید؟ نامه‌های بایگانی شده از لیست اصلی حذف می‌شوند.',
            'icon' => 'warning',
            'confirmButtonText' => 'بایگانی کن',
            'method' => 'archiveLetter',
            'params' => $letterId,
        ]);
    }
    // --- End Archiving Methods ---

    public function openSendPurposeModal(int $letterId): void
    {
        $this->letter_id = $letterId;
        $this->showSendPurposeModal = true;
        $this->resetSendPurposeData();
    }

    protected function sendQueuedNotificationsAfterCommit(): void
    {
        if (empty($this->notificationsToSend)) {
            return;
        }

        DB::afterCommit(function () {
            foreach ($this->notificationsToSend as $item) {
                Notification::send($item['user'], $item['notification']);
            }
            $this->dispatch('notificationSent');
        });
    }


    public function sendReferralReminder(int $letterId): void
    {
        try {
            $letter = Letter::findOrFail($letterId);
            $pendingReferrals = Referral::where('letter_id', $letterId)
                ->where('status', Referral::STATUS_PENDING)
                ->with('user')
                ->get();

            if ($pendingReferrals->isEmpty()) {
                $this->dispatch('show-toast', type: 'info', message: '💡 هیچ ارجاع فعالی برای این نامه وجود ندارد.');
                return;
            }

            $recipients = $pendingReferrals->pluck('user');
            $senderName = Auth::user()?->name ?? 'سیستم';
            $count = $recipients->count();

            if (class_exists(\App\Notifications\LetterReminder::class)) {

                $notification = new \App\Notifications\LetterReminder($letter->id, $senderName);

                \DB::transaction(function () use ($recipients, $notification, $letter, $count) {
                    Notification::send($recipients, $notification);

                    if (class_exists(LetterEvent::class)) {
                        \App\Models\LetterEvent::create([
                            'letter_id' => $letter->id,
                            'user_id' => Auth::id(),
                            'type' => 'reminder_sent',
                            'description' => "یادآوری ارجاع به {$count} نفر ارسال شد.",
                        ]);
                    }
                });

                $this->dispatch('show-toast', type: 'success', message: "🔔 یادآوری با موفقیت برای {$count} نفر ارسال شد.");

            } else {
                $this->dispatch('show-toast', type: 'warning', message: "🔔 یادآوری برای {$count} نفر ارسال شد (نوتیفیکیشن محلی).");
            }

        } catch (ModelNotFoundException $e) {
            $this->dispatch('show-toast', type: 'error', message: '❌ نامه مورد نظر یافت نشد.');
        } catch (\Exception $e) {
            \Log::error("Failed to send reminder: " . $e->getMessage());
            $this->dispatch('show-toast', type: 'error', message: '❌ خطای سیستمی در ارسال یادآوری.');
        }
    }


    // ------------------------------------------------------------
    // Modal & Tree Logic
    // ------------------------------------------------------------

    public function showPreview(int $letterId): void
    {
        try {
            $this->previewLetter = Letter::with(['user', 'referrals.user', 'receivers'])->findOrFail($letterId);
            $this->previewLetterId = $letterId;
            $this->showPreviewModal = true;
        } catch (\Exception $e) {
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در بارگذاری پیش‌نمایش');
        }
    }

    public function closePreviewModal(): void
    {
        $this->previewLetter = null;
        $this->previewLetterId = null;
        $this->showPreviewModal = false;
    }

    public function openReferralTreeModal(int $letterId): void
    {
        try {
            $letter = Letter::with(['referrals.user', 'events.user'])->findOrFail($letterId);
            $this->referralTree = $this->buildReferralTree($letter->referrals->sortBy('created_at'));
            $this->historyTimeline = $letter->events->sortByDesc('created_at');
            $this->showReferralTreeModal = true;
        } catch (\Exception $e) {
            \Log::error('Referral tree load error', ['msg' => $e->getMessage()]);
            $this->dispatch('show-toast', type: 'error', message: '❌ خطا در بارگذاری درخت ارجاع');
        }
    }

    protected function buildReferralTree(Collection $referrals): array
    {
        $grouped = $referrals->groupBy('parent_id');

        $build = function ($parentId = null) use (&$build, $grouped) {
            return $grouped->get($parentId, collect())->map(function ($referral) use (&$build) {
                return [
                    'referral' => $referral,
                    'children' => $build($referral->id),
                ];
            })->all();
        };

        return $build();
    }

    public function resetSendPurposeData(): void
    {
        $this->selectedUsers = [];
        $this->selectedPurposes = [];
        $this->searchUser = '';
        $this->resetErrorBag();
        $this->loadUsers();
    }

    public function closeSendPurposeModal(): void
    {
        $this->showSendPurposeModal = false;
        $this->resetSendPurposeData();
        $this->letter_id = null;
    }

    public function closeReferralTreeModal(): void
    {
        $this->showReferralTreeModal = false;
        $this->referralTree = [];
        $this->historyTimeline = collect();
    }

    // ------------------------------------------------------------
    // Queries
    // ------------------------------------------------------------

    #[Computed]
    public function letters()
    {
        return Letter::with([
            'user:id,name,image',
            'referrals:id,letter_id,user_id,status,type,created_at,parent_id',
            'referrals.user:id,name,image',
            'latestActiveReferral.user:id,name,image,department_id', // ✨ جدید
            'latestReferral.user:id,name,image,department_id', // ✨ جدید
            'latestActiveReferral.creator:id,name', // ✨ جدید
            'receivers' // ✨ برای محاسبه ReadPercentage
        ])
            ->withCount('referrals')
            ->where('sender_id', Auth::id())
            ->notArchived() // ✨ حذف نامه‌های بایگانی شده
            ->when($this->searchLetter, fn($q) => $q->where(function ($sub) {
                $sub->where('title', 'like', "%{$this->searchLetter}%")
                    ->orWhere('content', 'like', "%{$this->searchLetter}%");
            }))
            ->when($this->searchCode, fn($q) => $q->where(function ($sub) {
                $sub->where('code', 'like', "%{$this->searchCode}%")
                    ->orWhere('id', $this->searchCode);
            }))
            ->when($this->filterStatus, fn($q) => $q->where('status', $this->filterStatus))
            ->when($this->filterProject, fn($q) => $q->where('project_id', $this->filterProject)) // ✨ اعمال فیلتر پروژه
            ->when($this->filterUrgent, fn($q) => $q->where('priority', Letter::PRIORITY_URGENT)) // ✨ اعمال فیلتر فوری
            ->when($this->filterFromDate, fn($q) => $q->whereDate('created_at', '>=', $this->filterFromDate))
            ->when($this->filterToDate, fn($q) => $q->whereDate('created_at', '<=', $this->filterToDate))
            ->orderByDesc('created_at')
            ->paginate($this->perPage);
    }

    #[Computed]
    public function statusCounts(): \Illuminate\Support\Collection
    {
        return Letter::where('sender_id', Auth::id())
            ->notArchived() // ✨ شمارش وضعیت‌ها فقط در لیست اصلی
            ->selectRaw("status, COUNT(*) as count")
            ->groupBy('status')
            ->pluck('count', 'status');
    }

    public function render()
    {
        return view('livewire.admin.letter.user-letters', [
            'letters' => $this->letters,
            'statusCounts' => $this->statusCounts,
            'users' => $this->users,
            'userProjects' => $this->userProjects, // ✨ ارسال لیست پروژه‌ها به Blade
        ]);
    }
}
