<?php

namespace App\Livewire\Admin\MDR;

use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\WithPagination;
use App\Models\Project;
use App\Models\MasterDocument;
use App\Models\DocumentFile;
use App\Models\DocumentHistory;
use App\Models\DocumentCategory;
use App\Models\ProjectPoiWeight;
use App\Models\Discipline;
use App\Models\DocumentRevision;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\Mdr\MdrExport;
use ZipArchive;

#[Layout('admin.master')]
class ProjectMdrList extends Component
{
    use WithFileUploads, WithPagination;

    protected $paginationTheme = 'bootstrap';

    protected $queryString = [
        'search' => ['except' => ''],
        'filter_status' => ['except' => ''],
        'filter_discipline_id' => ['except' => ''],
        'filter_transmittal_status' => ['except' => ''],
        'perPage' => ['except' => 20],
        'sortField' => ['except' => 'id'],
        'sortDir' => ['except' => 'desc'],
    ];

    public $project;
    public $categories = [];
    public $disciplines = [];
    public $search = '';
    public $filter_status = '';
    public $filter_discipline_id = '';
    public $filter_transmittal_status = '';
    public $perPage = 20;
    public $sortField = 'id';
    public $sortDir = 'desc';
    public $weight = 1.00;

    // فرم
    public $editId = null, $document_number, $title, $category_id, $new_category, $discipline_id, $new_discipline, $document_type, $status;
    public $files = [];
    public $new_poi_type = 'IFA'; // مقدار پیش‌فرض برای POI جدید

    // انتخاب‌ها
    public $selectAll = false, $selectedDocs = [];

    // رویژن
    public $showRevisionModal = false, $revisions = [], $revisionDocId = null;
    public $new_revision_number, $new_revision_remark, $new_revision_poi_type, $new_revision_files = [];

    // برای افزودن فایل به رویژن خاص
    public $addFileToRevisionId = null;
    public $new_files_for_revision = [];

    // تاریخچه و ترنسمیتال
    public $selectedDocNumber = '', $selectedTransmittals = [];

    // پیش‌نمایش فایل
    public $previewFileUrl = null;

    protected function rules(): array
    {
        return [
            'document_number' => 'required|string|max:100',
            'title'           => 'required|string|max:255',
            'category_id'     => 'required|exists:document_categories,id',
            'discipline_id'   => 'required|exists:disciplines,id',
            'document_type'   => 'nullable|string|max:100',
            'status'          => 'nullable|string|max:50',
            'files'           => 'nullable|array',
            'weight'          => 'nullable|numeric|min:0.01|max:1000',
            'files.*'         => 'nullable|file|max:500480|mimes:pdf,doc,docx,xls,xlsx,zip,png,jpg,jpeg,dwg,dxf,sldprt,rar,7z',
        ];
    }

    protected $messages = [
        'title.required' => 'عنوان الزامی است.',
        'category_id.required' => 'کلاس را انتخاب کنید.',
        'discipline_id.required' => 'دیسپلین را انتخاب کنید.',
    ];

    public function mount($project)
    {
        $this->project = Project::with('poiWeights')->findOrFail($project);
        $this->categories = DocumentCategory::all();
        $this->disciplines = Discipline::where('project_id', $this->project->id)->get();
    }

    protected function baseQuery()
    {
        return $this->project->masterDocuments()
            ->when($this->search, function ($q) {
                $s = trim($this->search);
                $q->where(function($qq) use ($s){
                    $qq->where('document_number','like',"%{$s}%")
                        ->orWhere('title','like',"%{$s}%");
                });
            })
            ->when($this->filter_status, fn($q)=>$q->where('status',$this->filter_status))
            ->when($this->filter_discipline_id, fn($q)=>$q->where('discipline_id',$this->filter_discipline_id))
            ->when($this->filter_transmittal_status, function ($q) {
                $ts = $this->filter_transmittal_status;
                if ($ts === 'بدون ارسال') $q->whereDoesntHave('transmittalDocuments');
                else $q->whereHas('transmittalDocuments.transmittal', fn($qq)=>$qq->where('status',$ts));
            });
    }

    protected function applySort($q)
    {
        $allowed = ['id','document_number','title','document_type'];
        if ($this->sortField === 'category') {
            return $q->leftJoin('document_categories','master_documents.category_id','=','document_categories.id')
                ->select('master_documents.*')
                ->orderBy('document_categories.name', $this->sortDir);
        }
        if (!in_array($this->sortField, $allowed)) {
            $this->sortField = 'id'; $this->sortDir = 'desc';
        }
        return $q->orderBy($this->sortField, $this->sortDir);
    }

    public function getMasterDocumentsProperty()
    {
        $q = $this->baseQuery()
            ->with([
                'project.poiWeights:id,project_id,poi_type,weight_percent', // بهینه‌سازی لود
                'transmittalDocuments.transmittal:id,transmittal_number,sent_at,status',
                'category:id,name',
                'discipline:id,name',
                'revisions' => fn($r) => $r->select('id', 'master_document_id', 'revision', 'poi_type', 'date', 'remark')->latest('date')->take(1)->with(['files' => fn($f) => $f->select('id', 'attachable_id', 'file_path', 'mime_type', 'file_name')->latest()->take(1)]),
            ]);
        $q = $this->applySort($q);
        return $q->paginate($this->perPage);
    }

    public function getStatsProperty(): array
    {
        $q = $this->baseQuery();
        $total = cache()->remember('mdr_total_' . $this->project->id, 300, fn() => (clone $q)->count());
        return [
            'total' => $total,
            'sent' => cache()->remember('mdr_sent_' . $this->project->id, 300, fn() => (clone $q)->where('status','ارسال شده')->count()),
            'approved' => cache()->remember('mdr_approved_' . $this->project->id, 300, fn() => (clone $q)->where('status','تایید شده')->count()),
            'rejected' => cache()->remember('mdr_rejected_' . $this->project->id, 300, fn() => (clone $q)->where('status','رد شده')->count()),
        ];
    }

    public function sortBy(string $field): void
    {
        if ($this->sortField === $field) {
            $this->sortDir = $this->sortDir === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDir = 'asc';
        }
        $this->resetPage();
    }

    public function updated($name): void
    {
        if (in_array($name, ['search','filter_status','filter_discipline_id','filter_transmittal_status','perPage'])) {
            $this->resetPage();
            $this->selectedDocs = [];
            $this->selectAll = false;
        }
    }

    public function resetFilters(): void
    {
        $this->search = $this->filter_status = $this->filter_discipline_id = $this->filter_transmittal_status = '';
        $this->resetPage();
    }

    public function updatedSelectAll($checked): void
    {
        if ($checked) {
            $this->selectedDocs = $this->masterDocuments->pluck('id')->map(fn($id)=>(string)$id)->toArray();
        } else {
            $this->selectedDocs = [];
        }
    }

    public function updatedSelectedDocs(): void
    {
        $this->selectAll = count($this->selectedDocs) === $this->masterDocuments->count();
    }

    public function showAddForm()
    {
        $this->resetForm();
        $this->category_id = $this->categories->first()?->id ?? '';
        $this->discipline_id = $this->disciplines->first()?->id ?? '';
        $this->dispatch('open-form-modal');
    }

    public function showEditForm($id)
    {
        try {
            $doc = MasterDocument::findOrFail($id);
            $this->editId = $doc->id;
            $this->document_number = $doc->document_number;
            $this->title = $doc->title;
            $this->weight = $doc->weight;
            $this->category_id = $doc->category_id;
            $this->discipline_id = $doc->discipline_id;
            $this->document_type = $doc->document_type;
            $this->status = $doc->status;
            $this->files = [];
            $this->dispatch('open-form-modal');
        } catch (\Throwable $e) {
            Log::error('showEditForm error', ['e'=>$e]);
            $this->dispatch('alert', message: 'خطا در بارگذاری مدرک.', type: 'error');
        }
    }

    public function closeForm()
    {
        $this->resetForm();
        $this->dispatch('close-form-modal');
    }

    public function resetForm(): void
    {
        $this->document_number = '';
        $this->title = '';
        $this->category_id = '';
        $this->new_category = '';
        $this->discipline_id = '';
        $this->new_discipline = '';
        $this->document_type = '';
        $this->status = '';
        $this->files = [];
        $this->weight = 1.00;
        $this->editId = null;
    }

    public function addCategory()
    {
        $this->validate(['new_category'=>'required|string|max:100|unique:document_categories,name']);
        try {
            $cat = DocumentCategory::create(['name'=>$this->new_category]);
            $this->categories = DocumentCategory::all(); // Refresh
            $this->category_id = $cat->id;
            $this->new_category = '';
            $this->dispatch('alert', message: __('messages.category_added'), type: 'success');
        } catch (\Throwable $e) {
            Log::error('addCategory error',['e'=>$e]);
            $this->dispatch('alert', message: 'خطا در افزودن کلاس.', type: 'error');
        }
    }

    public function addDiscipline()
    {
        $this->validate([
            'new_discipline' => 'required|string|max:100|unique:disciplines,name,NULL,id,project_id,' . $this->project->id,
        ]);
        try {
            $d = Discipline::create(['name'=>$this->new_discipline,'project_id'=>$this->project->id]);
            $this->disciplines = Discipline::where('project_id', $this->project->id)->get(); // Refresh
            $this->discipline_id = $d->id;
            $this->new_discipline = '';
            $this->dispatch('alert', message: __('messages.discipline_added'), type: 'success');
        } catch (\Throwable $e) {
            Log::error('addDiscipline error',['e'=>$e]);
            $this->dispatch('alert', message: 'خطا در افزودن دیسپلین.', type: 'error');
        }
    }

    private function getNextRevisionNumber(MasterDocument $doc): string
    {
        $last = $doc->revisions()->orderByDesc('date')->first();
        if (!$last) return 'Rev.01';
        $num = intval(substr($last->revision, -2)) + 1;
        return 'Rev.' . str_pad($num, 2, '0', STR_PAD_LEFT);
    }

    public function saveDocument()
    {
        $this->validate();

        DB::beginTransaction();
        try {
            $data = [
                'project_id'      => $this->project->id,
                'document_number' => trim($this->document_number),
                'title'           => trim($this->title),
                'category_id'     => $this->category_id,
                'discipline_id'   => $this->discipline_id,
                'document_type'   => $this->document_type,
                'weight'          => (float)($this->weight ?: 1.00),
                'status'          => $this->status ?: 'در حال تهیه',
            ];

            if ($this->editId) {
                $doc = MasterDocument::lockForUpdate()->findOrFail($this->editId);
                $old = $doc->toArray();
                $doc->update($data);
                $action = 'ویرایش';
            } else {
                $doc = MasterDocument::create($data);
                $old = [];
                $action = 'ایجاد';
            }

            if (!empty($this->files) && is_array($this->files)) {
                $revisionNumber = $this->getNextRevisionNumber($doc);

                $revision = $doc->revisions()->create([
                    'revision_type' => $this->editId ? 'اصلاحی' : 'اولیه',
                    'revision'      => $revisionNumber,
                    'status'        => 'در انتظار',
                    'date'          => now(),
                    'remark'        => $this->editId ? 'آپلود فایل جدید' : 'ایجاد اولیه',
                    'poi_type'      => $this->new_poi_type,
                ]);

                foreach ($this->files as $file) {
                    if (!$file->getRealPath()) {
                        continue;
                    }

                    $originalName = $file->getClientOriginalName();
                    $fileSize = $file->getSize();
                    $mimeType = $file->getMimeType();
                    $storedFileName = time() . '_' . $originalName;

                    $path = $file->storeAs('documents/revisions', $storedFileName, 'public');
                    $fullPath = storage_path('app/public/' . $path);

                    DocumentFile::create([
                        'attachable_id'   => $revision->id,
                        'attachable_type' => DocumentRevision::class,
                        'file_path'       => $path,
                        'file_name'       => $originalName,
                        'uploaded_by'     => auth()->id(),
                        'size'            => $fileSize,
                        'mime_type'       => $mimeType,
                        'checksum'        => file_exists($fullPath) ? hash_file('sha256', $fullPath) : null,
                    ]);
                }
                $doc->update(['status' => 'در انتظار']);
            }

            DocumentHistory::create([
                'documentable_id'   => $doc->id,
                'documentable_type' => MasterDocument::class,
                'user_id'           => auth()->id(),
                'action'            => $action,
                'old_values'        => json_encode($old),
                'new_values'        => json_encode($doc->refresh()->toArray()),
                'description'       => "سند توسط " . auth()->user()->name . " $action شد.",
            ]);

            DB::commit();

            $this->dispatch('alert', message: __('messages.document_saved'), type: 'success');
            $this->closeForm();
            $this->reset('files');

        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('saveDocument error: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            $this->dispatch('alert', message: 'خطا در ذخیره سازی.', type: 'error');
        }
    }

    public function updatedFiles()
    {
        try {
            $this->validateOnly('files.*');
        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->addError('files', 'خطای اعتبارسنجی: ' . $e->getMessage());
            Log::error('Validation Error:', $e->errors());
        }
    }

    public function deleteDocument($id)
    {
        DB::beginTransaction();
        try {
            $doc = MasterDocument::lockForUpdate()->findOrFail($id);
            $old = $doc->toArray();
            foreach ($doc->revisions as $rev) {
                foreach ($rev->files as $file) {
                    if (Storage::disk('public')->exists($file->file_path)) {
                        Storage::disk('public')->delete($file->file_path);
                    }
                    $file->delete();
                }
                $rev->delete();
            }
            $doc->delete();

            DocumentHistory::create([
                'documentable_id'=>$id,
                'documentable_type'=>MasterDocument::class,
                'user_id'=>auth()->id(),
                'action'=>'حذف',
                'old_values'=>json_encode($old),
                'new_values'=>null,
                'description'=>'حذف توسط کاربر',
            ]);

            DB::commit();
            $this->dispatch('alert', message: __('messages.document_deleted'), type: 'success');
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('deleteDocument error',['e'=>$e]);
            $this->dispatch('alert', message: 'خطا در حذف مدرک.', type: 'error');
        }
    }

    public function showRevisions($docId)
    {
        try {
            $doc = MasterDocument::findOrFail($docId);
            $this->revisions = $doc->revisions()->with('files')->orderByDesc('date')->get();
            $this->revisionDocId = $docId;
            $this->showRevisionModal = true;
            $this->dispatch('open-revision-modal');
        } catch (\Throwable $e) {
            Log::error('showRevisions error',['e'=>$e]);
            $this->dispatch('alert', message: 'خطا در بارگذاری رویژن‌ها.', type: 'error');
        }
    }

    public function closeRevisionModal()
    {
        $this->showRevisionModal = false;
        $this->revisions = [];
        $this->revisionDocId = null;
        $this->resetRevisionForm();
        $this->addFileToRevisionId = null;
        $this->new_files_for_revision = [];
        $this->dispatch('close-revision-modal');
    }

    private function resetRevisionForm()
    {
        $this->new_revision_number = '';
        $this->new_revision_remark = '';
        $this->new_revision_poi_type = '';
        $this->new_revision_files = [];
    }

    public function addRevision()
    {
        $this->validate([
            'new_revision_number'   => 'required|string|max:20|unique:document_revisions,revision,NULL,id,master_document_id,' . $this->revisionDocId,
            'new_revision_remark'   => 'nullable|string|max:255',
            'new_revision_poi_type' => 'nullable|in:IFC,IFA,AFC,IFI,Rec',
            'new_revision_files'    => 'nullable|array',
            'new_revision_files.*'  => 'nullable|file|max:512000',
        ]);

        DB::beginTransaction();
        try {
            $doc = MasterDocument::lockForUpdate()->findOrFail($this->revisionDocId);

            $revision = $doc->revisions()->create([
                'revision_type' => 'اصلاحی',
                'revision'      => trim($this->new_revision_number),
                'status'        => 'در انتظار',
                'date'          => now(),
                'remark'        => $this->new_revision_remark,
                'poi_type'      => $this->new_revision_poi_type,
            ]);

            if (!empty($this->new_revision_files)) {
                foreach ($this->new_revision_files as $file) {
                    if (!$file->getRealPath()) {
                        throw new \Exception("فایل موقت توسط سیستم پاک شده است. لطفا دوباره تلاش کنید.");
                    }

                    $originalName = $file->getClientOriginalName();
                    $fileSize = $file->getSize();
                    $mimeType = $file->getMimeType();
                    $storedFileName = time() . '_' . $originalName;

                    $path = $file->storeAs('documents/revisions', $storedFileName, 'public');
                    $fullPath = storage_path('app/public/' . $path);

                    DocumentFile::create([
                        'attachable_id'   => $revision->id,
                        'attachable_type' => DocumentRevision::class,
                        'file_path'       => $path,
                        'file_name'       => $originalName,
                        'uploaded_by'     => auth()->id(),
                        'size'            => $fileSize,
                        'mime_type'       => $mimeType,
                        'checksum'        => file_exists($fullPath) ? hash_file('sha256', $fullPath) : null,
                    ]);
                }
                $this->reset('new_revision_files');
            }

            $doc->update(['status' => 'در انتظار']);
            DB::commit();

            $this->revisions = $doc->revisions()->with('files')->orderByDesc('date')->get();
            $this->resetRevisionForm();

            $this->dispatch('alert', message: 'رویژن با موفقیت ثبت شد', type: 'success');
            $this->dispatch('close-revision-modal');

        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('addRevision Error: ' . $e->getMessage());
            $this->dispatch('alert', message: 'خطا: ' . $e->getMessage(), type: 'error');
        }
    }

    public function addFilesToRevision($revisionId)
    {
        $this->validate([
            'new_files_for_revision' => 'required|array|min:1',
            'new_files_for_revision.*' => 'file|max:10240|mimes:pdf,doc,docx,xls,xlsx,zip,png,jpg,jpeg',
        ]);

        DB::beginTransaction();
        try {
            $revision = DocumentRevision::findOrFail($revisionId);
            foreach ($this->new_files_for_revision as $file) {
                if (!$file->getRealPath()) {
                    continue;
                }
                $originalName = $file->getClientOriginalName();
                $fileSize = $file->getSize();
                $mimeType = $file->getMimeType();
                $storedFileName = time() . '_' . $originalName;

                $path = $file->storeAs('documents/revisions', $storedFileName, 'public');
                $fullPath = storage_path('app/public/' . $path);

                DocumentFile::create([
                    'attachable_id'   => $revision->id,
                    'attachable_type' => DocumentRevision::class,
                    'file_path'       => $path,
                    'file_name'       => $originalName,
                    'uploaded_by'     => auth()->id(),
                    'size'            => $fileSize,
                    'mime_type'       => $mimeType,
                    'checksum'        => file_exists($fullPath) ? hash_file('sha256', $fullPath) : null,
                ]);
            }
            DB::commit();
            $this->revisions = MasterDocument::findOrFail($this->revisionDocId)
                ->revisions()->with('files')->orderByDesc('date')->get();
            $this->new_files_for_revision = [];
            $this->addFileToRevisionId = null;
            $this->dispatch('alert', message: 'فایل(ها) با موفقیت اضافه شد.', type: 'success');
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('addFilesToRevision error', ['e' => $e]);
            $this->dispatch('alert', message: 'خطا در افزودن فایل.', type: 'error');
        }
    }

    public function deleteRevision($revisionId)
    {
        DB::beginTransaction();
        try {
            $revision = DocumentRevision::lockForUpdate()->findOrFail($revisionId);
            foreach ($revision->files as $file) {
                if (Storage::disk('public')->exists($file->file_path)) {
                    Storage::disk('public')->delete($file->file_path);
                }
                $file->delete();
            }
            $revision->delete();

            $doc = MasterDocument::lockForUpdate()->findOrFail($this->revisionDocId);
            $lastRev = $doc->revisions()->orderByDesc('date')->first();
            $doc->update(['status' => $lastRev ? $lastRev->status : null]);

            DB::commit();
            $this->revisions = $doc->revisions()->with('files')->orderByDesc('date')->get();
            $this->dispatch('alert', message: __('messages.revision_deleted'), type: 'success');
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('deleteRevision error',['e'=>$e]);
            $this->dispatch('alert', message: 'خطا در حذف رویژن.', type: 'error');
        }
    }

    public function showTransmittalHistory($docId)
    {
        try {
            $doc = MasterDocument::findOrFail($docId);
            $this->selectedDocNumber = $doc->document_number;
            $this->selectedTransmittals = $doc->transmittalDocuments()->with('transmittal.sender','transmittal.receiver')
                ->orderByDesc(\DB::raw('COALESCE(transmittal_id,0)'))
                ->get()
                ->sortByDesc(fn($td)=>$td->transmittal?->sent_at);
            $this->dispatch('open-transmittal-modal');
        } catch (\Throwable $e) {
            Log::error('showTransmittalHistory error',['e'=>$e]);
            $this->dispatch('alert', message: 'خطا در بارگذاری تاریخچه ترنسمیتال‌ها.', type: 'error');
        }
    }

    public function closeTransmittalModal()
    {
        $this->selectedDocNumber = '';
        $this->selectedTransmittals = [];
        $this->dispatch('close-transmittal-modal');
    }

    public function preview($docId)
    {
        try {
            $doc = MasterDocument::findOrFail($docId);
            $latestRevision = $doc->revisions->last();
            $latestFile = $latestRevision?->files->last();
            if ($latestFile && in_array($latestFile->mime_type, ['application/pdf', 'image/jpeg', 'image/png'])) {
                $this->previewFileUrl = asset('storage/' . $latestFile->file_path);
                $this->dispatch('open-preview-modal');
            }
        } catch (\Throwable $e) {
            Log::error('preview error', ['e' => $e]);
            $this->dispatch('alert', message: 'خطا در پیش‌نمایش فایل.', type: 'error');
        }
    }

    public function closePreviewModal()
    {
        $this->previewFileUrl = null;
        $this->dispatch('close-preview-modal');
    }

    public function exportExcel()
    {
        try {
            $filters = [
                'search' => $this->search,
                'status' => $this->filter_status,
                'discipline_id' => $this->filter_discipline_id,
                'transmittal_status' => $this->filter_transmittal_status,
            ];
            $file = 'MDR-'.($this->project->code ?? 'project').'-'.now()->format('Ymd_His').'.xlsx';
            return Excel::download(new MdrExport($this->project->id, $filters), $file);
        } catch (\Throwable $e) {
            Log::error('exportExcel error',['e'=>$e]);
            $this->dispatch('alert', message: 'خطا در خروجی اکسل.', type: 'error');
        }
    }

    public function exportPdf()
    {
        try {
            $docs = $this->baseQuery()
                ->with([
                    'category:id,name',
                    'discipline:id,name',
                    'revisions' => fn($r) => $r->latest()->with('files'),
                    'transmittalDocuments.transmittal'
                ])
                ->orderBy('discipline_id')
                ->orderBy('document_number')
                ->get();
            $stats = $this->stats;
            $filters = [
                'search' => $this->search,
                'status' => $this->filter_status,
                'discipline_id' => $this->filter_discipline_id,
                'transmittal_status' => $this->filter_transmittal_status,
            ];
            $pdf = \PDF::loadView('exports.mdr-pdf', [
                'project'     => $this->project,
                'documents'   => $docs,
                'disciplines' => $this->disciplines,
                'filters'     => $filters,
                'stats'       => $stats,
            ])->setPaper('a4','landscape');
            $name = 'MDR-'.($this->project->code ?? 'project').'-'.now()->format('Ymd_His').'.pdf';
            return $pdf->download($name);
        } catch (\Throwable $e) {
            Log::error('exportPdf error',['e'=>$e]);
            $this->dispatch('alert', message: 'خطا در خروجی PDF.', type: 'error');
        }
    }

    public function downloadSelected()
    {
        try {
            if (!count($this->selectedDocs)) {
                $this->dispatch('alert', message: 'هیچ مدرکی انتخاب نشده است.', type: 'error');
                return;
            }
            $zip = new ZipArchive();
            $fileName = 'documents_'.time().'.zip';
            $zipPath = storage_path('app/public/'.$fileName);
            if ($zip->open($zipPath, ZipArchive::CREATE) === true) {
                MasterDocument::whereIn('id',$this->selectedDocs)->with('revisions.files')->chunk(100, function($docs) use ($zip){
                    foreach ($docs as $doc) {
                        $latestRev = $doc->revisions->last();
                        if ($latestRev) {
                            foreach ($latestRev->files as $file) {
                                $filePath = storage_path('app/public/'.$file->file_path);
                                if (is_file($filePath)) {
                                    $zip->addFile($filePath, $file->file_name ?: basename($filePath));
                                }
                            }
                        }
                    }
                });
                $zip->close();
                return response()->download($zipPath)->deleteFileAfterSend(true);
            }
            $this->dispatch('alert', message: 'خطا در ایجاد فایل ZIP.', type: 'error');
        } catch (\Throwable $e) {
            Log::error('downloadSelected error',['e'=>$e]);
            $this->dispatch('alert', message: 'خطا در دانلود گروهی.', type: 'error');
        }
    }

    public function render()
    {
        return view('livewire.admin.m-d-r.project-mdr-list', [
            'masterDocuments'   => $this->masterDocuments,
            'categories'        => $this->categories,
            'disciplines'       => $this->disciplines,
            'revisions'         => $this->revisions,
            'stats'             => $this->stats,
            'sortField'         => $this->sortField,
            'sortDir'           => $this->sortDir,
            'previewFileUrl'    => $this->previewFileUrl,
        ])->layout('admin.master');
    }
}
