<?php

namespace App\Livewire\Admin\Payment;

use App\Models\{ApprovalRule, PaymentRequest, PaymentRequestApproval, PaymentRequestAttachment, PurchaseRequest, Project, Payee};
use Illuminate\Support\Facades\{Auth, DB, Log, Storage};
use Livewire\Attributes\{Layout, Computed, On, Url};
use Livewire\{Component, WithFileUploads};
use Morilog\Jalali\Jalalian;
use Illuminate\Support\Str;

#[Layout('admin.master')]
class CreatePaymentRequest extends Component
{
    use WithFileUploads;

    // پارامتر URL برای دریافت خودکار پروژه از داشبورد
    #[Url(as: 'project')]
    public ?int $url_project_id = null;

    public int $currentStep = 1;

    // داده‌های اصلی و مدل‌ها
    public ?int $project_id = null;
    public ?int $payeeId = null;
    public ?Payee $selectedPayee = null;
    public ?int $purchaseRequestId = null;
    public ?PurchaseRequest $purchaseRequest = null;

    // فیلدهای مالی (استفاده از String برای هندل کردن جداکننده‌ها در فرانت)
    public string $amount_gross = "0";
    public int $tax_rate = 9;
    public $tax_amount = 0;
    public $amount = 0;

    // فیلدهای اطلاعاتی
    public string $prSearch = '';
    public string $payeeSearch = '';
    public string $recipient_name = '';
    public string $purpose = '';
    public string $due_date = '';
    public string $payment_method = 'transfer';
    public bool $is_urgent = false;
    public $attachments = [];

    // ویژگی‌های مانیتورینگ هوشمند
    public $budget_remaining = 0;
    public bool $is_duplicate_warning = false;

    protected $listeners = ['setDueDate'];

    public function mount()
    {
        // تنظیم تاریخ سررسید پیش‌فرض (۷ روز بعد)
        $this->due_date = Jalalian::fromCarbon(now()->addDays(7))->format('Y/m/d');

        if ($this->url_project_id) {
            $this->project_id = $this->url_project_id;
            $this->checkProjectBudget();
        }
    }

    /**
     * سیستم کنترل بودجه (Budget Guard)
     */
    public function checkProjectBudget()
    {
        if ($this->project_id) {
            $project = Project::find($this->project_id);
            // محاسبه مجموع درخواست‌های در جریان و پرداخت شده
            $totalSpent = PaymentRequest::where('project_id', $this->project_id)
                ->whereIn('status', ['approved', 'paid', 'pending'])
                ->sum('amount');

            $this->budget_remaining = ($project->budget ?? 0) - $totalSpent;
        }
    }

    // --- ویژگی‌های محاسباتی (Computed) ---

    #[Computed]
    public function approvedPurchaseRequests()
    {
        if (!$this->project_id) return collect();
        return PurchaseRequest::where('status', 'approved')
            ->where('project_id', $this->project_id)
            ->when($this->prSearch, fn($q) => $q->where('id', 'like', "%{$this->prSearch}%"))
            ->with('items')
            ->latest()->limit(5)->get();
    }

    #[Computed]
    public function availablePayees()
    {
        if (strlen($this->payeeSearch) < 2) return collect();
        return Payee::where('name', 'like', "%{$this->payeeSearch}%")
            ->orWhere('national_id', 'like', "%{$this->payeeSearch}%")
            ->limit(5)->get();
    }

    #[Computed]
    public function projectsList()
    {
        return Project::select('id', 'name')->orderBy('name')->get();
    }

    // --- منطق‌های محاسباتی و امنیتی ---

    public function updatedAmountGross()
    {
        $this->sanitizeAmount();
        $this->updateTotals();
        $this->checkForDuplicates();
    }

    public function updatedTaxRate() { $this->updateTotals(); }

    public function updatedProjectId()
    {
        $this->checkProjectBudget();
        $this->resetPR(); // ریست کردن PR چون PR متعلق به پروژه قبلی است
    }

    protected function sanitizeAmount()
    {
        // حذف هرگونه کاراکتر غیر عددی (جلوگیری از حملات تزریق و خطای کستینگ)
        $this->amount_gross = preg_replace('/[^0-9]/', '', $this->amount_gross);
    }

    protected function updateTotals()
    {
        $gross = (float)($this->amount_gross ?: 0);
        $this->tax_amount = round(($gross * $this->tax_rate) / 100);
        $this->amount = $gross + $this->tax_amount;
    }

    protected function checkForDuplicates()
    {
        if ($this->payeeId && (float)$this->amount_gross > 0) {
            $this->is_duplicate_warning = PaymentRequest::where('payee_id', $this->payeeId)
                ->where('amount_gross', (float)$this->amount_gross)
                ->where('created_at', '>', now()->subDays(30))
                ->where('status', '!=', 'rejected')
                ->exists();
        }
    }

    // --- اکشن‌های پیمایش و انتخاب ---

    public function nextStep()
    {
        if ($this->currentStep == 1) {
            $this->validate([
                'project_id' => 'required|exists:projects,id',
                'payeeId' => 'required|exists:payees,id',
            ], [
                'project_id.required' => 'انتخاب پروژه الزامی است.',
                'payeeId.required' => 'تعیین ذینفع الزامی است.'
            ]);
        }
        $this->currentStep++;
    }

    public function previousStep() { $this->currentStep--; }

    public function selectPayee($id)
    {
        $this->selectedPayee = Payee::find($id);
        $this->payeeId = $id;
        $this->recipient_name = $this->selectedPayee->name;
        $this->payeeSearch = '';
        $this->checkForDuplicates();
    }

    public function selectPurchaseRequest($id)
    {
        $this->purchaseRequest = PurchaseRequest::with('items')->find($id);
        if ($this->purchaseRequest) {
            $this->purchaseRequestId = $id;
            // استخراج جمع مبلغ از آیتم‌های درخواست خرید
            $this->amount_gross = (string)$this->purchaseRequest->items->sum(fn($i) => $i->quantity * $i->price);
            $this->updateTotals();
            $this->purpose = "تسویه فاکتور PR-{$id} | پروژه: " . Project::find($this->project_id)->name;
        }
    }

    public function resetPayee() {
        $this->selectedPayee = null; $this->payeeId = null; $this->is_duplicate_warning = false;
    }

    public function resetPR() {
        $this->purchaseRequest = null; $this->purchaseRequestId = null;
    }

    #[On('setDueDate')]
    public function setDueDate($date) { $this->due_date = $date; }

    // --- عملیات نهایی ذخیره‌سازی ---

    public function savePaymentRequest()
    {
        $this->validate([
            'project_id' => 'required',
            'payeeId' => 'required',
            'amount_gross' => 'required|numeric|min:1000',
            'purpose' => 'required|min:10|max:1000',
            'due_date' => 'required|string',
            'attachments.*' => 'nullable|file|max:15360|mimes:pdf,jpg,jpeg,png',
        ]);

        try {
            DB::beginTransaction();

            // ۱. ثبت رکورد اصلی درخواست وجه
            $payment = PaymentRequest::create([
                'user_id' => Auth::id(),
                'project_id' => $this->project_id,
                'payee_id' => $this->payeeId,
                'purchase_request_id' => $this->purchaseRequestId,
                'recipient_name' => $this->recipient_name,
                'amount_gross' => (float)$this->amount_gross,
                'tax_amount' => (float)$this->tax_amount,
                'amount' => (float)$this->amount,
                'purpose' => ($this->is_urgent ? '🔴 [فوری] - ' : '') . $this->purpose,
                'due_date' => Jalalian::fromFormat('Y/m/d', $this->due_date)->toCarbon(),
                'payment_method' => $this->payment_method,
                'status' => 'pending',
            ]);

            // ۲. پردازش و ذخیره فایل‌های پیوست
            if ($this->attachments) {
                foreach ($this->attachments as $file) {
                    $fileName = Str::random(10) . '_' . $file->getClientOriginalName();
                    $path = $file->storeAs('payments/' . $payment->id, $fileName, 'public');

                    PaymentRequestAttachment::create([
                        'payment_request_id' => $payment->id,
                        'file_path' => $path,
                        'original_name' => $file->getClientOriginalName(),
                        'mime_type' => $file->getMimeType(),
                        'size' => $file->getSize(),
                    ]);
                }
            }

            // ۳. تخصیص تاییدکنندگان بر اساس Approval Rules
            $this->assignApprovers($payment);

            DB::commit();

            session()->flash('message', "درخواست وجه #{$payment->id} با موفقیت در سامانه ثبت شد.");
            return $this->redirect(route('admin.payment.list'), navigate: true);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Critical Error in Payment Creation: " . $e->getMessage());
            $this->addError('system', 'خطای فنی در سامانه. لطفاً با راهبر سیستم تماس بگیرید.');
        }
    }

    protected function assignApprovers(PaymentRequest $payment)
    {
        $totalAmount = (float) $payment->amount;
        $projectId = $payment->project_id;

        // یافتن قوانین منطبق (اولویت ۱: پروژه اختصاصی | اولویت ۲: قوانین پیش‌فرض سازمان)
        $rules = ApprovalRule::where('type', 'payment')
            ->where('is_active', true)
            ->where(function ($query) use ($projectId) {
                $query->where('project_id', $projectId)
                    ->orWhere(function ($q) {
                        $q->whereNull('project_id')->where('is_default', true);
                    });
            })
            ->where(function ($query) use ($totalAmount) {
                $query->where(function ($q) use ($totalAmount) {
                    $q->where('amount_min', '<=', $totalAmount)
                        ->where(fn($sub) => $sub->where('amount_max', '>=', $totalAmount)->orWhereNull('amount_max'));
                })->orWhere(fn($q) => $q->whereNull('amount_min')->whereNull('amount_max'));
            })
            ->orderBy('is_default', 'asc') // ابتدا قوانین پروژه
            ->get();

        $assignedUserIds = [];
        foreach ($rules as $rule) {
            foreach ($rule->approvers()->orderBy('priority')->get() as $approver) {
                if (!in_array($approver->user_id, $assignedUserIds)) {
                    $payment->approvals()->create([
                        'user_id' => $approver->user_id,
                        'role_id' => $approver->role_id,
                        'priority' => $approver->priority,
                        'is_approved' => null,
                    ]);
                    $assignedUserIds[] = $approver->user_id;
                }
            }
        }
    }

    public function render()
    {
        return view('livewire.admin.payment.create-payment-request');
    }
}
