<?php

namespace App\Livewire\Admin\Payment;

use App\Models\SalesInvoice;
use App\Models\SalesInvoiceItem;
use App\Models\Project;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\Attributes\Computed;
use Morilog\Jalali\Jalalian;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\WithFileUploads;

#[Layout('admin.master')]
class CreateSalesInvoice extends Component
{
    use WithFileUploads;

    // --- فیلدهای اصلی فاکتور ---
    public $project_id;
    public $invoice_number;
    public $client_name;
    public $issue_date;
    public $due_date;
    public $description;
    public $tax_rate = 9;

    // 💡 فیلدهای مدیریت ارز
    public string $currency = 'IRR';
    public float $exchange_rate = 1.0;

    // --- فیلدهای موقت افزودن آیتم ---
    public array $items = [];
    public $new_description = '';
    public $new_quantity = 1;
    public $new_unit_price = 0;

    #[Computed]
    public function projects(): Collection
    {
        return Project::select('id', 'name')->get();
    }

    #[Computed]
    public function availableCurrencies(): array
    {
        return [
            'IRR' => 'ریال (IRR)',
            'USD' => 'دلار آمریکا (USD)',
            'EUR' => 'یورو (EUR)',
        ];
    }

    #[Computed]
    public function totals()
    {
        $net = collect($this->items)->sum(fn($item) => $item['quantity'] * $item['unit_price']);
        $tax = round(($net * $this->tax_rate) / 100);
        $total_foreign = $net + $tax;
        $total_irr = $total_foreign * $this->exchange_rate;

        return [
            'net' => $net,
            'tax' => $tax,
            'total_foreign' => $total_foreign,
            'total_irr' => round($total_irr),
        ];
    }

    public function mount()
    {
        $today = Jalalian::forge('today')->format('Y/m/d');
        $this->issue_date = $today;
        $this->due_date = Jalalian::fromCarbon(now()->addDays(30))->format('Y/m/d');
    }

    public function updatedIssueDate($value)
    {
        try {
            $issueDateCarbon = Jalalian::fromFormat('Y/m/d', $value)->toCarbon();
            $this->due_date = Jalalian::fromCarbon($issueDateCarbon->addDays(30))->format('Y/m/d');
        } catch (\Exception $e) {
            // silent fail or log if needed
        }
    }

    public function updatedCurrency($value)
    {
        if ($value === 'IRR') {
            $this->exchange_rate = 1.0;
        } else {
            // مقدار پیش‌فرض منطقی برای دلار/یورو (قابل ویرایش توسط کاربر)
            $this->exchange_rate = 50000; // مثلاً ۵۰ هزار تومان برای دلار
        }
    }

    public function addItem()
    {
        $this->validate([
            'new_description' => 'required|string|max:255',
            'new_quantity' => 'required|numeric|min:1',
            'new_unit_price' => 'required|numeric|min:0',
        ]);

        $this->items[] = [
            'description' => $this->new_description,
            'quantity' => (int) $this->new_quantity,
            'unit_price' => (float) $this->new_unit_price,
        ];

        $this->reset(['new_description', 'new_quantity', 'new_unit_price']);
        $this->resetValidation();
    }

    public function removeItem(int $index)
    {
        if (isset($this->items[$index])) {
            unset($this->items[$index]);
            $this->items = array_values($this->items);
        }
    }

    protected $rules = [
        'project_id' => 'required|exists:projects,id',
        'invoice_number' => 'required|string|max:100|unique:sales_invoices,invoice_number',
        'client_name' => 'required|string|max:255',
        'issue_date' => 'required|date_format:Y/m/d',
        'due_date' => 'required|date_format:Y/m/d|after_or_equal:issue_date',
        'tax_rate' => 'required|numeric|min:0|max:100',
        'currency' => 'required|string|in:IRR,USD,EUR',
        'exchange_rate' => 'required_if:currency,!=,IRR|numeric|min:0.01',
        'description' => 'nullable|string|max:500',
        'items' => 'required|array|min:1',
        'items.*.description' => 'required|string|min:3',
        'items.*.quantity' => 'required|numeric|min:1',
        'items.*.unit_price' => 'required|numeric|min:0',
    ];

    public function saveInvoice()
    {
        $this->validate();

        try {
            $issueDateCarbon = Jalalian::fromFormat('Y/m/d', $this->issue_date)->toCarbon();
            $dueDateCarbon = Jalalian::fromFormat('Y/m/d', $this->due_date)->toCarbon();
        } catch (\Exception $e) {
            $this->addError('issue_date', 'فرمت تاریخ صدور نامعتبر است.');
            $this->addError('due_date', 'فرمت تاریخ سررسید نامعتبر است.');
            return;
        }

        if ($this->totals['total_foreign'] <= 0) {
            $this->addError('items', 'مجموع مبالغ فاکتور باید بیشتر از صفر باشد.');
            return;
        }

        try {
            DB::beginTransaction();

            $invoice = SalesInvoice::create([
                'user_id' => Auth::id(),
                'project_id' => $this->project_id,
                'invoice_number' => $this->invoice_number,
                'client_name' => $this->client_name,
                'issue_date' => $issueDateCarbon,
                'due_date' => $dueDateCarbon,
                'currency' => $this->currency,
                'exchange_rate' => $this->exchange_rate,
                'net_amount' => $this->totals['net'],
                'tax_amount' => $this->totals['tax'],
                'total_foreign' => $this->totals['total_foreign'], // ✅ اصلاح شده
                'total_amount_irr' => $this->totals['total_irr'],
                'description' => $this->description,
                'status' => 'issued',
            ]);

            foreach ($this->items as $item) {
                SalesInvoiceItem::create([
                    'sales_invoice_id' => $invoice->id, // ✅ درست
                    'description' => $item['description'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                   'subtotal' => $item['quantity'] * $item['unit_price'],
                ]);
            }

            DB::commit();

            session()->flash('message', "فاکتور فروش #{$this->invoice_number} با موفقیت ثبت شد.");
            return $this->redirect(route('admin.sales.list'), navigate: true);

        } catch (\Illuminate\Database\QueryException $e) {
            DB::rollBack();
            if (str_contains($e->getMessage(), 'Duplicate entry') || $e->getCode() === '23000') {
                session()->flash('error', 'شماره فاکتور تکراری است. لطفاً شماره دیگری وارد کنید.');
            } else {
                // 🔥 برای دیباگ: متن خطا را نمایش بده
                session()->flash('error', 'خطا در دیتابیس: ' . $e->getMessage());
            }
        } catch (\Exception $e) {
            DB::rollBack();
            session()->flash('error', 'خطای غیرمنتظره‌ای رخ داد. لطفاً مجدداً تلاش کنید.');
        }
    }




    public function render()
    {
        return view('livewire.admin.payment.create-sales-invoice', [
            'projects' => $this->projects,
            'totals' => $this->totals,
            'availableCurrencies' => $this->availableCurrencies,
        ]);
    }
}
