<?php

namespace App\Livewire\Admin\Payment;

use App\Models\{PaymentRequest, Project};
use Illuminate\Support\Facades\{Auth, DB};
use Livewire\Attributes\{Layout, Computed, Url};
use Livewire\{Component, WithPagination};

#[Layout('admin.master')]
class MyPaymentRequests extends Component
{
    use WithPagination;

    #[Url(history: true)]
    public string $search = '';
    #[Url(history: true)]
    public string $statusFilter = '';
    #[Url(history: true)]
    public string $projectFilter = '';

    public string $viewMode = 'grid';
    protected string $paginationTheme = 'bootstrap';

    public function mount()
    {
        $this->viewMode = session('payment_view_mode', 'grid');
    }

    public function setViewMode(string $mode)
    {
        $this->viewMode = $mode;
        session(['payment_view_mode' => $mode]);
    }

    public function updatingSearch() { $this->resetPage(); }
    public function updatingStatusFilter() { $this->resetPage(); }
    public function updatingProjectFilter() { $this->resetPage(); }

    /**
     * هاب آماری هوشمند با محاسبات کلیدی
     */
    #[Computed]
    public function stats()
    {
        $baseQuery = PaymentRequest::where('user_id', Auth::id());
        $totalCount = $baseQuery->count();

        return [
            'total_count' => $totalCount,
            'pending_amount' => $baseQuery->clone()->where('status', 'pending')->sum('amount'),
            'paid_amount' => $baseQuery->clone()->where('status', 'paid')->sum('amount'),
            'approved_count' => $baseQuery->clone()->where('status', 'approved')->count(),
            // درصد نقدینگی تسویه شده
            'completion_rate' => $totalCount > 0
                ? round(($baseQuery->clone()->where('status', 'paid')->count() / $totalCount) * 100)
                : 0,
        ];
    }

    public function deleteRequest(int $id)
    {
        try {
            DB::beginTransaction();
            $request = PaymentRequest::where('id', $id)
                ->where('user_id', Auth::id())
                ->where('status', 'pending')
                ->firstOrFail();

            $request->approvals()->delete();
            $request->delete();

            DB::commit();
            $this->dispatch('notify', ['type' => 'success', 'message' => 'درخواست وجه با موفقیت ابطال شد.']);
        } catch (\Exception $e) {
            DB::rollBack();
            $this->dispatch('notify', ['type' => 'error', 'message' => 'خطا: این پرونده در جریان تاییدات اداری است.']);
        }
    }

    public function getStatusData(string $status): array
    {
        return match ($status) {
            'approved' => ['label' => 'تأیید نهایی', 'color' => 'success', 'icon' => 'ti ti-circle-check-filled', 'bg' => 'st-approved'],
            'rejected' => ['label' => 'رد شده', 'color' => 'danger', 'icon' => 'ti ti-circle-x-filled', 'bg' => 'st-rejected'],
            'paid'     => ['label' => 'تسویه مالی', 'color' => 'primary', 'icon' => 'ti ti-receipt-2', 'bg' => 'st-paid'],
            default    => ['label' => 'در انتظار', 'color' => 'warning', 'icon' => 'ti ti-clock-pause', 'bg' => 'st-pending'],
        };
    }

    #[Computed]
    public function paymentRequests()
    {
        return PaymentRequest::with(['project:id,name', 'approvals.user'])
            ->where('user_id', Auth::id())
            ->when($this->search, function($q) {
                $q->where(fn($sub) => $sub->where('purpose', 'like', "%{$this->search}%")
                    ->orWhere('recipient_name', 'like', "%{$this->search}%")
                    ->orWhere('id', 'like', "%{$this->search}%"));
            })
            ->when($this->statusFilter, fn($q) => $q->where('status', $this->statusFilter))
            ->when($this->projectFilter, fn($q) => $q->where('project_id', $this->projectFilter))
            ->orderByDesc('is_urgent')
            ->orderByDesc('created_at')
            ->paginate($this->viewMode === 'grid' ? 9 : 15);
    }

    #[Computed]
    public function projects()
    {
        return Project::select('id', 'name')->orderBy('name')->get();
    }

    public function render()
    {
        return view('livewire.admin.payment.my-payment-requests');
    }
}
