<?php

namespace App\Livewire\Admin\Payment;

use App\Models\{PaymentRequest, PaymentRequestApproval};
use Illuminate\Support\Facades\{Auth, DB, Log, Schema};
use Livewire\Attributes\{Layout, Computed, On, Url};
use Livewire\{Component, WithPagination};

#[Layout('admin.master')]
class PaymentApprovalInbox extends Component
{
    use WithPagination;

    #[Url(history: true)]
    public $search = '';

    #[Url(history: true)]
    public $filterStatus = 'pending'; // pending, processed

    public $selectedRequestId = null;
    public $comment = '';

    protected $paginationTheme = 'bootstrap';

    public function updatingSearch() { $this->resetPage(); }
    public function updatingFilterStatus() { $this->resetPage(); }

    /**
     * کوئری هوشمند استخراج درخواست‌ها بر اساس منطق زنجیره‌ای
     */
    #[Computed]
    public function pendingRequests()
    {
        $userId = Auth::id();

        return PaymentRequest::query()
            ->with(['project', 'user', 'attachments', 'approvals.user', 'approvals.role'])
            ->whereHas('approvals', function ($query) use ($userId) {
                $query->where('user_id', $userId)
                    ->when($this->filterStatus === 'pending', fn($q) => $q->whereNull('is_approved'))
                    ->when($this->filterStatus === 'processed', fn($q) => $q->whereNotNull('is_approved'));
            })
            // منطق طلایی: فقط درخواست‌هایی که نوبت تایید این کاربر است (در حالت Pending)
            ->when($this->filterStatus === 'pending', function($query) use ($userId) {
                $query->where(function($sub) use ($userId) {
                    $sub->whereRaw("
                        (SELECT min(priority) FROM payment_request_approvals
                         WHERE payment_request_id = payment_requests.id
                         AND is_approved IS NULL) =
                        (SELECT priority FROM payment_request_approvals
                         WHERE payment_request_id = payment_requests.id
                         AND user_id = ? LIMIT 1)
                    ", [$userId]);
                });
            })
            ->when($this->search, function($q) {
                $q->where(fn($sub) =>
                $sub->where('purpose', 'like', "%{$this->search}%")
                    ->orWhere('id', 'like', "%{$this->search}%")
                    ->orWhereHas('user', fn($u) => $u->where('name', 'like', "%{$this->search}%"))
                );
            })
            // بررسی وجود ستون برای جلوگیری از خطا در صورت عدم وجود Migration
            ->when(Schema::hasColumn('payment_requests', 'is_urgent'), fn($q) => $q->orderBy('is_urgent', 'desc'))
            ->orderBy('created_at', 'asc')
            ->paginate(12);
    }

    /**
     * آمار لحظه‌ای کارتابل کاربر
     */
    #[Computed]
    public function stats()
    {
        $userId = Auth::id();
        return [
            'pending_count' => PaymentRequest::whereHas('approvals', fn($q) => $q->where('user_id', $userId)->whereNull('is_approved'))->count(),
            'pending_value' => PaymentRequest::whereHas('approvals', fn($q) => $q->where('user_id', $userId)->whereNull('is_approved'))->sum('amount'),
            'my_actions_today' => PaymentRequestApproval::where('user_id', $userId)->whereDate('approved_at', now())->count(),
        ];
    }

    public function openDecisionModal($id)
    {
        $this->selectedRequestId = $id;
        $this->comment = '';
        $this->dispatch('open-modal', 'decisionModal');
    }

    public function processDecision($isApproved)
    {
        if (!$this->selectedRequestId) return;

        try {
            DB::beginTransaction();

            $request = PaymentRequest::findOrFail($this->selectedRequestId);
            $currentApproval = $request->approvals()
                ->where('user_id', Auth::id())
                ->whereNull('is_approved')
                ->firstOrFail();

            $currentApproval->update([
                'is_approved' => $isApproved,
                'approved_at' => now(),
                'comment' => $this->comment,
            ]);

            // متد به‌روزرسانی وضعیت کلی در مدل PaymentRequest
            if (method_exists($request, 'updateStatus')) {
                $request->updateStatus();
            }

            DB::commit();

            $this->dispatch('close-modal', 'decisionModal');
            $this->dispatch('notify', [
                'type' => $isApproved ? 'success' : 'warning',
                'message' => $isApproved ? 'درخواست وجه تایید و به مرحله بعد ارسال شد.' : 'درخواست وجه رد شد.'
            ]);

            $this->selectedRequestId = null;

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Approval Critical Error: " . $e->getMessage());
            $this->addError('decision', 'خطای سیستمی در ثبت تصمیم.');
        }
    }

    public function render()
    {
        return view('livewire.admin.payment.payment-approval-inbox');
    }
}
