<?php

namespace App\Livewire\Admin\Payment;

use App\Models\ApprovalRule;
use App\Models\Approver;
use App\Models\Project;
use App\Models\User;
use App\Models\Role;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Livewire\WithPagination;

class PaymentApprovalManagement extends Component
{
    use WithPagination;
    protected $paginationTheme = 'bootstrap';

    // --- فیلدهای فرم Rule ---
    public $name;
    public $projectId;
    public $amountMin;
    public $amountMax;
    public $priority = 1;
    public $isActive = 1;
    public $isDefault = 0;
    public $editRuleId;

    // --- فیلدهای فرم Approver ---
    public $ruleId;
    public $userId;
    public $roleId;
    public $approverPriority = 1;

    public $searchRule = '';
    public $statusFilter = '';
    public $isDefaultFilter = '';
    public $activeProject = null;
    public $viewMode = 'table';

    protected function dispatchSuccess($message)
    {
        $this->dispatch('show-toast', type: 'success', message: $message);
    }
    protected function dispatchError($message)
    {
        $this->dispatch('show-toast', type: 'error', message: $message);
    }

    protected function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'projectId' => 'sometimes|nullable|exists:projects,id',
            'amountMin' => 'sometimes|nullable|numeric|min:0',
            'amountMax' => 'sometimes|nullable|numeric|min:0',
            'priority' => 'required|integer|min:1',
            'isActive' => 'required|boolean',
            'isDefault' => 'sometimes|boolean',
        ];
    }
    protected $approverRules = [
        'ruleId' => 'required|exists:approval_rules,id',
        'userId' => 'required|exists:users,id',
        'roleId' => 'required|exists:roles,id',
        'approverPriority' => 'required|integer|min:1',
    ];


    #[Layout('admin.master')]
    public function render()
    {
        $projects = Project::all();
        $users = User::all();
        $roles = Role::all();

        // 💡 فیلتر کلیدی: فقط قوانین نوع 'payment'
        $rules = ApprovalRule::with(['project', 'approvers.user', 'approvers.role'])
            ->where('type', 'payment')
            ->when($this->searchRule, function ($query) {
                $query->where('name', 'like', '%' . $this->searchRule . '%')
                    ->orWhereHas('project', fn ($q) => $q->where('name', 'like', '%' . $this->searchRule . '%'));
            })
            ->when($this->statusFilter !== '', fn ($query) => $query->where('is_active', $this->statusFilter))
            ->when($this->isDefaultFilter !== '', fn ($query) => $query->where('is_default', $this->isDefaultFilter))
            ->when($this->activeProject, fn ($query) => $query->where('project_id', $this->activeProject))
            ->orderBy('priority')
            ->paginate(10);

        return view('livewire.admin.payment.payment-approval-management', [
            'rules' => $rules,
            'projects' => $projects,
            'users' => $users,
            'roles' => $roles,
        ]);
    }

    // 💡 متد saveRule: تضمین می‌کند که نوع 'payment' ذخیره شود.
    public function saveRule()
    {
        $this->validate($this->rules());

        if ($this->isDefault) {
            ApprovalRule::query()->where('type', 'payment')->update(['is_default' => 0]);
        }

        ApprovalRule::create([
            'name' => $this->name,
            'project_id' => $this->projectId,
            'amount_min' => $this->amountMin,
            'amount_max' => $this->amountMax,
            'priority' => $this->priority,
            'is_active' => $this->isActive,
            'is_default' => $this->isDefault,
            'type' => 'payment', // ✅ ست کردن نوع ثابت
        ]);

        $this->resetForm();
        $this->dispatchSuccess('قانون پرداخت با موفقیت اضافه شد.');
    }

    // 💡 متد updateRule: باید type را در کوئری به‌روزرسانی نهایی لحاظ کند.
    public function updateRule()
    {
        $this->validate($this->rules());

        if ($this->isDefault) {
            ApprovalRule::query()->where('type', 'payment')->update(['is_default' => 0]);
        }

        $rule = ApprovalRule::findOrFail($this->editRuleId);
        // 💡 چک می‌کنیم که قانون متعلق به Payment باشد
        if ($rule->type !== 'payment') {
            $this->dispatchError('این قانون متعلق به بخش دیگری است و قابل ویرایش نیست.');
            return;
        }

        $rule->update([
            'name' => $this->name,
            'project_id' => $this->projectId,
            'amount_min' => $this->amountMin,
            'amount_max' => $this->amountMax,
            'priority' => $this->priority,
            'is_active' => $this->isActive,
            'is_default' => $this->isDefault,
        ]);

        $this->resetForm();
        $this->dispatchSuccess('قانون پرداخت با موفقیت ویرایش شد.');
    }

    public function resetForm()
    {
        $this->reset(['name', 'projectId', 'amountMin', 'amountMax', 'priority', 'isActive', 'isDefault', 'editRuleId']);
        $this->resetValidation();
    }

    public function deleteRule($id)
    {
        $rule = ApprovalRule::findOrFail($id);
        if ($rule->type !== 'payment') {
            $this->dispatchError('فقط مجاز به حذف قوانین پرداخت هستید.');
            return;
        }
        $rule->delete();
        $this->dispatchSuccess('قانون با موفقیت حذف شد.');
    }

    public function setDefaultRule($id)
    {
        ApprovalRule::query()->where('type', 'payment')->update(['is_default' => 0]);
        $rule = ApprovalRule::findOrFail($id);

        if ($rule->type !== 'payment') {
            $this->dispatchError('این قانون متعلق به بخش دیگری است.');
            return;
        }

        $rule->update(['is_default' => 1]);
        $this->dispatchSuccess('قانون پیش‌فرض پرداخت با موفقیت تعیین شد.');
    }

    public function saveApprover()
    {
        // ... (منطق saveApprover) ...
        $this->validate($this->approverRules);

        $exists = Approver::where('approval_rule_id', $this->ruleId)
            ->where('user_id', $this->userId)
            ->exists();

        if ($exists) {
            $this->dispatchError('این کاربر قبلاً به این قانون اضافه شده است.');
            return;
        }

        Approver::create([
            'approval_rule_id' => $this->ruleId,
            'user_id' => $this->userId,
            'role_id' => $this->roleId,
            'priority' => $this->approverPriority,
        ]);

        $this->reset(['ruleId', 'userId', 'roleId', 'approverPriority']);
        $this->dispatchSuccess('تأییدکننده با موفقیت اضافه شد.');
    }

    public function deleteApprover($id)
    {
        Approver::findOrFail($id)->delete();
        $this->dispatchSuccess('تأییدکننده با موفقیت حذف شد.');
    }
}
