<?php

namespace App\Livewire\Admin\Payment;

use App\Models\{PaymentRequest, Project};
use Illuminate\Support\Facades\{DB, Log};
use Livewire\Attributes\{Layout, Computed, Url, On};
use Livewire\{Component, WithPagination};
use Morilog\Jalali\Jalalian;

#[Layout('admin.master')]
class PaymentRequestList extends Component
{
    use WithPagination;

    // --- فیلترهای هوشمند ---
    #[Url(history: true)]
    public string $search = '';
    #[Url(history: true)]
    public string $statusFilter = '';
    #[Url(history: true)]
    public string $projectFilter = '';
    #[Url(history: true)]
    public $minAmount = null;
    #[Url(history: true)]
    public $maxAmount = null;
    public $dateFrom = '';
    public $dateTo = '';

    // --- مدیریت جدول ---
    public string $sortField = 'created_at';
    public string $sortDirection = 'desc';
    public int $perPage = 12;

    // --- عملیات دسته‌جمعی ---
    public array $selectedRows = [];
    public bool $selectAll = false;

    protected string $paginationTheme = 'bootstrap';

    // --- Listeners برای ارتباط با سایر بخش‌ها ---
    protected $listeners = ['refreshList' => '$refresh'];

    // --- مدیریت بروزرسانی فیلترها ---
    public function updatedSearch() { $this->resetPage(); }
    public function updatedStatusFilter() { $this->resetPage(); }
    public function updatedProjectFilter() { $this->resetPage(); }
    public function updatedSelectAll($value)
    {
        $this->selectedRows = $value ? $this->paymentRequests->pluck('id')->map(fn($id) => (string)$id)->toArray() : [];
    }

    public function sortBy($field)
    {
        $this->sortDirection = ($this->sortField === $field && $this->sortDirection === 'asc') ? 'desc' : 'asc';
        $this->sortField = $field;
    }

    /**
     * محاسبات آماری داینامیک بر اساس فیلترهای جاری
     */
    #[Computed]
    public function filteredStats()
    {
        $query = $this->applyFilters(PaymentRequest::query());
        return [
            'count' => $query->count(),
            'sum' => $query->sum('amount'),
            'avg' => $query->avg('amount') ?? 0,
        ];
    }

    /**
     * دریافت لیست پروژه‌ها برای فیلتر
     */
    #[Computed]
    public function projects()
    {
        return Project::select('id', 'name')->orderBy('name')->get();
    }

    /**
     * سیستم فیلترینگ متمرکز
     */
    private function applyFilters($query)
    {
        return $query->when($this->search, function($q) {
            $q->where(fn($sub) => $sub->where('purpose', 'like', "%{$this->search}%")
                ->orWhere('recipient_name', 'like', "%{$this->search}%")
                ->orWhere('id', 'like', "%{$this->search}%"));
        })
            ->when($this->statusFilter, fn($q) => $q->where('status', $this->statusFilter))
            ->when($this->projectFilter, fn($q) => $q->where('project_id', $this->projectFilter))
            ->when($this->minAmount, fn($q) => $q->where('amount', '>=', $this->minAmount))
            ->when($this->maxAmount, fn($q) => $q->where('amount', '<=', $this->maxAmount));
    }

    /**
     * کوئری اصلی نمایش لیست با اعمال تمام فیلترها و مرتب‌سازی
     */
    #[Computed]
    public function paymentRequests()
    {
        $query = PaymentRequest::with(['project:id,name', 'user:id,name', 'approvals']);
        $query = $this->applyFilters($query);

        return $query->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);
    }

    // --- عملیات کاربردی (Actions) ---

    /**
     * تغییر وضعیت دسته‌جمعی
     */
    public function bulkChangeStatus($newStatus)
    {
        if (empty($this->selectedRows)) return;

        PaymentRequest::whereIn('id', $this->selectedRows)->update(['status' => $newStatus]);

        $this->reset(['selectedRows', 'selectAll']);
        $this->dispatch('notify', message: 'وضعیت درخواست‌های انتخاب شده بروزرسانی شد.');
    }

    /**
     * خروجی Excel ساده (CSV)
     */
    public function exportSelected()
    {
        $requests = PaymentRequest::whereIn('id', $this->selectedRows)->get();
        // در اینجا می‌توانید منطق تولید فایل CSV یا Excel را قرار دهید
        $this->dispatch('notify', message: 'فایل گزارش در حال آماده‌سازی است...');
    }

    /**
     * نمایش وضعیت‌ها با استایل‌های مدرن
     */
    public function getStatusData(string $status): array
    {
        return match ($status) {
            'approved' => ['label' => 'تأیید شده', 'class' => 'status-approved', 'icon' => 'ti ti-circle-check', 'bg' => 'bg-success'],
            'rejected' => ['label' => 'رد شده', 'class' => 'status-rejected', 'icon' => 'ti ti-circle-x', 'bg' => 'bg-danger'],
            'paid'     => ['label' => 'پرداخت شده', 'class' => 'status-paid', 'icon' => 'ti ti-cash', 'bg' => 'bg-info'],
            default    => ['label' => 'در انتظار', 'class' => 'status-pending', 'icon' => 'ti ti-clock-hour-4', 'bg' => 'bg-warning'],
        };
    }

    public function render()
    {
        return view('livewire.admin.payment.payment-request-list');
    }
}
