<?php

namespace App\Livewire\Admin\Payment;

use App\Models\PaymentRequest;
use Livewire\Attributes\{Layout, Computed};
use Livewire\Component;
use Mpdf\Mpdf;
use Illuminate\Support\Facades\File;

#[Layout('admin.master')]
class ShowPaymentRequest extends Component
{
    public PaymentRequest $paymentRequest;

    public function mount(PaymentRequest $paymentRequest)
    {
        // 🔒 بارگذاری کامل برای نمایش ۱۰۰٪ اطلاعات دیتابیس
        $this->paymentRequest = $paymentRequest->load([
            'project',
            'user.company',
            'payee',
            'approvals.user:id,name,sign',
            'approvals.role:id,name',
            'attachments',
            'purchaseRequest.items.item',
            'purchaseRequest.user:id,name'
        ]);
    }

    #[Computed]
    public function iranianBanks()
    {
        return [
            'bmi' => 'بانک ملی ایران', 'sepah' => 'بانک سپه', 'bsi' => 'بانک صادرات ایران',
            'mellat' => 'بانک ملت', 'tejarat' => 'بانک تجارت', 'ayandeh' => 'بانک آینده',
            'pasargad' => 'بانک پاسارگاد', 'parsian' => 'بانک پارسیان', 'saman' => 'بانک سامان',
            'shahr' => 'بانک شهر', 'en' => 'بانک اقتصاد نوین', 'post' => 'پست بانک ایران',
        ];
    }

    /**
     * 🌳 آماده‌سازی گره‌های درخت‌واره گردش کار
     */
    #[Computed]
    public function workflowSteps()
    {
        $steps = [];
        if ($pr = $this->paymentRequest->purchaseRequest) {
            $steps[] = [
                'title' => 'درخواست خرید',
                'desc' => 'ثبت توسط ' . $pr->user->name,
                'date' => $pr->created_at_shamsi,
                'status' => 'completed',
                'icon' => 'ti-shopping-cart',
                'link' => route('purchase.my-requests', $pr->id)
            ];
        }

        $steps[] = [
            'title' => 'صدور تسویه',
            'desc' => 'مبلغ ' . number_format($this->paymentRequest->amount) . ' ریال',
            'date' => $this->paymentRequest->created_at_shamsi,
            'status' => 'current',
            'icon' => 'ti-file-invoice',
            'link' => null
        ];

        foreach ($this->paymentRequest->approvals->sortBy('priority') as $approval) {
            $steps[] = [
                'title' => $approval->role->name ?? 'تاییدکننده',
                'desc' => $approval->user->name,
                'date' => $approval->is_approved !== null ? $approval->updated_at_shamsi : 'در انتظار',
                'status' => $approval->is_approved === 1 ? 'completed' : ($approval->is_approved === 0 ? 'rejected' : 'pending'),
                'icon' => 'ti-user-check',
                'link' => null
            ];
        }
        return $steps;
    }

    /**
     * 🖋️ آماده‌سازی امضاها برای خروجی PDF
     */
    private function prepareApprovalsData($request)
    {
        return $request->approvals->sortBy('priority')->map(function ($app) {
            $base64Sign = null;
            if ($app->is_approved && $app->user && $app->user->sign) {
                $path = storage_path('app/public/' . $app->user->sign);
                if (File::exists($path)) {
                    $imgData = File::get($path);
                    $ext = pathinfo($path, PATHINFO_EXTENSION);
                    $base64Sign = 'data:image/' . $ext . ';base64,' . base64_encode($imgData);
                }
            }
            return [
                'role' => $app->role->name ?? '---',
                'name' => $app->user->name ?? '---',
                'date' => $app->updated_at ? jdate($app->updated_at)->format('Y/m/d H:i') : null,
                'signature' => $base64Sign,
                'is_approved' => $app->is_approved
            ];
        });
    }

    /**
     * 📄 تولید و دانلود فایل PDF رسمی
     */
    public function downloadPdf()
    {
        $request = $this->paymentRequest;

        // ۱. لوگوی شرکت متقاضی (Base64)
        $companyLogoBase64 = null;
        $filename = $request->user->company->image ?? null;
        if ($filename) {
            $logoPath = public_path('app/images/company/logo/' . $filename);
            if (File::exists($logoPath)) {
                $logoData = File::get($logoPath);
                $companyLogoBase64 = 'data:image/' . pathinfo($logoPath, PATHINFO_EXTENSION) . ';base64,' . base64_encode($logoData);
            }
        }

        // ۲. تنظیمات mPDF
        $mpdf = new Mpdf([
            'mode' => 'utf-8', 'format' => 'A4', 'default_font' => 'vazir',
            'margin_left' => 10, 'margin_right' => 10, 'margin_top' => 10, 'margin_bottom' => 10
        ]);
        $mpdf->SetDirectionality('rtl');

        // ۳. رندر HTML مخصوص چاپ
        $html = view('pdf.payment-request-print', [
            'request' => $request,
            'companyLogo' => $companyLogoBase64,
            'approvals' => $this->prepareApprovalsData($request),
            'banks' => $this->iranianBanks(),
        ])->render();

        $mpdf->WriteHTML($html);
        return response()->streamDownload(fn() => print($mpdf->Output('', 'S')), "PRQ-{$request->id}.pdf");
    }

    /**
     * 📊 محاسبات شاخص‌های مالی
     */
    #[Computed]
    public function financialMetrics()
    {
        $projectBudget = (float) ($this->paymentRequest->project->budget ?? 0);
        $amount = (float) $this->paymentRequest->amount;
        $tax = (float) $this->paymentRequest->tax_amount;
        $gross = (float) $this->paymentRequest->amount_gross;

        return [
            'total_budget' => $projectBudget,
            'usage_percent' => ($projectBudget > 0) ? round(($amount / $projectBudget) * 100, 1) : 0,
            'tax_percent' => ($gross > 0) ? round(($tax / $gross) * 100, 1) : 9, // پیش‌فرض ۹ درصد
            'gross_amount' => $gross,
            'tax_amount' => $tax,
            'used_budget' => $amount, // یا مجموع هزینه‌های قبلی پروژه
        ];
    }
    public function render() { return view('livewire.admin.payment.show-payment-request'); }
}
