<?php

namespace App\Livewire\Admin\Permission;

use App\Models\Permission;
use Livewire\Attributes\Layout;
use Livewire\Component;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\DB;

class AddPermission extends Component
{
    public $name; // نام فنی (مثلاً: create_project)
    public $label; // نام نمایشی (مثلاً: ایجاد پروژه جدید)
    public $group; // گروه (مثلاً: projects)
    public $permissionId;
    public $isEditMode = false;
    public $permissions; // لیست مجوزهای موجود

    // FIX: تعریف $rules به عنوان متد عمومی برای اعتبارسنجی های پویا
    protected function rules(): array
    {
        $uniqueRule = 'unique:permissions,name';
        if ($this->isEditMode) {
            // FIX: برای حالت ویرایش، مجوز فعلی نادیده گرفته شود
            $uniqueRule = 'unique:permissions,name,' . $this->permissionId;
        }

        return [
            'name' => ['required', 'min:3', 'max:50', $uniqueRule],
            'label' => 'required|min:3|max:100',
            'group' => 'required|min:2|max:50',
        ];
    }

    // FIX: حذف متد updated($propertyName) برای جلوگیری از خطای validation در رفرش های زنده
    // (اعتبارسنجی live همچنان با wire:model.live کار می کند)

    // واکشی لیست مجوزها
    public function loadPermissions()
    {
        $this->permissions = Permission::orderBy('group')->orderBy('name')->get();
    }

    public function SavePermission()
    {
        // FIX: اعتبارسنجی قبل از ذخیره
        $this->validate();

        Permission::create([
            'name' => $this->name,
            'label' => $this->label,
            'group' => $this->group,
        ]);

        session()->flash('success', 'مجوز با موفقیت ایجاد شد.');
        $this->resetInput();
        $this->dispatch('Permissions-Created');
    }

    public function editPermission(int $id)
    {
        $permission = Permission::findOrFail($id);
        $this->permissionId = $permission->id;
        $this->name = $permission->name;
        $this->label = $permission->label;
        $this->group = $permission->group;
        $this->isEditMode = true;
    }

    public function updatePermission()
    {
        // FIX: اعتبارسنجی را اینجا اجرا می کنیم تا قوانین unique به روز شده اعمال شوند
        $this->validate();

        Permission::findOrFail($this->permissionId)->update([
            'name' => $this->name,
            'label' => $this->label,
            'group' => $this->group,
        ]);

        session()->flash('success', 'مجوز با موفقیت به‌روزرسانی شد.');
        $this->resetInput();
        $this->dispatch('Permissions-Updated');
    }

    public function deletePermission(int $id)
    {
        Permission::destroy($id);
        session()->flash('success', 'مجوز حذف شد.');
        $this->resetInput();
        $this->dispatch('Permissions-Deleted');
    }

    public function resetInput()
    {
        $this->reset(['name', 'label', 'group', 'permissionId', 'isEditMode']);
        $this->loadPermissions();
    }


    #[Layout('admin.master')]
    public function render()
    {
        $this->loadPermissions();

        return view('livewire.admin.permission.add-permission', [
            // واکشی گروه‌های منحصر به فرد
            'groups' => Permission::select('group')->distinct()->pluck('group'),
        ]);
    }
}
