<?php

namespace App\Livewire\Admin\PettyCash;

use App\Models\PettyCash;
use App\Models\PettyCashApproval;
use App\Models\Project;
use Livewire\Component;
use Livewire\Attributes\On;
use Livewire\Attributes\Validate; // استفاده از Attribute Validation جدید

class CreateRequest extends Component
{
    // از Validate Attribute استفاده می‌کنیم
    #[Validate('required|numeric|min:1000', message: 'مبلغ کل الزامی است و باید حداقل ۱,۰۰۰ تومان باشد.')]
    public $amount = 0;

    #[Validate('required|string|max:255', message: 'دلیل درخواست الزامی است.')]
    public $reason = '';

    #[Validate([
        'details' => 'required|array|min:1',
        'details.*.item' => 'required|string|max:100',
        'details.*.qty' => 'required|integer|min:1',
        'details.*.price' => 'required|numeric|min:100',
    ], message: [
        'details.required' => 'لطفاً حداقل یک آیتم درخواستی اضافه کنید.',
        'details.*.item.required' => 'نام آیتم الزامی است.',
        'details.*.qty.required' => 'تعداد آیتم الزامی است.',
        'details.*.price.required' => 'قیمت واحد الزامی است.',
    ])]
    public $details = [];

    #[Validate('nullable|array')]
    public $project_ids = [];

    // حذف متغیر $draftId که در اینجا استفاده‌ای ندارد.

    public function mount()
    {
        // بازیابی پیش‌نویس از localStorage
        $this->addDetail(); // برای شروع، یک آیتم خالی اضافه شود
        $this->dispatch('loadDraft');
    }

    /**
     * متد کمکی برای پاکسازی و تبدیل اعداد (استفاده داخلی)
     */
    protected function cleanNumber($value)
    {
        if (is_numeric($value)) return $value;

        $persian = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
        $arabic  = ['٠', '١', '٢', '٣', '٤', '٥', '٦', '٧', '٨', '٩'];
        $english = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];

        $value = str_replace($persian, $english, $value);
        $value = str_replace($arabic, $english, $value);
        $value = preg_replace('/[^0-9.]/', '', $value);

        return $value === '' ? 0 : (float) $value;
    }

    public function addDetail()
    {
        $this->details[] = ['item' => '', 'qty' => 1, 'price' => 0];
        $this->saveDraft();
    }

    public function removeDetail($index)
    {
        unset($this->details[$index]);
        $this->details = array_values($this->details);
        $this->recalculateAmount(); // فراخوانی مستقیم
        $this->saveDraft();
    }

    /**
     * @inheritdoc
     * هنگام بروزرسانی هر پراپرتی، پیش‌نویس ذخیره شود.
     */
    public function updated($property)
    {
        // برای ورودی‌های آیتم‌ها، مبلغ را محاسبه کنید
        if (str_contains($property, 'details')) {
            $this->recalculateAmount();
        }
        $this->saveDraft();
    }

    /**
     * متد اختصاصی برای محاسبه مجدد مبلغ کل
     */
    public function recalculateAmount()
    {
        foreach ($this->details as $index => $detail) {
            // اطمینان از تمیز بودن ورودی قبل از محاسبه
            $this->details[$index]['qty'] = (int) $this->cleanNumber($detail['qty'] ?? 0);
            $this->details[$index]['price'] = (float) $this->cleanNumber($detail['price'] ?? 0);
        }

        $total = collect($this->details)->sum(fn($d) => ($d['qty'] ?? 0) * ($d['price'] ?? 0));
        $this->amount = $total > 0 ? $total : 0;
    }


    #[On('saveDraft')]
    public function saveDraft()
    {
        $this->dispatch('saveDraft', [
            'reason' => $this->reason,
            'details' => $this->details,
            'project_ids' => $this->project_ids,
            'amount' => $this->amount,
        ]);
    }

    /**
     * متد اصلی ثبت درخواست
     */
    public function save()
    {
        $this->recalculateAmount(); // اطمینان از محاسبه نهایی قبل از اعتبارسنجی
        $this->validate();

        $amount = $this->amount; // مقدار تمیز شده قبلاً در amount ذخیره شده است

        $request = PettyCash::create([
            'user_id' => auth()->id(),
            'amount' => $amount,
            'reason' => $this->reason,
            'status' => PettyCash::STATUS_PENDING,
            'details_json' => $this->details, // دیگر نیازی به map کردن و clean کردن نیست
        ]);

        if (!empty($this->project_ids)) {
            $request->projects()->attach($this->project_ids);
        }

        // 🎯 بهینه‌سازی فرآیند تأیید: استفاده از متد مدل
        $approvers = $request->getRequiredApprovers(); // متد اصلاح شده در مدل PettyCash
        foreach ($approvers as $approver) {
            PettyCashApproval::create([
                'petty_cash_id' => $request->id,
                'user_id' => $approver->id,
                'status' => PettyCashApproval::STATUS_PENDING,
            ]);
            // فرض می‌کنیم نوتیفیکیشن درست تعریف شده است
            $approver->notify(new \App\Notifications\PettyCashNeedsApproval($request));
        }

        // حذف پیش‌نویس و ارسال پیام موفقیت
        $this->dispatch('clearDraft');

        session()->flash('alert', [
            'type' => 'success',
            'message' => '✅ درخواست شما با موفقیت ثبت شد و به تأییدکنندگان ارسال گردید.'
        ]);

        return redirect()->route('admin.petty-cash.index');
    }

    public function render()
    {
        // 🎯 بهینه‌سازی: فقط پروژه‌های فعال یا قابل دسترسی را نشان دهید
        $projects = Project::query()->orderBy('name')->get();
        return view('livewire.admin.petty-cash.create-request', compact('projects'))
            ->layout('admin.master');
    }
}
