<?php

namespace App\Livewire\Admin\PettyCash;

use App\Models\PettyCash;
use App\Models\PettyCashApproval;
use App\Models\User;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class ListRequests extends Component
{
    use WithPagination;

    public $selectedId;
    public $comments = '';
    public $action = '';
    public $statusFilter = 'all';
    public $search = '';

    public $detailRequest = null;

    protected $rules = [
        'comments' => 'nullable|string|max:500',
    ];

    /**
     * ✨ متد کمکی برای نمایش تصویر کاربر (حل مشکل نمایش عکس)
     */
    public function getUserProfilePhotoUrl($user): string
    {
        $name = is_object($user) && isset($user->name) ? $user->name : 'Unknown';
        $initial = mb_substr($name, 0, 1);
        $encodedInitial = urlencode($initial);
        $default = "https://ui-avatars.com/api/?name={$encodedInitial}&background=6366f1&color=fff";

        if (!is_object($user)) return $default;

        if (isset($user->image) && $user->image) {
            // اگر آدرس کامل بود (URL)
            if (str_starts_with($user->image, 'http')) {
                return $user->image;
            }
            // اگر در استورج محلی بود
            if (Storage::disk('public')->exists($user->image)) {
                return Storage::url($user->image);
            }
        }
        return $default;
    }

    public function updated($property)
    {
        if (in_array($property, ['statusFilter', 'search'])) {
            $this->resetPage();
        }
    }

    public function resetFilters()
    {
        $this->reset(['search', 'statusFilter']);
        $this->resetPage();
    }

    public function confirmAction($id, $type)
    {
        $this->selectedId = $id;
        $this->action = $type;
        $this->comments = '';
        $this->dispatch('openModal', name: 'approval-modal');
    }

    public function submit()
    {
        $this->validate();

        $request = PettyCash::with('approvals')->findOrFail($this->selectedId);
        $currentUserId = auth()->id();

        if (!$request->isApprover($currentUserId)) {
            $this->dispatch('alert', type: 'error', message: 'شما مجاز به اقدام روی این درخواست نیستید.');
            return;
        }

        $approvalRecord = $request->approvals->where('user_id', $currentUserId)->first();

        if ($approvalRecord && in_array($approvalRecord->status, [PettyCashApproval::STATUS_APPROVED, PettyCashApproval::STATUS_REJECTED])) {
            $this->dispatch('alert', type: 'warning', message: 'شما قبلاً اقدام نهایی خود را ثبت کرده‌اید.');
            return;
        }

        $status = $this->action === 'approve'
            ? PettyCashApproval::STATUS_APPROVED
            : PettyCashApproval::STATUS_REJECTED;

        PettyCashApproval::updateOrCreate(
            ['petty_cash_id' => $request->id, 'user_id' => $currentUserId],
            ['status' => $status, 'comments' => $this->comments ?: null, 'approved_at' => now()]
        );

        $request->refresh()->load('approvals');
        $request->updateStatusBasedOnApprovals();

        $this->dispatch('alert', type: 'success', message: 'عملیات با موفقیت در هسته مالی ثبت شد.');
        $this->reset(['selectedId', 'comments', 'action']);
        $this->dispatch('closeModal', name: 'approval-modal');
    }

    public function markAsPaid($id)
    {
        if (!$this->isCurrentUserAdmin()) {
            $this->dispatch('alert', type: 'error', message: 'دسترسی غیرمجاز.');
            return;
        }

        $request = PettyCash::findOrFail($id);
        if ($request->status !== PettyCash::STATUS_APPROVED) {
            $this->dispatch('alert', type: 'warning', message: 'فقط درخواست‌های تأیید شده قابل پرداخت هستند.');
            return;
        }

        $request->update(['status' => PettyCash::STATUS_PAID]);
        $this->dispatch('alert', type: 'success', message: 'تراکنش با موفقیت تسویه شد.');
    }

    public function viewDetails($id)
    {
        $this->selectedId = $id;
        $this->detailRequest = PettyCash::with(['user', 'approvals.approver', 'projects'])->findOrFail($id);
        $this->dispatch('openModal', name: 'details-modal');
    }

    protected function isCurrentUserAdmin(): bool
    {
        $user = auth()->user();
        return method_exists($user, 'isAdmin') ? $user->isAdmin() : ($user->role === 'admin');
    }

    public function render()
    {
        $isAdmin = $this->isCurrentUserAdmin();
        $query = PettyCash::with(['user', 'approvals.approver', 'projects']);

        // 🔒 لایه امنیتی نمایش داده‌ها
        if (!$isAdmin) {
            $query->where(function ($q) {
                $q->where('user_id', auth()->id())
                    ->orWhereHas('approvals', fn($a) => $a->where('user_id', auth()->id()));
            });
        }

        // فیلترها
        if ($this->statusFilter !== 'all') {
            $query->where('status', $this->statusFilter);
        }

        if ($this->search) {
            $query->where(function ($q) {
                $q->where('reason', 'like', "%{$this->search}%")
                    ->orWhereHas('user', fn($u) => $u->where('name', 'like', "%{$this->search}%"));
            });
        }

        // تنظیمات آمار برای استایل v10
        $statsCounts = PettyCash::query()
            ->when(!$isAdmin, function($q) {
                $q->where('user_id', auth()->id())->orWhereHas('approvals', fn($a) => $a->where('user_id', auth()->id()));
            })
            ->selectRaw("COUNT(*) as total")
            ->selectRaw("COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending")
            ->selectRaw("COUNT(CASE WHEN status = 'approved' THEN 1 END) as approved")
            ->selectRaw("COUNT(CASE WHEN status = 'paid' THEN 1 END) as paid")
            ->selectRaw("COUNT(CASE WHEN status = 'rejected' THEN 1 END) as rejected")
            ->first();

        $stats = [
            'total'    => $statsCounts->total ?? 0,
            'pending'  => $statsCounts->pending ?? 0,
            'approved' => $statsCounts->approved ?? 0,
            'paid'     => $statsCounts->paid ?? 0,
            'rejected' => $statsCounts->rejected ?? 0,
        ];

        // پیکربندی آیکون‌ها و رنگ‌ها برای View
        $statsConfig = [
            'total'    => ['icon' => 'ti ti-files', 'color' => 'primary', 'label' => 'کل درخواست‌ها'],
            'pending'  => ['icon' => 'ti ti-clock-hour-4', 'color' => 'warning', 'label' => 'در انتظار تایید'],
            'approved' => ['icon' => 'ti ti-circle-check', 'color' => 'indigo', 'label' => 'تایید شده'],
            'paid'     => ['icon' => 'ti ti-coin', 'color' => 'success', 'label' => 'پرداخت شده'],
            'rejected' => ['icon' => 'ti ti-circle-x', 'color' => 'danger', 'label' => 'رد شده'],
        ];

        return view('livewire.admin.petty-cash.list-requests', [
            'requests' => $query->latest()->paginate(12),
            'stats' => $stats,
            'statsConfig' => $statsConfig
        ])->layout('admin.master');
    }
}
