<?php
// FILE: App/Livewire/Admin/PettyCash/ManageApprovalRules.php (کلاس PHP)

namespace App\Livewire\Admin\PettyCash;

use App\Models\PettyCashApprovalRule;
use App\Models\User;
use App\Models\Company;
use App\Models\Department;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Gate;

class ManageApprovalRules extends Component
{
    use WithPagination;

    public $ruleId;
    public $min_amount;
    public $max_amount;
    public $approver_ids = [];
    public $required_approvals = 1;
    public $searchApprovers = '';
    public $roleFilter = 'all';

    public $companyIdFilter = '';
    public $companies = [];
    public $departments = [];

    protected array $messages = [
        'required_approvals.max' => 'تعداد تأییدهای لازم نباید از تعداد تأییدکنندگان انتخابی بیشتر باشد.',
        'approver_ids.required' => 'لطفاً حداقل یک تأییدکننده انتخاب کنید.',
        'min_amount.required' => 'حداقل مبلغ الزامی است.',
        'max_amount.gt' => 'حداکثر مبلغ باید بزرگتر از حداقل مبلغ باشد.',
    ];

    protected function rules()
    {
        return [
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'nullable|numeric|min:0|gt:min_amount',
            'approver_ids' => 'required|array|min:1',
            'approver_ids.*' => 'exists:users,id',
            'required_approvals' => 'required|integer|min:1|max:' . count($this->approver_ids),
            'companyIdFilter' => 'nullable|exists:companies,id', // 💡 شرکت هدف قانون
        ];
    }

    public function mount()
    {
        if (Gate::denies('manage-positions')) {
            abort(403, 'شما مجوز مدیریت جایگاه‌های سازمانی را ندارید.');
        }

        $this->companies = Company::select('id', 'name')->orderBy('name')->get();
        $this->resetForm();
    }

    public function updatedCompanyIdFilter()
    {
        $this->resetPage();
    }

    #[Computed]
    public function approvers()
    {
        $query = User::where('is_active', true)
            ->with(['department.company'])
            ->orderBy('name');

        // 🛑 فیلتر تأییدکنندگان بر اساس شرکت (برای نمایش در مودال)
        if ($this->companyIdFilter) {
            $query->whereHas('department.company', fn($q) => $q->where('companies.id', $this->companyIdFilter));
        }

        if ($this->roleFilter !== 'all') {
            $query->where('role', $this->roleFilter);
        }

        $searchedApprovers = (clone $query)
            ->when($this->searchApprovers, fn($q) => $q->where('name', 'like', "%{$this->searchApprovers}%"))
            ->pluck('id');

        $selectedApprovers = collect($this->approver_ids);

        $ids = $searchedApprovers->merge($selectedApprovers)->unique();

        return User::whereIn('id', $ids)->with(['department.company'])->take(500)->orderBy('name')->get();
    }

    public function create()
    {
        $this->resetForm();
        $this->dispatch('openModal', name: 'rule-modal');
    }

    public function edit($id)
    {
        $rule = \App\Models\PettyCashApprovalRule::findOrFail($id);
        $this->ruleId = $rule->id;
        $this->min_amount = $rule->min_amount;
        $this->max_amount = $rule->max_amount;
        $this->approver_ids = $rule->approver_ids;
        $this->required_approvals = $rule->required_approvals;
        // 🛑 لود شرکت هدف قانون
        // $this->companyIdFilter = $rule->company_id; // اگر در جدول Rule ذخیره شده باشد

        $this->searchApprovers = '';
        $this->roleFilter = 'all';
        $this->dispatch('openModal', name: 'rule-modal');
    }

    public function save()
    {
        if (Gate::denies('manage-positions')) { return; }

        $this->validate();

        if ($this->ruleId) {
            $rule = \App\Models\PettyCashApprovalRule::findOrFail($this->ruleId);
        } else {
            $rule = new \App\Models\PettyCashApprovalRule();
        }

        $rule->min_amount = $this->min_amount;
        $rule->max_amount = $this->max_amount ?: null;
        $rule->approver_ids = $this->approver_ids;
        $rule->required_approvals = $this->required_approvals;
        // 🛑 ذخیره شرکت هدف قانون
        // $rule->company_id = $this->companyIdFilter ?: null;
        $rule->save();

        $this->resetForm();
        $this->dispatch('closeModal', name: 'rule-modal');
        $this->dispatch('alert', type: 'success', message: $this->ruleId ? 'قانون با موفقیت به‌روز شد.' : 'قانون جدید ایجاد شد.');
    }

    public function delete($id)
    {
        if (Gate::denies('manage-positions')) { return; }
        $this->ruleId = $id;
        $this->dispatch('showConfirmDelete', id: $id);
    }

    #[On('confirmedDelete')]
    public function confirmedDelete()
    {
        if ($this->ruleId) {
            \App\Models\PettyCashApprovalRule::destroy($this->ruleId);
            $this->ruleId = null;
            $this->dispatch('alert', type: 'success', message: 'قانون با موفقیت حذف شد.');
        }
    }

    public function resetForm()
    {
        $this->reset(['ruleId', 'min_amount', 'max_amount', 'approver_ids', 'required_approvals', 'searchApprovers', 'roleFilter', 'companyIdFilter']);
        $this->required_approvals = 1;
    }

    public function render()
    {
        $rules = \App\Models\PettyCashApprovalRule::orderBy('min_amount')->paginate(10);

        $roles = [
            'all' => 'همه نقش‌ها',
            'admin' => 'مدیر سیستم',
            'manager' => 'مدیر',
            'user' => 'کاربر',
        ];

        return view('livewire.admin.petty-cash.manage-approval-rules', compact('rules', 'roles'))->layout('admin.master');
    }
}
