<?php

namespace App\Livewire\Admin\PettyCash;

use App\Models\PettyCash;
use Livewire\Component;
use Livewire\WithPagination;

class ProjectPettyCashList extends Component
{


    use WithPagination;

    public ?int $projectId = null;

    // ارث‌بری public properties از ListRequests (بدون extends — برای جلوگیری از خطا در mount)
    public $selectedId;
    public $comments = '';
    public $action = '';
    public $statusFilter = 'all';
    public $search = '';
    public $detailRequest = null;

    protected $rules = [
        'comments' => 'nullable|string|max:500',
    ];

    protected $listeners = ['setProject' => 'setProjectId'];

    public function mount(?int $projectId = null)
    {
        $this->projectId = $projectId;
        // مقادیر پیش‌فرض برای هماهنگی با ListRequests
        $this->statusFilter = 'all';
        $this->search = '';
    }

    public function setProjectId(int $id)
    {
        $this->projectId = $id;
        $this->resetPage();
    }

    /**
     * ✨ متد کمکی برای نمایش تصویر کاربر (همانند ListRequests)
     */
    public function getUserProfilePhotoUrl($user): string
    {
        $name = is_object($user) && isset($user->name) ? $user->name : 'Unknown';
        $initial = mb_substr($name, 0, 1);
        $encodedInitial = urlencode($initial);
        $default = "https://ui-avatars.com/api/?name={$encodedInitial}&background=6366f1&color=fff";

        if (!is_object($user)) return $default;

        if (isset($user->image) && $user->image) {
            if (str_starts_with($user->image, 'http')) {
                return $user->image;
            }
            if (\Storage::disk('public')->exists($user->image)) {
                return \Storage::url($user->image);
            }
        }
        return $default;
    }

    public function updated($property)
    {
        if (in_array($property, ['statusFilter', 'search', 'projectId'])) {
            $this->resetPage();
        }
    }

    public function resetFilters()
    {
        $this->reset(['search', 'statusFilter']);
        $this->resetPage();
    }

    // 👇 این متد‌ها را از ListRequests کپی می‌کنیم — برای حفظ رفتار تعاملی
    public function confirmAction($id, $type)
    {
        $this->selectedId = $id;
        $this->action = $type;
        $this->comments = '';
        $this->dispatch('openModal', name: 'approval-modal');
    }

    public function submit()
    {
        $this->validate();

        $request = PettyCash::with('approvals')->findOrFail($this->selectedId);
        $currentUserId = auth()->id();

        if (!$request->isApprover($currentUserId)) {
            $this->dispatch('alert', type: 'error', message: 'شما مجاز به اقدام روی این درخواست نیستید.');
            return;
        }

        $approvalRecord = $request->approvals->where('user_id', $currentUserId)->first();

        if ($approvalRecord && in_array($approvalRecord->status, [\App\Models\PettyCashApproval::STATUS_APPROVED, \App\Models\PettyCashApproval::STATUS_REJECTED])) {
            $this->dispatch('alert', type: 'warning', message: 'شما قبلاً اقدام نهایی خود را ثبت کرده‌اید.');
            return;
        }

        $status = $this->action === 'approve'
            ? \App\Models\PettyCashApproval::STATUS_APPROVED
            : \App\Models\PettyCashApproval::STATUS_REJECTED;

        \App\Models\PettyCashApproval::updateOrCreate(
            ['petty_cash_id' => $request->id, 'user_id' => $currentUserId],
            ['status' => $status, 'comments' => $this->comments ?: null, 'approved_at' => now()]
        );

        $request->refresh()->load('approvals');
        $request->updateStatusBasedOnApprovals();

        $this->dispatch('alert', type: 'success', message: 'عملیات با موفقیت در هسته مالی ثبت شد.');
        $this->reset(['selectedId', 'comments', 'action']);
        $this->dispatch('closeModal', name: 'approval-modal');
    }

    public function markAsPaid($id)
    {
        if (!$this->isCurrentUserAdmin()) {
            $this->dispatch('alert', type: 'error', message: 'دسترسی غیرمجاز.');
            return;
        }

        $request = PettyCash::findOrFail($id);
        if ($request->status !== \App\Models\PettyCash::STATUS_APPROVED) {
            $this->dispatch('alert', type: 'warning', message: 'فقط درخواست‌های تأیید شده قابل پرداخت هستند.');
            return;
        }

        $request->update(['status' => \App\Models\PettyCash::STATUS_PAID]);
        $this->dispatch('alert', type: 'success', message: 'تراکنش با موفقیت تسویه شد.');
    }

    public function viewDetails($id)
    {
        $this->selectedId = $id;
        $this->detailRequest = PettyCash::with(['user', 'approvals.approver', 'projects'])->findOrFail($id);
        $this->dispatch('openModal', name: 'details-modal');
    }

    protected function isCurrentUserAdmin(): bool
    {
        $user = auth()->user();
        return method_exists($user, 'isAdmin') ? $user->isAdmin() : ($user->role === 'admin');
    }

    public function render()
    {

//        \Log::info('Project ID:', [$this->projectId]);
//        \Log::info('Is Admin:', [$isAdmin]);
//        \Log::info('Auth ID:', [auth()->id()]);
//        \Log::info('Raw Query:', [$query->toSql()]);
//        \Log::info('Bindings:', [$query->getBindings()]);

        // اگر projectId null باشد، کوئری اجرا نشود (جلوگیری از ambiguity و خطا)
        if (is_null($this->projectId)) {
            return view('livewire.admin.petty-cash.project-petty-cash-list', [
                'requests' => collect(), // یا paginate خالی
                'stats' => ['total' => 0, 'pending' => 0, 'approved' => 0, 'paid' => 0, 'rejected' => 0],
                'statsConfig' => $this->getStatsConfig(),
                'projectId' => null,
            ])->layout('admin.master');
        }

        $isAdmin = $this->isCurrentUserAdmin();

        // ✅ رفع ambiguity: تمام فیلدها با نام جدول مشخص شوند
        // در متد render
        $query = PettyCash::withTrashed() // ✅ از این متد اینجا استفاده کنید
        ->with(['user', 'approvals.approver', 'projects'])
            ->whereHas('projects', fn($q) => $q->where('projects.id', $this->projectId));

        if (!$isAdmin) {
            $query->where(function ($q) {
                $q->where('petty_cash.user_id', auth()->id())
                    ->orWhereHas('approvals', fn($a) => $a->where('user_id', auth()->id()));
            });
        }

        if ($this->statusFilter !== 'all') {
            $query->where('petty_cash.status', $this->statusFilter);
        }

        if ($this->search) {
            $query->where(function ($q) {
                $q->where('petty_cash.reason', 'like', "%{$this->search}%")
                    ->orWhereHas('user', fn($u) => $u->where('name', 'like', "%{$this->search}%"));
            });
        }

        // ✅ آمار با پیشوند جدول — رفع خطای 1052
        $statsCounts = PettyCash::query()
            ->whereHas('projects', fn($q) => $q->where('projects.id', $this->projectId))
            ->when(!$isAdmin, function ($q) {
                $q->where('petty_cash.user_id', auth()->id())
                    ->orWhereHas('approvals', fn($a) => $a->where('user_id', auth()->id()));
            })
            ->selectRaw("COUNT(*) as total")
            ->selectRaw("COUNT(CASE WHEN petty_cash.status = 'pending' THEN 1 END) as pending")
            ->selectRaw("COUNT(CASE WHEN petty_cash.status = 'approved' THEN 1 END) as approved")
            ->selectRaw("COUNT(CASE WHEN petty_cash.status = 'paid' THEN 1 END) as paid")
            ->selectRaw("COUNT(CASE WHEN petty_cash.status = 'rejected' THEN 1 END) as rejected")
            ->first();

        $stats = [
            'total'    => $statsCounts->total ?? 0,
            'pending'  => $statsCounts->pending ?? 0,
            'approved' => $statsCounts->approved ?? 0,
            'paid'     => $statsCounts->paid ?? 0,
            'rejected' => $statsCounts->rejected ?? 0,
        ];

        return view('livewire.admin.petty-cash.project-petty-cash-list', [
            'requests'    => $query->latest('petty_cash.created_at')->paginate(12),
            'stats'       => $stats,
            'statsConfig' => $this->getStatsConfig(),
            'projectId'   => $this->projectId,
        ])->layout('admin.master');
    }

    protected function getStatsConfig(): array
    {
        return [
            'total'    => ['icon' => 'ti ti-files', 'color' => 'primary', 'label' => 'کل درخواست‌ها'],
            'pending'  => ['icon' => 'ti ti-clock-hour-4', 'color' => 'warning', 'label' => 'در انتظار تایید'],
            'approved' => ['icon' => 'ti ti-circle-check', 'color' => 'indigo', 'label' => 'تایید شده'],
            'paid'     => ['icon' => 'ti ti-coin', 'color' => 'success', 'label' => 'پرداخت شده'],
            'rejected' => ['icon' => 'ti ti-circle-x', 'color' => 'danger', 'label' => 'رد شده'],
        ];
    }
}
