<?php

namespace App\Livewire\Admin\Post;

use App\Models\Post;
use App\Models\BlogCategory;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Computed;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

// تعیین قالب (Layout) مدیریتی
#[Layout('admin.master')]
class CreatePost extends Component
{
    use WithFileUploads;

    // فیلدهای اصلی پست
    public $title;
    public $content;

    // فیلدهای مدیای آپلودی
    public $image;      // برای تصویر شاخص (فقط تصویر)
    public $media_file; // برای فایل‌های پیوست (ویدیو، PDF، اکسل و...)

    // فیلد دسته‌بندی و وضعیت
    public $category_id;
    public $is_published = true;

    // متغیرهای موقت برای پیش‌نمایش و نمایش نام فایل
    public $imagePreviewUrl = null;
    public $mediaFileName = null;

    /**
     * تعریف قوانین اعتبارسنجی
     */
    protected function rules(): array
    {
        return [
            'title' => 'required|string|max:255',
            'content' => 'required|string|min:20',
            // اعتبارسنجی category_id: الزامی و باید در جدول blog_categories وجود داشته باشد
            'category_id' => 'required|exists:blog_categories,id',
            // اعتبارسنجی image: اختیاری، باید تصویر باشد، حداکثر 3 مگابایت
            'image' => 'nullable|image|max:3072',
            // اعتبارسنجی media_file: اختیاری، باید فایل باشد، حداکثر 20 مگابایت، با فرمت‌های مشخص
            'media_file' => 'nullable|file|max:20480|mimes:pdf,docx,xlsx,mp4,avi,mov',
            'is_published' => 'boolean',
        ];
    }

    /**
     * Hook: پیش‌نمایش تصویر و اعتبارسنجی لحظه‌ای تصویر
     */
    public function updatedImage(): void
    {
        $this->validateOnly('image');
        $this->imagePreviewUrl = $this->image ? $this->image->temporaryUrl() : null;
    }

    /**
     * Hook: نمایش نام فایل پیوست و اعتبارسنجی لحظه‌ای فایل
     */
    public function updatedMediaFile(): void
    {
        $this->validateOnly('media_file');
        $this->mediaFileName = $this->media_file ? $this->media_file->getClientOriginalName() : null;
    }

    /**
     * حذف تصویر شاخص
     */
    public function removeImage(): void
    {
        $this->image = null;
        $this->imagePreviewUrl = null;
    }

    /**
     * حذف فایل پیوست
     */
    public function removeMediaFile(): void
    {
        $this->media_file = null;
        $this->mediaFileName = null;
    }

    /**
     * متد اصلی برای ایجاد و ذخیره پست جدید
     */
    public function createPost()
    {
        $this->validate();

        try {
            // 1. آپلود تصویر شاخص
            $imagePath = null;
            if ($this->image) {
                $imagePath = $this->image->store('posts/images', 'public');
            }

            // 2. آپلود فایل پیوست
            $mediaPath = null;
            if ($this->media_file) {
                $mediaPath = $this->media_file->store('posts/media', 'public');
            }

            // 3. ذخیره اطلاعات در دیتابیس
            Post::create([
                'author_id'      => auth()->id(),
                'blog_category_id' => $this->category_id, // 👈 باید اینجا ارسال شود
                'title'          => $this->title,
                'content'        => $this->content,
                'image'          => $imagePath,
                'media_file'     => $mediaPath,
                'is_published'   => $this->is_published,
                'slug'           => Str::slug($this->title), // 👈 افزودن این خط
            ]);

            session()->flash('success', '✅ پست و فایل‌های پیوست با موفقیت ایجاد و ذخیره شد.');
            // ریست کردن فیلدهای فرم پس از ذخیره موفق
            $this->reset();

        } catch (ValidationException $e) {
            throw $e; // نمایش خطاهای اعتبارسنجی در صورت وجود
        } catch (\Exception $e) {
            // ثبت خطا در لاگ سرور
            Log::error("Post creation failed: " . $e->getMessage(), ['user' => auth()->id()]);
            session()->flash('error', '❌ خطای سرور هنگام ایجاد پست.');
        }
    }

    /**
     * Computed Property: دریافت لیست دسته‌بندی‌ها
     * این لیست تنها زمانی محاسبه می‌شود که نیاز باشد و نتایج کش می‌شود.
     */
    #[Computed]
    public function categories()
    {
        return BlogCategory::select('id', 'name')->orderBy('name')->get();
    }

    /**
     * متد رندر کامپوننت
     */
    public function render()
    {
        return view('livewire.admin.post.create-post');
    }
}
