<?php

namespace App\Livewire\Admin\Post;

use App\Models\Post;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Rule;
use Livewire\Component;
use Livewire\WithFileUploads;

#[Layout('admin.master')]
class EditPost extends Component
{
    use WithFileUploads;

    public Post $post;

    #[Rule('required|string|min:5|max:255')]
    public ?string $title = '';

    #[Rule('required|string|min:10')]
    public ?string $content = '';

    #[Rule('boolean')]
    public bool $is_published = false;

    // تصویر شاخص جدید
    #[Rule('nullable|image|max:2048')] // حداکثر ۲ مگابایت
    public $image = null;

    // فایل پیوست جدید
    #[Rule('nullable|file|max:10240')] // حداکثر ۱۰ مگابایت
    public $media_file = null;

    // مسیرهای فعلی
    public ?string $current_image = null;
    public ?string $current_media_file = null;

    // فلگ حذف
    public bool $remove_image = false;
    public bool $remove_media_file = false;

    public function mount(Post $post): void
    {
        if ($post->author_id !== Auth::id()) {
            abort(403, 'شما مجاز به ویرایش این پست نیستید.');
        }

        $this->post = $post;

        $this->fill([
            'title'            => $post->title,
            'content'          => $post->content,
            'is_published'     => $post->is_published,
            'current_image'    => $post->image,
            'current_media_file' => $post->media_file,
        ]);
    }

    public function updatePost(): void
    {
        $validated = $this->validate();

        // فقط فیلدهای متنی/منطقی رو به مدل می‌ریزیم
        $data = collect($validated)->only(['title', 'content', 'is_published'])->toArray();
        $this->post->fill($data);

        // --- مدیریت تصویر شاخص ---

        // حذف تصویر فعلی
        if ($this->remove_image && $this->post->image) {
            Storage::disk('public')->delete($this->post->image);
            $this->post->image = null;
            $this->current_image = null;
        }

        // آپلود تصویر جدید
        if ($this->image) {
            // اگر قبلاً تصویر داشت، حذفش کن
            if ($this->post->image) {
                Storage::disk('public')->delete($this->post->image);
            }

            $path = $this->image->store('posts/images', 'public');
            $this->post->image = $path;
            $this->current_image = $path;
        }

        // --- مدیریت فایل پیوست ---

        if ($this->remove_media_file && $this->post->media_file) {
            Storage::disk('public')->delete($this->post->media_file);
            $this->post->media_file = null;
            $this->current_media_file = null;
        }

        if ($this->media_file) {
            if ($this->post->media_file) {
                Storage::disk('public')->delete($this->post->media_file);
            }

            $path = $this->media_file->store('posts/media', 'public');
            $this->post->media_file = $path;
            $this->current_media_file = $path;
        }

        $this->post->save();

        // ریست ورودی‌های فایل (که تو فرم خالی بشن)
        $this->image = null;
        $this->media_file = null;
        $this->remove_image = false;
        $this->remove_media_file = false;

        session()->flash('success', 'پست با موفقیت به‌روزرسانی شد ✅');
    }

    public function render()
    {
        return view('livewire.admin.post.edit-post');
    }
}
