<?php

namespace App\Livewire\Admin\Post;

use App\Models\Post;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

#[Layout('admin.master')]
class MyPosts extends Component
{
    use WithPagination;

    /**
     * تم صفحه‌بندی بوت‌استرپ
     */
    protected string $paginationTheme = 'bootstrap-5';

    #[Url(except: '')]
    public string $search = '';

    #[Url(except: 'all')]
    public string $statusFilter = 'all'; // all | published | draft

    #[Url(except: 'created_at')]
    public string $sortBy = 'created_at';

    #[Url(except: 'desc')]
    public string $sortDirection = 'desc';

    #[Url(except: 10)]
    public int $perPage = 10;

    /**
     * حذف پست
     */
    public function deletePost(int $postId): void
    {
        $post = Post::findOrFail($postId);

        if ($post->author_id !== auth()->id()) {
            session()->flash('error', 'شما اجازه حذف این پست را ندارید.');
            return;
        }

        // TODO: حذف فایل‌های مرتبط در صورت نیاز

        $title = $post->title;
        $post->delete();

        session()->flash('success', "پست «{$title}» با موفقیت حذف شد.");
        $this->resetPage();
    }

    /**
     * لیست پست‌ها (Computed)
     */
    #[Computed]
    public function posts()
    {
        $currentUserId = auth()->id();

        $query = Post::query()
            ->where('author_id', $currentUserId)
            ->when($this->search !== '', function ($q) {
                $term = trim($this->search);
                $q->where('title', 'like', "%{$term}%");
            });

        // فیلتر وضعیت
        if ($this->statusFilter === 'published') {
            $query->where('is_published', true);
        } elseif ($this->statusFilter === 'draft') {
            $query->where('is_published', false);
        }

        // مرتب‌سازی
        $query->orderBy($this->sortBy, $this->sortDirection);

        return $query->paginate($this->perPage);
    }

    public function render()
    {
        return view('livewire.admin.post.my-posts');
    }

    /**
     * مرتب‌سازی بر اساس ستون
     */
    public function sort(string $field): void
    {
        if ($this->sortBy === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $field;
            $this->sortDirection = 'desc';
        }

        $this->resetPage();
    }

    /**
     * ریست صفحه روی تغییر فیلتر/سرچ
     */
    public function updated($property): void
    {
        if (in_array($property, ['search', 'statusFilter', 'perPage'], true)) {
            $this->resetPage();
        }
    }
}
