<?php

namespace App\Livewire\Admin\ProjectFiles;

use App\Models\{Project, Section, ProjectFile};
use Livewire\{Component, WithFileUploads, Attributes\Layout, Attributes\Rule, Attributes\Computed};
use Illuminate\Support\Facades\{Auth, Storage};
use Illuminate\Support\Str;

class CreateFile extends Component
{
    use WithFileUploads;

    public $name;
    public $description = '';
    public $files = [];
    public $project_id = '';
    public $section_id = '';
    public $sections = [];

    protected function rules() {
        return [
            'name' => 'required|min:3|max:255',
            'project_id' => 'required|exists:projects,id',
            'section_id' => 'required|exists:sections,id',
            'files.*' => 'required|file|max:102400', // 100MB
        ];
    }

    protected $validationAttributes = [
        'files.*' => 'فایل انتخابی',
        'project_id' => 'پروژه مرجع',
        'section_id' => 'فاز عملیاتی'
    ];

    #[Computed]
    public function projects()
    {
        return Project::orderBy('name')->get(['id', 'name']);
    }

    public function updatedProjectId($value)
    {
        if ($value) {
            $this->sections = Section::where('project_id', $value)
                ->orderBy('name')->get(['id', 'name'])->toArray();
        } else {
            $this->sections = [];
        }
        $this->reset('section_id');
    }

    public function removeFile($index)
    {
        array_splice($this->files, $index, 1);
    }

    public function save()
    {
        $this->validate();

        try {
            $project = Project::findOrFail($this->project_id);
            $folderPath = "project-assets/{$project->id}/" . now()->format('Y/m');

            foreach ($this->files as $file) {
                $extension = $file->getClientOriginalExtension();
                // تولید نام امن برای جلوگیری از تداخل کاراکترهای فارسی در سیستم فایل
                $safeName = Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME));
                $fileName = time() . '_' . $safeName . '.' . $extension;

                $filePath = $file->storeAs($folderPath, $fileName, 'public');

                ProjectFile::create([
                    'name' => $this->name,
                    'description' => $this->description,
                    'path' => $filePath,
                    'mime_type' => $file->getMimeType(),
                    'size' => $file->getSize(),
                    'extension' => $extension,
                    'user_id' => Auth::id(),
                    'project_id' => $this->project_id,
                    'section_id' => $this->section_id,
                ]);
            }

            $this->reset(['name', 'description', 'files', 'project_id', 'section_id']);
            $this->dispatch('alert', ['type' => 'success', 'message' => 'مستندات با موفقیت در مخزن ابری پروژه ذخیره شدند.']);

        } catch (\Exception $e) {
            $this->dispatch('alert', ['type' => 'error', 'message' => 'خطای سیستمی در فرآیند آپلود!']);
        }
    }

    #[Layout('admin.master')]
    public function render()
    {
        return view('livewire.admin.project-files.create-fle');
    }
}
